package com.bizunited.empower.business.product.service.handler;

import com.bizunited.empower.business.product.dto.DiscountResultDto;

import java.util.List;
import java.util.Objects;

/**
 * DiscountHandler 打折/优惠的责任链抽象类
 *
 * @Author: hefan
 * @Date: 2021/4/2 14:18
 */
public abstract class DiscountHandler {

  private DiscountHandler discountHandler;

  public final void setDiscountHandler(DiscountHandler discountHandler) {
    this.discountHandler = discountHandler;
  }

  /**
   * 处理模板方法
   *
   * @param codeDto 商品、客户
   * @param prevResult 商品、客户、单位、价格
   * @return
   */
  public final List<DiscountResultDto> process(DiscountResultDto codeDto, List<DiscountResultDto> prevResult) {
    this.validateClosedLoop();
    List<DiscountResultDto> currentBase = this.basedOnOriginalPrice(codeDto.getCustomerCode(), codeDto.getProductCode());
    List<DiscountResultDto> currentResult = this.addUpDiscountResult(prevResult, currentBase);
    if (!this.isOver()) {
      // 有下一个链
      if (Objects.nonNull(this.discountHandler)) {
        currentResult = this.discountHandler.process(codeDto, currentResult);
      }
    }
    return currentResult;
  }

  /**
   * 检验，并使子类链闭环
   */
  public void validateClosedLoop() {}

  /**
   * 本次折算/优惠 【基于原价】的计算结果
   *
   * @param customerCode
   * @param productCode
   * @return
   */
  public abstract List<DiscountResultDto> basedOnOriginalPrice(String customerCode, String productCode);

  /**
   * 累计折算结果(本环节的最终价格)
   * 1.累计之前的价格。
   * 2.抛弃之前的价格，以本轮为最终价格。
   * 3.抛弃本轮的价格，以上一次价格为准
   *
   * @param prevResult   之前已经计算过的折算
   * @param currentBase  当前基于原价的折算
   * @return
   */
  public abstract List<DiscountResultDto> addUpDiscountResult(List<DiscountResultDto> prevResult,List<DiscountResultDto> currentBase);

  /**
   * 阻断之后的折算
   */
  public abstract Boolean isOver();

}
