package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.product.dto.DiscountResultDto;
import com.bizunited.empower.business.product.dto.ProductSpecificationAndUnitDto;
import com.bizunited.empower.business.product.entity.ProductSpecification;
import com.bizunited.empower.business.product.entity.ProductUnit;
import com.bizunited.empower.business.product.entity.ProductUnitSpecificationAndPrice;
import com.bizunited.empower.business.product.repository.ProductSpecificationRepository;
import com.bizunited.empower.business.product.repository.ProductUnitRepository;
import com.bizunited.empower.business.product.repository.ProductUnitSpecificationAndPriceRepository;
import com.bizunited.empower.business.product.service.ProductActionService;
import com.bizunited.empower.business.product.service.handler.DiscountHandler;
import com.bizunited.empower.business.product.vo.ProductPriceVo;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service("ProductActionServiceImpl")
public class ProductActionServiceImpl implements ProductActionService {

  @Autowired
  ProductUnitSpecificationAndPriceRepository productUnitSpecificationAndPriceRepository;

  @Autowired
  ProductSpecificationRepository productSpecificationRepository;

  @Autowired
  ProductUnitRepository productUnitRepository;

  @Autowired
  @Qualifier("VirtualHeadDiscountHandler")
  DiscountHandler discountHandler;


  @Override
  public List<DiscountResultDto> findProductFinalPrice(List<DiscountResultDto> discountResultDtoList) {
    int numberOfUnits = 5;
    List<DiscountResultDto> resultDtos = new ArrayList<>(discountResultDtoList.size() * numberOfUnits);
    for (DiscountResultDto dto : discountResultDtoList) {
      List<DiscountResultDto> process = this.discountHandler.process(dto, null);
      if (!CollectionUtils.isEmpty(process)) {
        resultDtos.addAll(process);
      }
    }
    return resultDtos;
  }

  @Override
  public List<ProductPriceVo> findProductSellingPrice(List<String> productCodeList) {
    Validate.isTrue(!CollectionUtils.isEmpty(productCodeList),"请传入商品编码集合");
    //查询这些商品的默认单位的销售价格
    List<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrice = productUnitSpecificationAndPriceRepository.findByTenantCodeAndProductCodeInAndIsBasicUnit(TenantUtils.getTenantCode(),productCodeList,true);
    //复制属性
    List<ProductPriceVo> productPriceVos;
    if (!CollectionUtils.isEmpty(productUnitSpecificationAndPrice)) {
      productPriceVos = new ArrayList<>(productUnitSpecificationAndPrice.size());
      for (ProductUnitSpecificationAndPrice unitAndPrice : productUnitSpecificationAndPrice) {
        ProductPriceVo vo = new ProductPriceVo();
        BeanUtils.copyProperties(unitAndPrice, vo);
        vo.setProductCode(unitAndPrice.getProduct().getProductCode());
        productPriceVos.add(vo);
      }
      //筛选出每个商品编码下最低的价格
      Map<String, ProductPriceVo> productPriceVoMap = productPriceVos.parallelStream().collect(Collectors.toMap(ProductPriceVo::getProductCode, Function.identity(), (v1, v2) -> v1.getSellingPrice().compareTo(v2.getSellingPrice()) < 0 ? v1 : v2));
      return Lists.newArrayList(productPriceVoMap.values());
    }else {
      productPriceVos = Lists.newArrayList();
    }
    return productPriceVos;
  }
  @Override
  public Boolean validityUnitCode(String productCode, String unitCode) {
    //TODO: 这里根据单位获取到的价格是一个集合， 那么这个接口如何去修改？
    List<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrice = productUnitSpecificationAndPriceRepository.findByProductCodeAndUnitCode(productCode, TenantUtils.getTenantCode(), unitCode);
    return !CollectionUtils.isEmpty(productUnitSpecificationAndPrice) && productUnitSpecificationAndPrice.get(0).getAllowPurchase();
  }

  @Override
  public Boolean validateSpecificationExist(List<String> productSpecificationCodes) {
    Validate.isTrue(!CollectionUtils.isEmpty(productSpecificationCodes),"商品规格数据集合不能为空");
    Integer count = productSpecificationRepository.countByProductSpecificationCodeInAndTenantCode(productSpecificationCodes, TenantUtils.getTenantCode());
    return count.equals(productSpecificationCodes.size());
  }

  @Override
  public ProductSpecificationAndUnitDto findSpecNameAndUnitName(List<Pair<String, String>> pairList) {
    ProductSpecificationAndUnitDto productSpecificationAndUnitDto = new ProductSpecificationAndUnitDto();
    if (CollectionUtils.isEmpty(pairList)) {
      return productSpecificationAndUnitDto;
    }
    /**
     * Pair：左：规格，右：单位
     * 1. 根据入参，获取 规格code集合，单位code集合
     * 2. 根据规格code集合获取规格信息
     * 3. 根据单位code集合获取单位信息
     * 4. 将查询结果收集为Map
     * 5. 拼装结果DTO
     */
    //根据入参，获取 规格code集合，单位code集合
    List<String> specificationCodeList = new ArrayList<>(pairList.size());
    List<String> unitCodeList =  new ArrayList<>(pairList.size());
    for (Pair<String, String> pair : pairList) {
      specificationCodeList.add(pair.getLeft());
      unitCodeList.add(pair.getRight());
    }
    // 查询数据库信息
    List<ProductSpecification> specificationList = productSpecificationRepository.findByTenantCodeAndProductSpecificationCodeIn(TenantUtils.getTenantCode(), specificationCodeList);
    List<ProductUnit> unitList = productUnitRepository.findByTenantCodeAndUnitCodeIn(TenantUtils.getTenantCode(), unitCodeList);
    List<Map<String,Object>> codeList =  productUnitSpecificationAndPriceRepository.findSpecificationCodeAndUnitCodeBySpecificationCodeIn(TenantUtils.getTenantCode(), specificationCodeList);

    // 将查询结果收集为Map
    Map<String, String> specMap = new HashMap<>(pairList.size());
    Map<String, String> unitMap = new HashMap<>(pairList.size());
    Map<String, Object> pairMap = new HashMap<>(pairList.size());
    for (ProductSpecification specification : specificationList) {
      specMap.put(specification.getProductSpecificationCode(), specification.getProductSpecificationName());
    }
    for (ProductUnit unit : unitList) {
      unitMap.put(unit.getUnitCode(), unit.getUnitName());
    }
    for (Map<String, Object> map : codeList) {
      String productSpecificationCode = (String) map.get("product_specification_code");
      String unitCode = (String) map.get("unit_code");
      pairMap.put(StringUtils.join(productSpecificationCode, unitCode), map);
    }
    // 拼装结果DTO
    for (Pair<String, String> pair : pairList) {
      String left = pair.getLeft();
      String right = pair.getRight();
      ProductSpecificationAndUnitDto internal = new ProductSpecificationAndUnitDto();
      internal.setProductSpecificationCode(left);
      internal.setProductSpecificationName(specMap.get(left));
      internal.setUnitCode(right);
      internal.setUnitName(unitMap.get(right));
      Object obj = pairMap.get(StringUtils.join(left, right));
      internal.setErrorPair(Objects.isNull(obj));
      productSpecificationAndUnitDto.setProductSpecificationAndUnitDto(internal);
    }
    return productSpecificationAndUnitDto;
  }

  @Override
  public void coverDiscountResults(String customer, Set<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrices) {
    if(StringUtils.isBlank(customer)){
      return;
    }
    if(CollectionUtils.isEmpty(productUnitSpecificationAndPrices)){
      return;
    }
    Map<String,DiscountResultDto> queryMap = new HashMap<>();
    for (ProductUnitSpecificationAndPrice productUnitSpecificationAndPrice : productUnitSpecificationAndPrices) {
      DiscountResultDto dto = new DiscountResultDto(customer, productUnitSpecificationAndPrice.getProduct().getProductCode());
      queryMap.put(dto.getProductCode(),dto);
    }
    List<DiscountResultDto> query = Lists.newArrayList(queryMap.values());

    List<DiscountResultDto> productFinalPrice = this.findProductFinalPrice(query);
    if(CollectionUtils.isEmpty(productFinalPrice)){
      return;
    }
    Map<String, List<DiscountResultDto>> puMapDiscountResult = productFinalPrice.stream().collect(Collectors.groupingBy(discountResultDto -> {
      String dotProductCode = discountResultDto.getProductCode();
      String dtoUnitCode = discountResultDto.getUnitCode();
      String productSpecificationCode = discountResultDto.getProductSpecificationCode();
      return StringUtils.join(dotProductCode, dtoUnitCode, productSpecificationCode);
    }));
    for (ProductUnitSpecificationAndPrice unitSpecificationAndPrice : productUnitSpecificationAndPrices) {
      String productCode = unitSpecificationAndPrice.getProduct().getProductCode();
      String unitCode = unitSpecificationAndPrice.getProductUnit().getUnitCode();
      String productSpecificationCode = unitSpecificationAndPrice.getProductSpecification().getProductSpecificationCode();
      String join = StringUtils.join(productCode, unitCode, productSpecificationCode);
      List<DiscountResultDto> resultDtos = puMapDiscountResult.get(join);
      if (!CollectionUtils.isEmpty(resultDtos)) {
        BigDecimal afterDiscountPrice = resultDtos.get(0).getAfterDiscountPrice();
        unitSpecificationAndPrice.setSellingPrice(afterDiscountPrice);
      }
    }
  }
}
