package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.product.entity.ProductBarCodeInfo;
import com.bizunited.empower.business.product.entity.ProductSpecification;
import com.bizunited.empower.business.product.repository.ProductBarCodeInfoRepository;
import com.bizunited.empower.business.product.service.ProductBarCodeInfoService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Sets;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * ProductBarCodeInfo业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("ProductBarCodeInfoServiceImpl")
public class ProductBarCodeInfoServiceImpl implements ProductBarCodeInfoService {
  @Autowired
  private ProductBarCodeInfoRepository productBarCodeInfoRepository;

  final static Pattern THIRTEEN_DIGITS = Pattern.compile("\\d{13}");

  /**
   * 批量新增条码
   *
   * @param productBarCodeInfos
   * @param specification
   */
  @Override
  public void createInsertAbleEntitySet(Set<ProductBarCodeInfo> productBarCodeInfos, ProductSpecification specification) {
    Validate.notNull(specification, "未传入规格信息！");
    if (CollectionUtils.isNotEmpty(productBarCodeInfos)) {
      for (ProductBarCodeInfo productBarCodeInfo : productBarCodeInfos) {
        productBarCodeInfo.setProductSpecification(specification);
        this.createInsertAbleEntity(productBarCodeInfo);
      }
      productBarCodeInfoRepository.saveAll(productBarCodeInfos);
    }
  }

  @Override
  public void deleteByBatch(Set<ProductBarCodeInfo> barCodeInfos) {
    if (CollectionUtils.isNotEmpty(barCodeInfos)) {
      this.productBarCodeInfoRepository.deleteAll(barCodeInfos);
    }
  }

  @Transactional
  @Override
  public Set<ProductBarCodeInfo> updateFormBatch(Set<ProductBarCodeInfo> productBarCodeInfos, ProductSpecification productSpecification) {
    /*5.3.1 条码删去不存在的
     *5.3.2 条码新增
     *5.3.3 条码保存
     */
    // 特殊的前置校验
    this.preCheck(productBarCodeInfos, productSpecification);
    productBarCodeInfos = ObjectUtils.defaultIfNull(productBarCodeInfos, Sets.newHashSet());
    // 分为有ID的，无ID的
    Map<Boolean, List<ProductBarCodeInfo>> idBooleanMap = productBarCodeInfos.stream()
        .collect(Collectors.partitioningBy(productBarCodeInfo -> StringUtils.isNotBlank(productBarCodeInfo.getId())));
    List<ProductBarCodeInfo> idTrueList = new ArrayList<>();
    if (idBooleanMap.containsKey(true)) {
      idTrueList = idBooleanMap.get(true);
    }
    List<ProductBarCodeInfo> idFalseList = new ArrayList<>();
    if (idBooleanMap.containsKey(false)) {
      idFalseList = idBooleanMap.get(false);
    }
    // 5.1删去不存在了的
    List<String> ids = idTrueList.stream().map(ProductBarCodeInfo::getId).filter(StringUtils::isNotEmpty).collect(Collectors.toList());
    Set<ProductBarCodeInfo> needDelete;
    if (CollectionUtils.isEmpty(ids)) {
      needDelete = productBarCodeInfoRepository.findByProductSpecification(TenantUtils.getTenantCode(), productSpecification.getProductSpecificationCode());
    } else {
      needDelete = productBarCodeInfoRepository.findByProductSpecificationAndIdNotIn(TenantUtils.getTenantCode(), productSpecification.getProductSpecificationCode(), ids);
    }
    if (CollectionUtils.isNotEmpty(needDelete)) {
      productBarCodeInfoRepository.deleteAll(needDelete);
    }
    // 5.2为新增信息赋值
    Set<ProductBarCodeInfo> saveAll = new HashSet<>(productBarCodeInfos.size());
    for (ProductBarCodeInfo productBarCodeInfo : idFalseList) {
      productBarCodeInfo.setProductSpecification(productSpecification);
      this.createInsertAbleEntity(productBarCodeInfo);
      productBarCodeInfoRepository.saveAndFlush(productBarCodeInfo);
      saveAll.add(productBarCodeInfo);
    }
    // 5.3 旧信息覆盖
    for (ProductBarCodeInfo productBarCodeInfo : idTrueList) {
      Optional<ProductBarCodeInfo> op_currentProductBarCodeInfo = this.productBarCodeInfoRepository.findById(productBarCodeInfo.getId());
      ProductBarCodeInfo currentProductBarCodeInfo = op_currentProductBarCodeInfo.orElse(null);
      currentProductBarCodeInfo = Validate.notNull(currentProductBarCodeInfo, "未发现指定的原始模型对象信");

      productBarCodeInfo.setCreateAccount(currentProductBarCodeInfo.getCreateAccount());
      productBarCodeInfo.setCreateTime(currentProductBarCodeInfo.getCreateTime());
      productBarCodeInfo.setProductSpecification(productSpecification);
      Date now = new Date();
      productBarCodeInfo.setModifyAccount(SecurityUtils.getUserAccount());
      productBarCodeInfo.setModifyTime(now);
      productBarCodeInfo.setTenantCode(TenantUtils.getTenantCode());
      this.updateValidation(productBarCodeInfo);
      productBarCodeInfoRepository.saveAndFlush(productBarCodeInfo);
      saveAll.add(productBarCodeInfo);
    }
    return saveAll;
  }

  @Override
  public ProductBarCodeInfo findByProductSpecificationCodeAndUnitCode(String productSpecificationCode, String unitCode) {
    if (StringUtils.isAnyBlank(productSpecificationCode, unitCode)) {
      return null;
    }
    String tenantCode = TenantUtils.getTenantCode();
    return productBarCodeInfoRepository.findByTenantCodeAndProductSpecificationCodeAndUnitCode(tenantCode, productSpecificationCode, unitCode);
  }

  /**
   * 前置校验
   *
   * @param productBarCodeInfos
   * @param productSpecification
   */
  private void preCheck(Set<ProductBarCodeInfo> productBarCodeInfos, ProductSpecification productSpecification) {
    Validate.notNull(productSpecification, "规格信息不能为空");
    //Validate.isTrue(CollectionUtils.isNotEmpty(productBarCodeInfos),"条码信息不能为空");
  }

  /**
   * 新增自建数据
   *
   * @param productBarCodeInfo
   */
  private void createInsertAbleEntity(ProductBarCodeInfo productBarCodeInfo) {
    //
    Date now = new Date();
    productBarCodeInfo.setCreateAccount(SecurityUtils.getUserAccount());
    productBarCodeInfo.setCreateTime(now);
    productBarCodeInfo.setModifyAccount(SecurityUtils.getUserAccount());
    productBarCodeInfo.setModifyTime(now);
    //
    productBarCodeInfo.setTenantCode(TenantUtils.getTenantCode());

    this.createValidation(productBarCodeInfo);

  }

  @Transactional
  @Override
  public ProductBarCodeInfo create(ProductBarCodeInfo productBarCodeInfo) {
    ProductBarCodeInfo current = this.createForm(productBarCodeInfo);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductBarCodeInfo createForm(ProductBarCodeInfo productBarCodeInfo) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    Date now = new Date();
    productBarCodeInfo.setCreateAccount(SecurityUtils.getUserAccount());
    productBarCodeInfo.setCreateTime(now);
    productBarCodeInfo.setModifyAccount(SecurityUtils.getUserAccount());
    productBarCodeInfo.setModifyTime(now);

    this.createValidation(productBarCodeInfo);

    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================

    this.productBarCodeInfoRepository.save(productBarCodeInfo);
    // 返回最终处理的结果，里面带有详细的关联信息
    return productBarCodeInfo;
  }

  /**
   * 在创建一个新的ProductBarCodeInfo模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProductBarCodeInfo productBarCodeInfo) {
    Validate.notNull(productBarCodeInfo, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(productBarCodeInfo.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    productBarCodeInfo.setId(null);
    Validate.notBlank(productBarCodeInfo.getUnitCode(), "添加信息时，单位编号不能为空！");
    Validate.notBlank(productBarCodeInfo.getBarCode(), "添加信息时，条形码不能为空！");
    //条码不能重复
    Integer count = productBarCodeInfoRepository.countByBarCodeAndTenantCode(productBarCodeInfo.getBarCode(), TenantUtils.getTenantCode());
    Validate.isTrue(count==0,"条形码%s已经存在，不能重复添加",productBarCodeInfo.getBarCode());
    //13位数字
    Validate.isTrue(THIRTEEN_DIGITS.matcher(productBarCodeInfo.getBarCode()).matches(), "条形码必须为13位数字！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(productBarCodeInfo.getUnitCode() == null || productBarCodeInfo.getUnitCode().length() < 64, "单位编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productBarCodeInfo.getBarCode() == null || productBarCodeInfo.getBarCode().length() < 64, "条形码,在进行添加时填入值超过了限定长度(64)，请检查!");
  }

  @Transactional
  @Override
  public ProductBarCodeInfo update(ProductBarCodeInfo productBarCodeInfo) {
    ProductBarCodeInfo current = this.updateForm(productBarCodeInfo);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductBarCodeInfo updateForm(ProductBarCodeInfo productBarCodeInfo) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(productBarCodeInfo);
    // ===================基本信息
    String currentId = productBarCodeInfo.getId();
    Optional<ProductBarCodeInfo> op_currentProductBarCodeInfo = this.productBarCodeInfoRepository.findById(currentId);
    ProductBarCodeInfo currentProductBarCodeInfo = op_currentProductBarCodeInfo.orElse(null);
    currentProductBarCodeInfo = Validate.notNull(currentProductBarCodeInfo, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentProductBarCodeInfo.setModifyAccount(SecurityUtils.getUserAccount());
    currentProductBarCodeInfo.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentProductBarCodeInfo.setUnitCode(productBarCodeInfo.getUnitCode());
    currentProductBarCodeInfo.setBarCode(productBarCodeInfo.getBarCode());

    this.productBarCodeInfoRepository.saveAndFlush(currentProductBarCodeInfo);
    return currentProductBarCodeInfo;
  }

  /**
   * 在更新一个已有的ProductBarCodeInfo模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ProductBarCodeInfo productBarCodeInfo) {
    Validate.isTrue(!StringUtils.isBlank(productBarCodeInfo.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");

    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(productBarCodeInfo.getUnitCode(), "修改信息时，单位编号不能为空！");
    Validate.notBlank(productBarCodeInfo.getBarCode(), "修改信息时，条形码不能为空！");

    //条码不能重复
    Integer count = productBarCodeInfoRepository.countByBarCodeAndTenantCodeAndIdNot(productBarCodeInfo.getBarCode(), TenantUtils.getTenantCode(),productBarCodeInfo.getId());
    Validate.isTrue(count==0,"条形码%s已经存在，不能重复添加",productBarCodeInfo.getBarCode());

    Validate.isTrue(THIRTEEN_DIGITS.matcher(productBarCodeInfo.getBarCode()).matches(), "条形码必须为13位数字！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(productBarCodeInfo.getUnitCode() == null || productBarCodeInfo.getUnitCode().length() < 64, "单位编号,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productBarCodeInfo.getBarCode() == null || productBarCodeInfo.getBarCode().length() < 64, "条形码,在进行修改时填入值超过了限定长度(64)，请检查!");
  }

  @Override
  public ProductBarCodeInfo findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.productBarCodeInfoRepository.findDetailsById(id);
  }

  @Override
  public ProductBarCodeInfo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<ProductBarCodeInfo> op = productBarCodeInfoRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    ProductBarCodeInfo current = this.findById(id);
    if (current != null) {
      this.productBarCodeInfoRepository.delete(current);
    }
  }
} 
