package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.empower.business.product.entity.ProductBrand;
import com.bizunited.empower.business.product.repository.ProductBrandRepository;
import com.bizunited.empower.business.product.service.ProductBrandService;
import com.bizunited.empower.business.product.service.notifier.ProductBrandEventListener;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * ProductBrand业务模型的服务层接口实现
 * @author saturn
 */
@Service("ProductBrandServiceImpl")
public class ProductBrandServiceImpl implements ProductBrandService {

  /**
   * 最大层级限制
   */
  final static Integer MAX_LEVEL = 3;

  /**
   * 顶层
   */
  final static int TOP_LEVEL = 0;

  @Autowired(required = false)
  private List<ProductBrandEventListener> productBrandEventListener;

  @Autowired
  private ProductBrandRepository productBrandRepository;
  @Autowired
  private RedisMutexService redisMutexService;
  /**
   * 商品品牌编号前缀
   */
  private static final String PRODUCT_BRAND_CODE_PREFIX = "PB";
  /**
   * 商品品牌编号生成所使用的redis分布式锁的前缀
   */
  private static final String PRODUCT_BRAND_REDIS_LOCK_CODE = "PB_PRODUCT_BRAND_";

  @Transactional
  @Override
  public ProductBrand create(ProductBrand productBrand) {
    ProductBrand current = this.createForm(productBrand);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }
  @Transactional
  @Override
  public ProductBrand createForm(ProductBrand productBrand) {
   /*
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */
    Date now = new Date();
    productBrand.setCreateAccount(SecurityUtils.getUserAccount());
    productBrand.setCreateTime(now);
    productBrand.setModifyAccount(SecurityUtils.getUserAccount());
    productBrand.setModifyTime(now);
    // =============系统自构数据==================
    String tenantCode = this.getTenantCode();
    productBrand.setTenantCode(tenantCode);
    String productBrandCode = this.generateCode(tenantCode);
    productBrand.setBrandCode(productBrandCode);
    // ===============================
    this.createValidation(productBrand);

    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================

    this.productBrandRepository.save(productBrand);

    // 返回最终处理的结果，里面带有详细的关联信息
    return productBrand;
  }

  private String generateCode(String tenantCode) {
    String productBrandCode = null;
    String lockCode = StringUtils.join(PRODUCT_BRAND_REDIS_LOCK_CODE , tenantCode);
    try {
      this.redisMutexService.lock(PRODUCT_BRAND_REDIS_LOCK_CODE);
      String atomicNumber = this.redisMutexService.getAndIncrement(lockCode, 1,6);
      productBrandCode = StringUtils.join(PRODUCT_BRAND_CODE_PREFIX , atomicNumber);
    } catch (Exception e) {
      throw new RuntimeException("生成商品品牌编码失败",e);
    }finally {
      this.redisMutexService.unlock(PRODUCT_BRAND_REDIS_LOCK_CODE);
    }
    return productBrandCode;
  }

  public ProductBrand findByBrandCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    return this.productBrandRepository.findByBrandCode(code,TenantUtils.getTenantCode());
  }

  private String getTenantCode() {
    String tenantCode = TenantUtils.getTenantCode();
    return tenantCode;
  }


  /**
   * 在创建一个新的ProductBrand模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProductBrand productBrand) {
    Validate.notNull(productBrand , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(productBrand.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    productBrand.setId(null);
    Validate.notBlank(productBrand.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(productBrand.getBrandName(), "添加信息时，品牌名称不能为空！");
    Validate.notBlank(productBrand.getBrandCode(), "添加信息时，品牌编码不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(productBrand.getTenantCode() == null || productBrand.getTenantCode().length() < 255 , "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productBrand.getBrandName() == null || productBrand.getBrandName().length() < 1024 , "品牌名称,在进行添加时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(productBrand.getLogoRelativePath() == null || productBrand.getLogoRelativePath().length() < 1024 , "logo相对路径,在进行添加时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(productBrand.getLogoFileName() == null || productBrand.getLogoFileName().length() < 1024 , "logo文件名,在进行添加时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(productBrand.getBrandCode() == null || productBrand.getBrandCode().length() < 64 , "品牌编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    List<ProductBrand> list = productBrandRepository.findByBrandNameAndTenantCode(productBrand.getBrandName(), TenantUtils.getTenantCode());
    Validate.isTrue(CollectionUtils.isEmpty(list),"添加信息时，品牌名称已存在！");

  }
  @Transactional
  @Override
  public ProductBrand update(ProductBrand productBrand) {
    ProductBrand current = this.updateForm(productBrand);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }
  @Transactional
  @Override
  public ProductBrand updateForm(ProductBrand productBrand) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(productBrand);
    // ===================基本信息
    String currentId = productBrand.getId();
    Optional<ProductBrand> op_currentProductBrand = this.productBrandRepository.findById(currentId);
    ProductBrand currentProductBrand = op_currentProductBrand.orElse(null);
    currentProductBrand = Validate.notNull(currentProductBrand ,"未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentProductBrand.setModifyAccount(SecurityUtils.getUserAccount());
    currentProductBrand.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentProductBrand.setTenantCode(productBrand.getTenantCode());
    currentProductBrand.setBrandName(productBrand.getBrandName());
    currentProductBrand.setLogoRelativePath(productBrand.getLogoRelativePath());
    currentProductBrand.setLogoFileName(productBrand.getLogoFileName());
    currentProductBrand.setBrandCode(productBrand.getBrandCode());

    this.productBrandRepository.saveAndFlush(currentProductBrand);
    return currentProductBrand;
  }

  /**
   * 在更新一个已有的ProductBrand模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */

  private void updateValidation(ProductBrand productBrand) {
    Validate.isTrue(!StringUtils.isBlank(productBrand.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    productBrand.setTenantCode(TenantUtils.getTenantCode());
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(productBrand.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(productBrand.getBrandName(), "修改信息时，品牌名称不能为空！");
    Validate.notBlank(productBrand.getBrandCode(), "修改信息时，品牌编码不能为空！");

    // 重复性判断，同租户下名称不能重复
    Optional<ProductBrand> byId = productBrandRepository.findById(productBrand.getId());
    Validate.isTrue(byId.isPresent(), "修改信息时，修改对象在数据库中不存在！");
    ProductBrand currentForCode = byId.get();
    if (!currentForCode.getBrandName().equals(productBrand.getBrandName())) {
      List<ProductBrand> list = productBrandRepository.findByBrandNameAndTenantCode(productBrand.getBrandName(), TenantUtils.getTenantCode());
      Validate.isTrue(CollectionUtils.isEmpty(list),"修改信息时，名称已存在！");
    }
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(productBrand.getTenantCode() == null || productBrand.getTenantCode().length() < 255 , "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productBrand.getBrandName() == null || productBrand.getBrandName().length() < 1024 , "品牌名称,在进行修改时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(productBrand.getLogoRelativePath() == null || productBrand.getLogoRelativePath().length() < 1024 , "logo相对路径,在进行修改时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(productBrand.getLogoFileName() == null || productBrand.getLogoFileName().length() < 1024 , "logo文件名,在进行修改时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(productBrand.getBrandCode() == null || productBrand.getBrandCode().length() < 64 , "品牌编码,在进行修改时填入值超过了限定长度(64)，请检查!");
  }
  @Override
  public ProductBrand findDetailsById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    return this.productBrandRepository.findDetailsById(id);
  }
  @Override
  public ProductBrand findById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }

    Optional<ProductBrand> op = productBrandRepository.findById(id);
    return op.orElse(null);
  }
  @Override
  @Transactional
  public String deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    ProductBrand current = this.findById(id);
    if(current != null) {
      Set<Product> products = current.getProducts();
      Validate.isTrue(CollectionUtils.isEmpty(products),"已绑定商品，不允许删除！");
      this.productBrandRepository.delete(current);
      StringBuilder msg = new StringBuilder();
      // 触发事件
      if (!CollectionUtils.isEmpty(productBrandEventListener)) {
        for (ProductBrandEventListener listener : productBrandEventListener) {
          String resultMsg = listener.onDelete(current);
          if (StringUtils.isNotBlank(resultMsg)) {
            msg.append(resultMsg);
          }
        }
      }
      return msg.toString();
    }
    return "";
  }

  @Override
  public List<ProductBrand> findByCodeList(List<String> codeList) {
    if (CollectionUtils.isEmpty(codeList)) {
      return Lists.newLinkedList();
    }
    List<ProductBrand> list = productBrandRepository.findByTenantCodeAndBrandCodeIn(TenantUtils.getTenantCode(), codeList);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    // codeList 的顺序 要和结果List里的一致
    Map<String, ProductBrand> map = new HashMap<>(codeList.size());
    for (ProductBrand brand : list) {
      map.put(brand.getBrandCode(), brand);
    }
    List<ProductBrand> resultList = new ArrayList<>(codeList.size());
    for (String code : codeList) {
      resultList.add(map.get(code));
    }
    return resultList;
  }

  @Override
  public List<ProductBrand> findAllByTenantCode() {
    List<ProductBrand> brands = productBrandRepository.findByTenantCode(TenantUtils.getTenantCode());
    return brands;
  }

}
