package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.product.dto.ProductCategoryDto;
import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.empower.business.product.entity.ProductCategory;
import com.bizunited.empower.business.product.repository.ProductCategoryRepository;
import com.bizunited.empower.business.product.service.ProductCategoryService;
import com.bizunited.empower.business.product.service.notifier.ProductCategoryEventListener;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * ProductCategory业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("ProductCategoryServiceImpl")
public class ProductCategoryServiceImpl implements ProductCategoryService {

  /**
   * 最大层级限制
   */
  final static Integer MAX_LEVEL = 3;

  /**
   * 顶层标识
   */
  final static Integer TOP_LEVEL = 0;

  @Autowired(required = false)
  private List<ProductCategoryEventListener> productCategoryEventListener;

  @Autowired
  private ProductCategoryRepository productCategoryRepository;
  @Autowired
  private RedisMutexService redisMutexService;
  /**
   * 商品品牌编号前缀
   */
  private static final String PRODUCT_CATEGORY_CODE_PREFIX = "PC";
  /**
   * 商品品牌编号生成所使用的redis分布式锁的前缀
   */
  private static final String PRODUCT_CATEGORY_REDIS_LOCK_CODE = "PC_PRODUCT_CATEGORY_";

  @Transactional
  @Override
  public ProductCategory create(ProductCategory productCategory) {
    ProductCategory current = this.createForm(productCategory);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductCategory createForm(ProductCategory productCategory) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    Date now = new Date();
    productCategory.setCreateAccount(SecurityUtils.getUserAccount());
    productCategory.setCreateTime(now);
    productCategory.setModifyAccount(SecurityUtils.getUserAccount());
    productCategory.setModifyTime(now);

    // =============系统自构数据==================
    String tenantCode = this.getTenantCode();
    productCategory.setTenantCode(tenantCode);
    String productCategoryCode = this.generateCode(tenantCode);
    productCategory.setCode(productCategoryCode);
    // 获取父级的flatCode；
    String pFlatCode = this.getParentFlatCode(productCategory.getParentCode());
    productCategory.setFlatCode(StringUtils.join(pFlatCode, productCategoryCode));
    // ===============================
    this.createValidation(productCategory);

    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================

    this.productCategoryRepository.save(productCategory);

    // 返回最终处理的结果，里面带有详细的关联信息
    return productCategory;
  }

  private String generateCode(String tenantCode) {
    String productBrandCode = null;
    String lockCode = StringUtils.join(PRODUCT_CATEGORY_REDIS_LOCK_CODE, tenantCode);
    try {
      this.redisMutexService.lock(PRODUCT_CATEGORY_REDIS_LOCK_CODE);
      String atomicNumber = this.redisMutexService.getAndIncrement(lockCode, 1, 6);
      productBrandCode = StringUtils.join(PRODUCT_CATEGORY_CODE_PREFIX, atomicNumber);
    } catch (Exception e) {
      throw new RuntimeException("生成商品分类编码失败", e);
    } finally {
      this.redisMutexService.unlock(PRODUCT_CATEGORY_REDIS_LOCK_CODE);
    }
    return productBrandCode;
  }

  private String getTenantCode() {
    String tenantCode = TenantUtils.getTenantCode();
    return tenantCode;
  }

  private String getParentFlatCode(String pCode) {
    String flatCode = "";
    if (StringUtils.isNotBlank(pCode)) {
      ProductCategory category = this.findByCode(pCode);
      Validate.notNull(category, "父级分类不存在");
      flatCode = category.getFlatCode();
    }
    return flatCode;
  }

  /**
   * 在创建一个新的ProductCategory模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProductCategory productCategory) {
    Validate.notNull(productCategory, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(productCategory.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    productCategory.setId(null);
    Validate.notBlank(productCategory.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(productCategory.getName(), "添加信息时，分类名称不能为空！");
    Validate.notBlank(productCategory.getCode(), "添加信息时，分类编码不能为空！");
    Validate.isTrue(productCategory.getLevel() != null && productCategory.getLevel() < MAX_LEVEL, "添加信息时，层级只有【%d】级", MAX_LEVEL);
    if (productCategory.getLevel() != 0) {
      Validate.notBlank(productCategory.getParentCode(), "添加信息时，父级编号不能为空！");
      ProductCategory productCategoryParent = productCategoryRepository.findByCodeAndTenantCode(productCategory.getParentCode(), TenantUtils.getTenantCode());
      // 检验父节点是不是末位节点
      Validate.notNull(productCategoryParent, "添加信息时，父节点不存在！");
      Validate.isTrue(productCategoryParent.getLevel() < MAX_LEVEL - 1, "添加信息时，末位节点不允许再添加下级节点");
    }
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(productCategory.getTenantCode() == null || productCategory.getTenantCode().length() < 255, "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productCategory.getName() == null || productCategory.getName().length() < 255, "分类名称,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productCategory.getCode() == null || productCategory.getCode().length() < 64, "分类编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productCategory.getParentCode() == null || productCategory.getParentCode().length() < 64, "父级编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productCategory.getFlatCode() == null || productCategory.getFlatCode().length() < 255, "快速编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    ProductCategory currentProductCategory = this.findByCode(productCategory.getCode());
    Validate.isTrue(currentProductCategory == null, "分类编码已存在,请检查");
    List<ProductCategory> list = productCategoryRepository.findByNameAndTenantCode(productCategory.getName(), TenantUtils.getTenantCode());
    Validate.isTrue(CollectionUtils.isEmpty(list), "添加信息时，分类名称已存在！");
  }

  @Transactional
  @Override
  public ProductCategory update(ProductCategory productCategory) {
    ProductCategory current = this.updateForm(productCategory);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductCategory updateForm(ProductCategory productCategory) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(productCategory);
    // ===================基本信息
    String currentId = productCategory.getId();
    Optional<ProductCategory> op_currentProductCategory = this.productCategoryRepository.findById(currentId);
    ProductCategory currentProductCategory = op_currentProductCategory.orElse(null);
    currentProductCategory = Validate.notNull(currentProductCategory, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentProductCategory.setModifyAccount(SecurityUtils.getUserAccount());
    currentProductCategory.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentProductCategory.setTenantCode(productCategory.getTenantCode());
    currentProductCategory.setName(productCategory.getName());
    currentProductCategory.setCode(productCategory.getCode());
    currentProductCategory.setLevel(productCategory.getLevel());
    currentProductCategory.setParentCode(productCategory.getParentCode());
    currentProductCategory.setFlatCode(productCategory.getFlatCode());

    this.productCategoryRepository.saveAndFlush(currentProductCategory);
    return currentProductCategory;
  }

  /**
   * 在更新一个已有的ProductCategory模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ProductCategory productCategory) {
    Validate.isTrue(!StringUtils.isBlank(productCategory.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    productCategory.setTenantCode(TenantUtils.getTenantCode());
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(productCategory.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(productCategory.getName(), "修改信息时，分类名称不能为空！");
    Validate.notBlank(productCategory.getCode(), "修改信息时，分类编码不能为空！");
    if (productCategory.getLevel() != 0) {
      Validate.notBlank(productCategory.getParentCode(), "修改信息时，父级编号不能为空！");
      // 检验父节点是不是末位节点
      ProductCategory productCategoryParent = productCategoryRepository.findByCodeAndTenantCode(productCategory.getParentCode(), TenantUtils.getTenantCode());
      Validate.notNull(productCategoryParent, "修改信息时，父节点不存在！");
      Validate.isTrue(productCategoryParent.getLevel() < MAX_LEVEL - 1, "修改信息时，末位节点不允许作为父级节点");
    }

    // 重复性判断，基本属性，需要满足unique = true
    ProductCategory currentForCode = this.findByCode(productCategory.getCode());
    Validate.isTrue(currentForCode != null && StringUtils.equals(currentForCode.getId(), productCategory.getId()), "分类编码已存在,请检查");
    // 重复性判断，同租户下名称不能重复
    if (!currentForCode.getName().equals(productCategory.getName())) {
      List<ProductCategory> list = productCategoryRepository.findByNameAndTenantCode(productCategory.getName(), TenantUtils.getTenantCode());
      Validate.isTrue(CollectionUtils.isEmpty(list), "修改信息时，分类名称已存在！");
    }
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(productCategory.getTenantCode() == null || productCategory.getTenantCode().length() < 255, "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productCategory.getName() == null || productCategory.getName().length() < 255, "分类名称,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productCategory.getCode() == null || productCategory.getCode().length() < 64, "分类编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productCategory.getParentCode() == null || productCategory.getParentCode().length() < 64, "父级编号,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productCategory.getFlatCode() == null || productCategory.getFlatCode().length() < 255, "快速编号,在进行修改时填入值超过了限定长度(255)，请检查!");
  }

  @Override
  public ProductCategory findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.productCategoryRepository.findDetailsById(id);
  }

  @Override
  public ProductCategory findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<ProductCategory> op = productCategoryRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public String deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    ProductCategory current = this.findById(id);
    if (current != null) {
      Set<Product> products = current.getProducts();
      Validate.isTrue(CollectionUtils.isEmpty(products), "已绑定商品，不允许删除！");
      List<ProductCategory> child = this.findByPCode(current.getCode());
      Validate.isTrue(CollectionUtils.isEmpty(child), "当前分类下有子分类，不可删除！");
      this.productCategoryRepository.delete(current);
      StringBuilder msg = new StringBuilder();
      // 触发事件
      if (!org.springframework.util.CollectionUtils.isEmpty(productCategoryEventListener)) {
        for (ProductCategoryEventListener listener : productCategoryEventListener) {
          String resultMsg = listener.onDelete(current);
          if (StringUtils.isNotBlank(resultMsg)) {
            msg.append(resultMsg);
          }
        }
      }
      return msg.toString();
    }
    return "";
  }

  @Override
  public ProductCategory findByCode(String code) {
    if (StringUtils.isBlank(code)) {
      return null;
    }
    return this.productCategoryRepository.findByCode(code, TenantUtils.getTenantCode());
  }

  /**
   * 分页查询
   *
   * @param pageable
   * @return
   */
  @Override
  public Page<ProductCategory> findByConditions(Pageable pageable) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    Page<ProductCategory> rolePage = this.productCategoryRepository.findByTenantCodeOrderByCreateTimeDesc(TenantUtils.getTenantCode(), pageable);
    if (CollectionUtils.isEmpty(rolePage.getContent())) {
      return Page.empty(pageable);
    }
    return new PageImpl<>(Lists.newArrayList(rolePage.getContent()), pageable, rolePage.getTotalElements());

  }


  /**
   * 根据父级code查询下一级，不传则查询所有顶层
   *
   * @param pCode
   * @return
   */
  @Override
  public List<ProductCategory> findByPCode(String pCode) {
    if (StringUtils.isBlank(pCode)) {
      return this.productCategoryRepository.findByLevelAndTenantCodeOrderByCreateTimeDesc(TOP_LEVEL, TenantUtils.getTenantCode());
    }
    return this.productCategoryRepository.findByParentCode(pCode, TenantUtils.getTenantCode());
  }

  @Override
  public List<ProductCategory> findByCodeList(List<String> codeList) {
    if (CollectionUtils.isEmpty(codeList)) {
      return Lists.newLinkedList();
    }
    List<ProductCategory> list = productCategoryRepository.findByTenantCodeAndCodeIn(TenantUtils.getTenantCode(), codeList);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    // codeList 的顺序 要和结果List里的一致
    Map<String, ProductCategory> map = new HashMap<>(codeList.size());
    for (ProductCategory category : list) {
      map.put(category.getCode(), category);
    }
    List<ProductCategory> resultList = new ArrayList<>(codeList.size());
    for (String code : codeList) {
      resultList.add(map.get(code));
    }
    return resultList;
  }

  @Override
  @Transactional
  public void saveBatch(Set<ProductCategoryDto> productCategoryDtos) {
    Validate.notEmpty(productCategoryDtos, "批量保存分类信息不能为空");
    /**
     * 1.初始化分类对象信息
     * 2.递归保存对象信息
     */
    int level = TOP_LEVEL;
    this.saveChildren(productCategoryDtos, null, level);
  }

  /**
   * 递归保存子项信息
   *
   * @param productCategoryDtos
   * @param parent
   * @param level
   * @return
   */
  private void saveChildren(Set<ProductCategoryDto> productCategoryDtos, ProductCategory parent, int level) {
    for (ProductCategoryDto productCategoryDto : productCategoryDtos) {
      ProductCategory productCategory = new ProductCategory();
      productCategory.setName(productCategoryDto.getName());
      productCategory.setLevel(level);
      if (level != TOP_LEVEL) {
        productCategory.setParentCode(parent.getCode());
      }
      ProductCategory parentCategory = this.createForm(productCategory);
      if (!CollectionUtils.isEmpty(productCategoryDto.getChildren())) {
        int childrenLevel = level + 1;
        this.saveChildren(productCategoryDto.getChildren(), parentCategory, childrenLevel);
      }
    }
  }

}
