package com.bizunited.empower.business.product.service.internal;
import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.empower.business.product.entity.ProductDescription;
import com.bizunited.empower.business.product.repository.ProductDescriptionRepository;
import com.bizunited.empower.business.product.service.ProductDescriptionService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.util.HtmlUtils;

import javax.transaction.Transactional;
import java.util.Objects;
import java.util.Optional;

/**
 * ProductDescription业务模型的服务层接口实现
 * @author saturn
 */
@Service("ProductDescriptionServiceImpl")
public class ProductDescriptionServiceImpl implements ProductDescriptionService {
  @Autowired
  private ProductDescriptionRepository productDescriptionRepository;

  /**
   * 新增商品描述
   * @param description
   * @param product
   */
  @Override
  public void createInsertAbleEntity(ProductDescription description, Product product) {
    // 商品描述非必填
    if (Objects.isNull(description)) {
      return;
    }
    // 什么信息也不传，也认为是非必填
    if (StringUtils.isBlank(description.getDescription())) {
      return;
    }
    description.setProduct(product);
    this.createInsertAbleEntity(description);
    productDescriptionRepository.save(description);
  }

  /**
   * 为方便上层方法调用（已获取对象），写的根据对象删除
   * @param productDescription
   */
  @Override
  public void delete(ProductDescription productDescription) {
    if (Objects.nonNull(productDescription)) {
      this.productDescriptionRepository.delete(productDescription);
    }
  }

  /**
   * 根据编号查询描述信息
   * @param productCode
   * @return
   */
  @Override
  public ProductDescription findByProductCode(String productCode) {
    Validate.notBlank(productCode, "商品编码必传");
    return this.productDescriptionRepository.findByTenantCodeAndProductCode(TenantUtils.getTenantCode(),productCode);
  }


  private void createInsertAbleEntity(ProductDescription description) {
    description.setTenantCode(TenantUtils.getTenantCode());
    this.createValidation(description);
  }

  @Transactional
  @Override
  public ProductDescription create(ProductDescription productDescription) {
    ProductDescription current = this.createForm(productDescription);
    //====================================================
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }
  @Transactional
  @Override
  public ProductDescription createForm(ProductDescription productDescription) {
    this.createInsertAbleEntity(productDescription);

    // ===============================
    //  和业务有关的验证填写在这个区域
    // ===============================
    // 商品描述信息富文本内容转义
    if(StringUtils.isNotBlank(productDescription.getDescription())){
      productDescription.setDescription(HtmlUtils.htmlEscape(productDescription.getDescription()));
    }
    this.productDescriptionRepository.save(productDescription);
    // 返回最终处理的结果，里面带有详细的关联信息
    return productDescription;
  }
  /**
   * 在创建一个新的ProductDescription模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProductDescription productDescription) {

    Long descriptionCount = productDescriptionRepository.countProductDescriptionByTenantCodeAndProductId(TenantUtils.getTenantCode(), productDescription.getProduct().getId());
    Validate.isTrue(descriptionCount.equals(0L),"当前商品已有描述信息");
    Validate.notNull(productDescription , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(productDescription.getId()), "添加信息时，商品描述信息的数据编号（主键）不能有值！");
    productDescription.setId(null);
  }
  @Transactional
  @Override
  public ProductDescription update(ProductDescription productDescription) {
    ProductDescription current = this.updateForm(productDescription);
    //====================================================
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }
  @Transactional
  @Override
  public ProductDescription updateForm(ProductDescription productDescription) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(productDescription);
    // ===================基本信息
    String currentId = productDescription.getId();
    Optional<ProductDescription> op_currentProductDescription = this.productDescriptionRepository.findById(currentId);
    ProductDescription currentProductDescription = op_currentProductDescription.orElse(null);
    currentProductDescription = Validate.notNull(currentProductDescription ,"未发现指定的原始模型对象信息");
    // 开始赋值——更新时间与更新人
    // 开始重新赋值——一般属性
    //内容转义
    if(StringUtils.isNotBlank(productDescription.getDescription())){
      currentProductDescription.setDescription(HtmlUtils.htmlEscape(productDescription.getDescription()));
    }
    this.productDescriptionRepository.saveAndFlush(currentProductDescription);
    return currentProductDescription;
  }
  /**
   * 在更新一个已有的ProductDescription模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ProductDescription productDescription) {
    Validate.isTrue(!StringUtils.isBlank(productDescription.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
  }
  @Override
  public ProductDescription findDetailsByProduct(String productCode) {
    if(StringUtils.isBlank(productCode)) {
      return null;
    }
    ProductDescription productDescription = this.productDescriptionRepository.findByTenantCodeAndProductCode(TenantUtils.getTenantCode(),productCode);
    if(productDescription == null){
      ProductDescription pd = new ProductDescription();
      pd.setDescription("");
      return pd;
    }
    if(StringUtils.isNotBlank(productDescription.getDescription())){
      productDescription.setDescription(HtmlUtils.htmlUnescape(productDescription.getDescription()));
    }
    return productDescription;
  }
  @Override
  public ProductDescription findDetailsById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    ProductDescription productDescription = productDescriptionRepository.findDetailsById(id);
    if(productDescription == null){
      ProductDescription pd = new ProductDescription();
      pd.setDescription("");
      return pd;
    }
    if(StringUtils.isNotBlank(productDescription.getDescription())){
      productDescription.setDescription(HtmlUtils.htmlUnescape(productDescription.getDescription()));
    }
    return productDescription;
  }
  @Override
  public ProductDescription findById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    Optional<ProductDescription> op = productDescriptionRepository.findById(id);
    ProductDescription productDescription = op.orElse(null);
    if(productDescription == null){
      ProductDescription pd = new ProductDescription();
      pd.setDescription("");
      return pd;
    }
    if(StringUtils.isNotBlank(productDescription.getDescription())){
      productDescription.setDescription(HtmlUtils.htmlUnescape(productDescription.getDescription()));
    }
    return productDescription;
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    ProductDescription current = this.findById(id);
    Validate.notNull(current,"没有当前ID对应的信息");
    if(current != null) {
      this.productDescriptionRepository.delete(current);
    }
  }
}
