package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.empower.business.product.entity.ProductFile;
import com.bizunited.empower.business.product.entity.ProductImageResource;
import com.bizunited.empower.business.product.entity.ProductSpecification;
import com.bizunited.empower.business.product.repository.ProductFileRepository;
import com.bizunited.empower.business.product.service.ProductFileService;
import com.bizunited.empower.business.product.service.ProductImageResourceService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Sets;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * ProductFile业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("ProductFileServiceImpl")
public class ProductFileServiceImpl implements ProductFileService {
  @Autowired
  private ProductFileRepository productFileRepository;
  @Autowired
  private ProductImageResourceService productImageResourceService;

  /**
   * 最多5张图片
   */
  private static final Integer MAX_COUNT=5;


  /**
   * 批量新增商品文件
   *
   * @param files
   * @param product
   */
  @Override
  @Transactional
  public void createInsertAbleEntitySet(Set<ProductFile> files, Product product) {
    Validate.notNull(product, "未传入商品信息！");
    if (CollectionUtils.isNotEmpty(files)) {
      int basic = 0;
      for (ProductFile file : files) {
        file.setProduct(product);
        this.createInsertAbleEntity(file);
        Boolean isMainImage= file.getMainImage();
        if(isMainImage){
          basic++;
        }
      }
      Validate.isTrue(basic == 1, "主图必须设定");

      productFileRepository.saveAll(files);
    }
  }

  @Override
  public void deleteByBatch(Set<ProductFile> productFiles) {
    if (CollectionUtils.isNotEmpty(productFiles)) {
      this.productFileRepository.deleteAll(productFiles);
    }
  }

  @Override
  public void updateFormBatch(Set<ProductFile> files, Product product) {
    /** 6.修改商品文件
     *    6.1 删去不存在了的
     *    6.2 为新增信息赋值
     *    6.3 旧信息覆盖
     */
    this.preCheck(files, product);
    // 分为有ID的，无ID的
    Map<Boolean, List<ProductFile>> idBooleanMap = files.stream()
        .collect(Collectors.partitioningBy(productFile -> StringUtils.isNotBlank(productFile.getId())));
    List<ProductFile> idTrueList = idBooleanMap.get(true);
    List<ProductFile> idFalseList = idBooleanMap.get(false);
    // 6.1删去不存在了的
    List<String> ids = idTrueList.stream().map(ProductFile::getId).filter(StringUtils::isNotEmpty).collect(Collectors.toList());
    Set<ProductFile> needDelete ;
    if (CollectionUtils.isEmpty(ids)){
      needDelete = productFileRepository.findByProduct(TenantUtils.getTenantCode(), product.getProductCode());
    } else {
      needDelete = productFileRepository.findByProductAndIdNotIn(TenantUtils.getTenantCode(), product.getProductCode(), ids);
    }
    if (CollectionUtils.isNotEmpty(needDelete)) {
      //图片删除前修改资源图片状态
      for (ProductFile productFile : needDelete) {
        if(productFile.getProductImageResource()!=null){
          productImageResourceService.updateUseImage(productFile.getProductImageResource());
        }
      }

      productFileRepository.deleteAll(needDelete);
    }
    Set<ProductFile> saveAll = new HashSet<>(files.size());
    // 6.2为新增信息赋值
    for (ProductFile productFile : idFalseList) {
      productFile.setProduct(product);
      this.createInsertAbleEntity(productFile);
      saveAll.add(productFile);
    }
    // 6.3 旧信息覆盖
    for (ProductFile productFile : idTrueList) {
      //修改之前的图片资源状态
      ProductFile oldFile = this.findById(productFile.getId());
      productFile.setProduct(product);
      productFile.setTenantCode(TenantUtils.getTenantCode());

      //验证资源图片是否修改
      if (productFile.getProductImageResource()!=null&&StringUtils.isNotBlank(productFile.getProductImageResource().getId())&&oldFile.getProductImageResource()!=null&&oldFile.getProductImageResource().getId().equals(productFile.getProductImageResource().getId())) {
        productFile.setProductImageResource(oldFile.getProductImageResource());
      }else{
        //保存图片到资源库中
        ProductImageResource productImageResource = new ProductImageResource();
        productImageResource.setUseImage(true);
        productImageResource.setProduct(product);
        productImageResource.setFileName(productFile.getFileName());
        productImageResource.setRelativePath(productFile.getRelativePath());
        productImageResource.setType(productFile.getType());
        productImageResourceService.create(productImageResource);
        //设置图片资源库信息
        productFile.setProductImageResource(productImageResource);
        //将之前的图片设置为 未使用,这里处理之前老数据可能没有图片资源的情况
        if(oldFile.getProductImageResource()!=null) {
          productImageResourceService.updateUseImage(oldFile.getProductImageResource(), productFile.getId(), null);
        }
      }
      this.updateValidation(productFile);

      saveAll.add(productFile);
    }
    if (CollectionUtils.isNotEmpty(saveAll)) {
      productFileRepository.saveAll(saveAll);
    }
  }

  private void preCheck(Set<ProductFile> files, Product product) {
    Validate.notNull(product, "商品信息不能为空");
    if (CollectionUtils.isNotEmpty(files)) {
      int basic = 0;
      for (ProductFile file : files) {
        Boolean isMainImage= file.getMainImage();
        if(isMainImage){
          basic++;
        }
      }
      Validate.isTrue(basic == 1, "主图必须设定");
    }

  }

  /**
   * 赋值租户，并校验
   *
   * @param file
   */
  private void createInsertAbleEntity(ProductFile file) {
    file.setTenantCode(TenantUtils.getTenantCode());
    //保存图片到资源库中
    ProductImageResource productImageResource = new ProductImageResource();
    productImageResource.setUseImage(true);
    productImageResource.setProduct(file.getProduct());
    productImageResource.setFileName(file.getFileName());
    productImageResource.setRelativePath(file.getRelativePath());
    productImageResource.setType(file.getType());
    productImageResourceService.create(productImageResource);
    file.setProductImageResource(productImageResource);

    this.createValidation(file);
  }

  @Transactional
  @Override
  public ProductFile create(ProductFile productFile) {
    ProductFile current = this.createForm(productFile);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductFile createForm(ProductFile productFile) {

    //图片资源库设置
    Validate.notNull(productFile.getProductImageResource(),"图片资源信息不能为空");
    Validate.notBlank(productFile.getProductImageResource().getId(),"图片资源信息不能为空");
    Validate.notNull(productFile.getProduct(),"图片商品信息不能为空");
    Validate.notBlank(productFile.getProduct().getId(),"图片商品ID不能为空");

    Integer count = productFileRepository.countByProductId(productFile.getProduct().getId());
    Validate.isTrue(count<MAX_COUNT,"商品图片最多5张");

    ProductImageResource productImageResource = productImageResourceService.findById(productFile.getProductImageResource().getId());
    Validate.notNull(productImageResource,"图片资源信息不存在");
    //赋值 图片资源信息
    productFile.setProductImageResource(productImageResource);


    productFile.setTenantCode(TenantUtils.getTenantCode());
    this.createValidation(productFile);

    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================

    this.productFileRepository.save(productFile);

    //修改图片资源使用状态
    productImageResourceService.updateUseImage(productImageResource);

    // 返回最终处理的结果，里面带有详细的关联信息
    return productFile;
  }

  /**
   * 在创建一个新的ProductFile模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProductFile productFile) {
    Validate.notNull(productFile, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(productFile.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    productFile.setId(null);
    Validate.notBlank(productFile.getRelativePath(), "添加信息时，相对路径不能为空！");
    Validate.notBlank(productFile.getFileName(), "添加信息时，文件名不能为空！");
    Validate.notNull(productFile.getMainImage(), "添加信息时，主图不能为空！");
    Validate.notNull(productFile.getSortIndex(), "添加信息时，排序信息不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(productFile.getRelativePath() == null || productFile.getRelativePath().length() < 255, "相对路径,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productFile.getFileName() == null || productFile.getFileName().length() < 255, "文件名,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productFile.getType() == null || productFile.getType().length() < 255, "图片类型,在进行添加时填入值超过了限定长度(255)，请检查!");
  }

  @Transactional
  @Override
  public ProductFile update(ProductFile productFile) {
    ProductFile current = this.updateForm(productFile);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductFile updateForm(ProductFile productFile) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(productFile);
    // ===================基本信息
    String currentId = productFile.getId();
    Optional<ProductFile> op_currentProductFile = this.productFileRepository.findById(currentId);
    ProductFile currentProductFile = op_currentProductFile.orElse(null);
    currentProductFile = Validate.notNull(currentProductFile, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    // 开始重新赋值——一般属性
    currentProductFile.setRelativePath(productFile.getRelativePath());
    currentProductFile.setFileName(productFile.getFileName());
    currentProductFile.setType(productFile.getType());
    currentProductFile.setProduct(productFile.getProduct());

    this.productFileRepository.saveAndFlush(currentProductFile);
    return currentProductFile;
  }

  /**
   * 在更新一个已有的ProductFile模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ProductFile productFile) {
    Validate.isTrue(!StringUtils.isBlank(productFile.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");

    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(productFile.getRelativePath(), "修改信息时，相对路径不能为空！");
    Validate.notBlank(productFile.getFileName(), "修改信息时，文件名不能为空！");
    Validate.notNull(productFile.getMainImage(), "添加信息时，主图不能为空！");
    Validate.notNull(productFile.getSortIndex(), "添加信息时，排序信息不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(productFile.getRelativePath() == null || productFile.getRelativePath().length() < 255, "相对路径,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productFile.getFileName() == null || productFile.getFileName().length() < 255, "文件名,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productFile.getType() == null || productFile.getType().length() < 255, "图片类型,在进行修改时填入值超过了限定长度(255)，请检查!");

    // 关联性判断，关联属性判断，需要满足ManyToOne或者OneToOne，且not null 且是主模型
  }

  @Override
  public Set<ProductFile> findDetailsByProduct(String product) {
    if (StringUtils.isBlank(product)) {
      return Sets.newHashSet();
    }
    return this.productFileRepository.findDetailsByProduct(product,TenantUtils.getTenantCode());
  }

  @Override
  public ProductFile findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.productFileRepository.findDetailsById(id);
  }

  @Override
  public ProductFile findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<ProductFile> op = productFileRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    ProductFile current = this.findById(id);
    if (current != null) {
      this.productFileRepository.delete(current);
      if(current.getProductImageResource()!=null){
        //修改当前图片资源状态
        productImageResourceService.updateUseImage(current.getProductImageResource());


      }
    }
  }

  @Override
  public void updateIsMainImage(String id) {
    Validate.notBlank(id,"图片ID不能为空");
    ProductFile current = this.findById(id);
    Validate.notNull(current,"当前图片信息不存在");
    //获取旧的主图
    ProductFile oldFile = productFileRepository.findByProductIdAndIsMainImage(current.getProduct().getId(),true);
    //将旧的主图设置为否
    //如果 旧的主图不存在，那么直接设置当前对象为主图
    if(oldFile!=null){
     oldFile.setMainImage(false);
     productFileRepository.saveAndFlush(oldFile);
    }
    //设置当前图片为主图
    current.setMainImage(true);
    productFileRepository.saveAndFlush(current);


  }
}
