package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.common.util.SecurityUtils;
import com.bizunited.empower.business.product.entity.ProductImageResource;
import com.bizunited.empower.business.product.entity.ProductSpecification;
import com.bizunited.empower.business.product.repository.ProductFileRepository;
import com.bizunited.empower.business.product.repository.ProductImageResourceRepository;
import com.bizunited.empower.business.product.repository.ProductSpecificationRepository;
import com.bizunited.empower.business.product.service.ProductImageResourceService;
import com.bizunited.empower.business.product.service.ProductService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.platform.venus.sdk.service.file.FileHandleService;
import com.bizunited.platform.venus.sdk.vo.OrdinaryFileVo;
import com.google.common.collect.Sets;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * ProductImageResource业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("ProductImageResourceServiceImpl")
public class ProductImageResourceServiceImpl implements ProductImageResourceService {
  @Autowired
  private FileHandleService fileHandleService;
  @Autowired
  private ProductImageResourceRepository productImageResourceRepository;

  @Autowired
  private ProductSpecificationRepository productSpecificationRepository;

  @Autowired
  private ProductFileRepository productFileRepository;

  @Transactional
  @Override
  public ProductImageResource create(ProductImageResource productImageResource) {
    ProductImageResource current = this.createForm(productImageResource);
    //====================================================
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductImageResource createForm(ProductImageResource productImageResource) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    Date now = new Date();
    productImageResource.setCreateAccount(SecurityUtils.getUserAccount());
    productImageResource.setCreateTime(now);
    productImageResource.setModifyAccount(SecurityUtils.getUserAccount());
    productImageResource.setModifyTime(now);
    productImageResource.setTenantCode(TenantUtils.getTenantCode());

    this.createValidation(productImageResource);

    // ===============================
    //  和业务有关的验证填写在这个区域
    // ===============================

    this.productImageResourceRepository.save(productImageResource);

    // 返回最终处理的结果，里面带有详细的关联信息
    return productImageResource;
  }

  /**
   * 在创建一个新的ProductImageResource模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProductImageResource productImageResource) {
    Validate.notNull(productImageResource, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(productImageResource.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    productImageResource.setId(null);
    Validate.notBlank(productImageResource.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(productImageResource.getRelativePath(), "添加信息时，相对路径不能为空！");
    Validate.notBlank(productImageResource.getFileName(), "添加信息时，文件名不能为空！");
    Validate.notNull(productImageResource.getUseImage(), "添加信息时，是否使用图片不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(productImageResource.getTenantCode() == null || productImageResource.getTenantCode().length() < 255, "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productImageResource.getRelativePath() == null || productImageResource.getRelativePath().length() < 255, "相对路径,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productImageResource.getFileName() == null || productImageResource.getFileName().length() < 255, "文件名,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productImageResource.getType() == null || productImageResource.getType().length() < 255, "图片类型,在进行添加时填入值超过了限定长度(255)，请检查!");
  }

  @Transactional
  @Override
  public ProductImageResource update(ProductImageResource productImageResource) {
    ProductImageResource current = this.updateForm(productImageResource);
    //====================================================
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductImageResource updateForm(ProductImageResource productImageResource) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(productImageResource);
    // ===================基本信息
    String currentId = productImageResource.getId();
    Optional<ProductImageResource> op_currentProductImageResource = this.productImageResourceRepository.findById(currentId);
    ProductImageResource currentProductImageResource = op_currentProductImageResource.orElse(null);
    currentProductImageResource = Validate.notNull(currentProductImageResource, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentProductImageResource.setModifyAccount(SecurityUtils.getUserAccount());
    currentProductImageResource.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentProductImageResource.setTenantCode(productImageResource.getTenantCode());
    currentProductImageResource.setRelativePath(productImageResource.getRelativePath());
    currentProductImageResource.setFileName(productImageResource.getFileName());
    currentProductImageResource.setType(productImageResource.getType());
    currentProductImageResource.setUseImage(productImageResource.getUseImage());
    currentProductImageResource.setProduct(productImageResource.getProduct());

    this.productImageResourceRepository.saveAndFlush(currentProductImageResource);
    return currentProductImageResource;
  }

  /**
   * 在更新一个已有的ProductImageResource模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ProductImageResource productImageResource) {
    Validate.isTrue(!StringUtils.isBlank(productImageResource.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");

    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(productImageResource.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(productImageResource.getRelativePath(), "修改信息时，相对路径不能为空！");
    Validate.notBlank(productImageResource.getFileName(), "修改信息时，文件名不能为空！");
    Validate.notNull(productImageResource.getUseImage(), "修改信息时，是否使用图片不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(productImageResource.getTenantCode() == null || productImageResource.getTenantCode().length() < 255, "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productImageResource.getRelativePath() == null || productImageResource.getRelativePath().length() < 255, "相对路径,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productImageResource.getFileName() == null || productImageResource.getFileName().length() < 255, "文件名,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productImageResource.getType() == null || productImageResource.getType().length() < 255, "图片类型,在进行修改时填入值超过了限定长度(255)，请检查!");

    // 关联性判断，关联属性判断，需要满足ManyToOne或者OneToOne，且not null 且是主模型
  }

  @Override
  public Set<ProductImageResource> findDetailsByProduct(String product) {
    if (StringUtils.isBlank(product)) {
      return Sets.newHashSet();
    }
    return this.productImageResourceRepository.findDetailsByProduct(product);
  }

  @Override
  public ProductImageResource findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.productImageResourceRepository.findDetailsById(id);
  }

  @Override
  public List<ProductImageResource> findByProductIdAndUseImage(String productId, Boolean useImage) {
    if (StringUtils.isBlank(productId)) {
      return Lists.newArrayList();
    }
    //当图片使用筛选字段为空时，查询全部
    List<ProductImageResource> productImageResources = Lists.newArrayList();
    if (useImage == null) {
      productImageResources = productImageResourceRepository.findByProductId(productId);

    } else {
      productImageResources = productImageResourceRepository.findByProductIdAndUseImage(productId, useImage);
    }
    for (ProductImageResource productImageResource : productImageResources) {
      //丰富原文件名，和文件大小信息
      OrdinaryFileVo fileVo = fileHandleService.findByFileNameAndRelativeLocal(productImageResource.getFileName(), productImageResource.getRelativePath());
      if (fileVo == null) {
        continue;
      }
      //源文件名称
      productImageResource.setOriginalFileName(fileVo.getOriginalFileName());
      //文件大小
      productImageResource.setFileSize(fileVo.getFileSize());

    }

    return productImageResources;
  }

  @Override
  public ProductImageResource findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<ProductImageResource> op = productImageResourceRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    ProductImageResource current = this.findById(id);

    if (current != null) {
      Validate.isTrue(!current.getUseImage(),"该商品图片正在使用，不能删除");
      this.productImageResourceRepository.delete(current);
    }
  }

  @Override
  @Transactional
  public List<ProductImageResource> saveBatch(List<ProductImageResource> productImageResources) {
    Validate.notEmpty(productImageResources, "图片资源信息不能为空");
    for (ProductImageResource productImageResource : productImageResources) {
      this.createForm(productImageResource);
    }
    return productImageResources;
  }

  @Override
  @Transactional
  public void updateUseImage(ProductImageResource oldImageResource, ProductImageResource newImageResource) {
    Validate.notNull(newImageResource, "新图片资源信息不能为空");
    //修改新资源使用状态为已使用
    newImageResource.setUseImage(true);
    productImageResourceRepository.saveAndFlush(newImageResource);
    if (oldImageResource != null) {
      this.updateUseImage(oldImageResource);
    }
  }

  @Override
  @Transactional
  public void updateUseImage(ProductImageResource productImageResource) {
    Validate.notNull(productImageResource, "商品图片不能为空");
    //查询就资源使用数量 如果大于0 那么修改为已使用
    int fileCount = productFileRepository.countByProductImageResourceId(productImageResource.getId());
    int specificationCount = productSpecificationRepository.countByProductImageResourceId(productImageResource.getId());
    if (fileCount > 0 || specificationCount > 0) {
      productImageResource.setUseImage(true);
    } else {
      productImageResource.setUseImage(false);
    }
    productImageResourceRepository.saveAndFlush(productImageResource);

  }

  @Override
  @Transactional
  public void updateUseImage(ProductImageResource productImageResource, String fileId, String specificationId) {
    Validate.notNull(productImageResource, "商品图片不能为空");
    //查询就资源使用数量 如果大于0 那么修改为已使用
    int fileCount, specificationCount;
    if(StringUtils.isBlank(fileId)){
      fileCount = productFileRepository.countByProductImageResourceId(productImageResource.getId());
    }else{
      fileCount = productFileRepository.countByProductImageResourceIdAndIdNot(productImageResource.getId(),fileId);
    }
    if(StringUtils.isBlank(specificationId)){
      specificationCount = productSpecificationRepository.countByProductImageResourceId(productImageResource.getId());
    }else{
      specificationCount = productSpecificationRepository.countByProductImageResourceIdAndIdNot(productImageResource.getId(),specificationId);
    }
    productImageResource.setUseImage(fileCount > 0 || specificationCount > 0);
    productImageResourceRepository.saveAndFlush(productImageResource);
  }
}
