package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.empower.business.product.entity.ProductMultipleSpecification;
import com.bizunited.empower.business.product.repository.ProductMultipleSpecificationRepository;
import com.bizunited.empower.business.product.service.ProductMultipleSpecificationService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Sets;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * ProductMultipleSpecification业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("ProductMultipleSpecificationServiceImpl")
public class ProductMultipleSpecificationServiceImpl implements ProductMultipleSpecificationService {
  @Autowired
  private ProductMultipleSpecificationRepository productMultipleSpecificationRepository;

  /**
   * 批量新增多规格基本信息
   *
   * @param multipleSpecifications
   * @param product
   */
  @Override
  public void createInsertAbleEntitySet(Set<ProductMultipleSpecification> multipleSpecifications, Product product) {
    Validate.notNull(product, "未传入商品信息！");
    if (CollectionUtils.isNotEmpty(multipleSpecifications)) {
      for (ProductMultipleSpecification multipleSpecification : multipleSpecifications) {
        multipleSpecification.setProduct(product);
        this.createInsertAbleEntity(multipleSpecification);
      }
      productMultipleSpecificationRepository.saveAll(multipleSpecifications);
    }
  }

  @Override
  public void deleteByBatch(Set<ProductMultipleSpecification> productMultipleSpecifications) {
    if (CollectionUtils.isNotEmpty(productMultipleSpecifications)) {
      this.productMultipleSpecificationRepository.deleteAll(productMultipleSpecifications);
    }
  }

  @Override
  public void updateFormBatch(Set<ProductMultipleSpecification> multipleSpecifications, Product product) {
    /* 4.修改商品多规格基本信息
     *    特殊的前置校验
     *    分为有ID的，无ID的
     *    4.1删去不存在了的
     *    4.2为新增信息赋值
     *    4.3新，旧信息一起保存
     */
    this.preCheck(multipleSpecifications,product);
    if (CollectionUtils.isNotEmpty(multipleSpecifications)) {
      Map<Boolean, List<ProductMultipleSpecification>> idBooleanMap = multipleSpecifications.stream()
          .collect(Collectors.partitioningBy(multipleSpecification -> StringUtils.isNotBlank(multipleSpecification.getId())));
      List<ProductMultipleSpecification> idTrueList = idBooleanMap.get(true);
      List<ProductMultipleSpecification> idFalseList = idBooleanMap.get(false);
      // 4.1删去不存在了的
      List<String> ids = idTrueList.stream().map(ProductMultipleSpecification::getId).filter(StringUtils::isNotEmpty).collect(Collectors.toList());
      // 全删
      Set<ProductMultipleSpecification> needDelete;
      if (CollectionUtils.isEmpty(ids)) {
        needDelete = productMultipleSpecificationRepository.findByProduct(TenantUtils.getTenantCode(), product.getProductCode());
      } else {
        // 部分删
        needDelete = productMultipleSpecificationRepository.findByProductAndIdNotIn(TenantUtils.getTenantCode(), product.getProductCode(), ids);
      }
      if (CollectionUtils.isNotEmpty(needDelete)) {
        productMultipleSpecificationRepository.deleteAll(needDelete);
      }
      Set<ProductMultipleSpecification> saveAll = new HashSet<>(multipleSpecifications.size());
      // 4.2为新增信息赋值
      for (ProductMultipleSpecification multipleSpecification : idFalseList) {
        multipleSpecification.setProduct(product);
        this.createInsertAbleEntity(multipleSpecification);
        saveAll.add(multipleSpecification);
      }
      // 4.3 旧信息覆盖
      for (ProductMultipleSpecification multipleSpecification : idTrueList) {
        Optional<ProductMultipleSpecification> op_currentProductMultipleSpecification = this.productMultipleSpecificationRepository.findById(multipleSpecification.getId());
        ProductMultipleSpecification currentProductMultipleSpecification = op_currentProductMultipleSpecification.orElse(null);
        currentProductMultipleSpecification = Validate.notNull(currentProductMultipleSpecification, "未发现指定的原始模型对象信");

        multipleSpecification.setCreateAccount(currentProductMultipleSpecification.getCreateAccount());
        multipleSpecification.setCreateTime(currentProductMultipleSpecification.getCreateTime());
        multipleSpecification.setProduct(product);
        multipleSpecification.setTenantCode(TenantUtils.getTenantCode());
        Date now = new Date();
        multipleSpecification.setModifyAccount(SecurityUtils.getUserAccount());
        multipleSpecification.setModifyTime(now);
        this.updateValidation(multipleSpecification);
        saveAll.add(multipleSpecification);
      }
      productMultipleSpecificationRepository.saveAll(saveAll);
    }else{
      //全删
      Set<ProductMultipleSpecification> needDelete = productMultipleSpecificationRepository.findByProduct(TenantUtils.getTenantCode(), product.getProductCode());
      if (CollectionUtils.isNotEmpty(needDelete)) {
        productMultipleSpecificationRepository.deleteAll(needDelete);
      }
    }
  }

  /**
   * 多规格可以为空
   * @param multipleSpecifications
   * @param product
   */
  private void preCheck(Set<ProductMultipleSpecification> multipleSpecifications,Product product) {
    Validate.notNull(product, "未传入商品信息！");
  }

  private void createInsertAbleEntity(ProductMultipleSpecification multipleSpecification) {
    //
    Date now = new Date();
    multipleSpecification.setCreateAccount(SecurityUtils.getUserAccount());
    multipleSpecification.setCreateTime(now);
    multipleSpecification.setModifyAccount(SecurityUtils.getUserAccount());
    multipleSpecification.setModifyTime(now);
    //
    multipleSpecification.setTenantCode(TenantUtils.getTenantCode());

    this.createValidation(multipleSpecification);
  }

  @Transactional
  @Override
  public ProductMultipleSpecification create(ProductMultipleSpecification productMultipleSpecification) {
    ProductMultipleSpecification current = this.createForm(productMultipleSpecification);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductMultipleSpecification createForm(ProductMultipleSpecification productMultipleSpecification) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    Date now = new Date();
    productMultipleSpecification.setCreateAccount(SecurityUtils.getUserAccount());
    productMultipleSpecification.setCreateTime(now);
    productMultipleSpecification.setModifyAccount(SecurityUtils.getUserAccount());
    productMultipleSpecification.setModifyTime(now);

    this.createValidation(productMultipleSpecification);

    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================

    this.productMultipleSpecificationRepository.save(productMultipleSpecification);

    // 返回最终处理的结果，里面带有详细的关联信息
    return productMultipleSpecification;
  }

  /**
   * 在创建一个新的ProductMultipleSpecification模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProductMultipleSpecification productMultipleSpecification) {
    Validate.notNull(productMultipleSpecification, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(productMultipleSpecification.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    productMultipleSpecification.setId(null);
    Validate.notBlank(productMultipleSpecification.getSpecKey(), "添加信息时，规格名称不能为空！");
    Validate.notBlank(productMultipleSpecification.getSpecValue(), "添加信息时，规格值不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(productMultipleSpecification.getSpecKey() == null || productMultipleSpecification.getSpecKey().length() < 255, "规格键,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productMultipleSpecification.getSpecValue() == null || productMultipleSpecification.getSpecValue().length() < 255, "规格值,在进行添加时填入值超过了限定长度(255)，请检查!");
  }

  @Transactional
  @Override
  public ProductMultipleSpecification update(ProductMultipleSpecification productMultipleSpecification) {
    ProductMultipleSpecification current = this.updateForm(productMultipleSpecification);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductMultipleSpecification updateForm(ProductMultipleSpecification productMultipleSpecification) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(productMultipleSpecification);
    // ===================基本信息
    String currentId = productMultipleSpecification.getId();
    Optional<ProductMultipleSpecification> op_currentProductMultipleSpecification = this.productMultipleSpecificationRepository.findById(currentId);
    ProductMultipleSpecification currentProductMultipleSpecification = op_currentProductMultipleSpecification.orElse(null);
    currentProductMultipleSpecification = Validate.notNull(currentProductMultipleSpecification, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentProductMultipleSpecification.setModifyAccount(SecurityUtils.getUserAccount());
    currentProductMultipleSpecification.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentProductMultipleSpecification.setSpecKey(productMultipleSpecification.getSpecKey());
    currentProductMultipleSpecification.setSpecValue(productMultipleSpecification.getSpecValue());
    currentProductMultipleSpecification.setProduct(productMultipleSpecification.getProduct());

    this.productMultipleSpecificationRepository.saveAndFlush(currentProductMultipleSpecification);
    return currentProductMultipleSpecification;
  }

  /**
   * 在更新一个已有的ProductMultipleSpecification模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ProductMultipleSpecification productMultipleSpecification) {
    Validate.isTrue(!StringUtils.isBlank(productMultipleSpecification.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");

    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(productMultipleSpecification.getSpecKey(), "修改信息时，规格名称不能为空！");
    Validate.notBlank(productMultipleSpecification.getSpecValue(), "修改信息时，规格值不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(productMultipleSpecification.getSpecKey() == null || productMultipleSpecification.getSpecKey().length() < 255, "规格键,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productMultipleSpecification.getSpecValue() == null || productMultipleSpecification.getSpecValue().length() < 255, "规格值,在进行修改时填入值超过了限定长度(255)，请检查!");

    // 关联性判断，关联属性判断，需要满足ManyToOne或者OneToOne，且not null 且是主模型
  }

  @Override
  public Set<ProductMultipleSpecification> findDetailsByProduct(String product) {
    if (StringUtils.isBlank(product)) {
      return Sets.newHashSet();
    }
    return this.productMultipleSpecificationRepository.findDetailsByProduct(product,TenantUtils.getTenantCode());
  }

  @Override
  public ProductMultipleSpecification findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.productMultipleSpecificationRepository.findDetailsById(id);
  }

  @Override
  public ProductMultipleSpecification findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<ProductMultipleSpecification> op = productMultipleSpecificationRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    ProductMultipleSpecification current = this.findById(id);
    if (current != null) {
      this.productMultipleSpecificationRepository.delete(current);
    }
  }
} 
