package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.empower.business.product.entity.ProductPricing;
import com.bizunited.empower.business.product.entity.ProductPricingUnitSpecification;
import com.bizunited.empower.business.product.entity.ProductSpecification;
import com.bizunited.empower.business.product.repository.ProductPricingRepository;
import com.bizunited.empower.business.product.service.ProductPricingService;
import com.bizunited.empower.business.product.service.ProductPricingUnitSpecificationService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Sets;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * ProductPricing业务模型的服务层接口实现
 * @author saturn
 */
@Service("ProductPricingServiceImpl")
public class ProductPricingServiceImpl implements ProductPricingService {
  @Autowired
  private ProductPricingRepository productPricingRepository;

  @Autowired
  private ProductPricingUnitSpecificationService productPricingUnitSpecificationService;

  /**
   * 批量新增商品定价
   * @param productPricings
   * @param product
   */
  @Transactional
  @Override
  public void createInsertAbleEntitySet(Set<ProductPricing> productPricings, Product product, ProductSpecification productSpecification) {
    Validate.notNull(product, "未传入商品信息！");
    if (CollectionUtils.isNotEmpty(productPricings)) {
      for (ProductPricing productPricing : productPricings) {
        productPricing.setProduct(product);
        productPricing.setProductSpecification(productSpecification);
        this.createInsertAbleEntity(productPricing);
      }
      productPricingRepository.saveAll(productPricings);

      for (ProductPricing productPricing : productPricings) {
        Set<ProductPricingUnitSpecification> productPricingUnits = productPricing.getProductPricingUnitSpecifications();
        productPricingUnitSpecificationService.createInsertAbleEntitySet(productPricingUnits, productPricing,productSpecification);
      }
    }
  }

  @Transactional
  @Override
  public void deleteByBatch(Set<ProductPricing> productPricings) {
    if (CollectionUtils.isNotEmpty(productPricings)) {
      for (ProductPricing productPricing : productPricings) {
        Set<ProductPricingUnitSpecification> productPricingUnits = productPricing.getProductPricingUnitSpecifications();
        if (CollectionUtils.isNotEmpty(productPricingUnits)){
          this.productPricingUnitSpecificationService.deleteByBatch(productPricingUnits);
        }
      }
      this.productPricingRepository.deleteAll(productPricings);
    }
  }

  @Transactional
  @Override
  public void updateFormBatch(Set<ProductPricing> productPricings, Product product, ProductSpecification productSpecification) {
    /*3.修改商品定价
          分为有ID，无ID
     *    3.1删去不存在了的
     *    3.2为新增信息赋值
     *    3.3新，旧信息一起保存
     */
    this.preCheck(productPricings,product,productSpecification);
    if (CollectionUtils.isNotEmpty(productPricings)){
      Map<Boolean, List<ProductPricing>> idBooleanMap = productPricings.stream()
          .collect(Collectors.partitioningBy(productPricing -> StringUtils.isNotBlank(productPricing.getId())));
      List<ProductPricing> idTrueList = idBooleanMap.get(true);
      List<ProductPricing> idFalseList = idBooleanMap.get(false);
      //3.1删去不存在了的
      List<String> ids = idTrueList.stream().map(ProductPricing::getId).filter(StringUtils::isNotEmpty).collect(Collectors.toList());
      Set<ProductPricing> needDelete ;
      if(CollectionUtils.isEmpty(ids)){
        needDelete = productPricingRepository.findByProductAndProductSpecification(TenantUtils.getTenantCode(), product.getProductCode(),productSpecification.getProductSpecificationCode());
      }else {
        needDelete = productPricingRepository.findByProductAndIdNotInAndProductSpecification(TenantUtils.getTenantCode(), product.getProductCode(), ids,productSpecification.getProductSpecificationCode());
      }
      if (CollectionUtils.isNotEmpty(needDelete)) {
        this.deleteByBatch(needDelete);
      }
      // 新增
      this.createInsertAbleEntitySet(new HashSet<>(idFalseList),product,productSpecification);
      // 旧信息覆盖
      Set<ProductPricing> saveAll = new HashSet<>(productPricings.size());
      for (ProductPricing productPricing : idTrueList) {
        Optional<ProductPricing> op_currentProductPricing = productPricingRepository.findById(productPricing.getId());
        ProductPricing currentProductPricing = op_currentProductPricing.orElse(null);
        currentProductPricing = Validate.notNull(currentProductPricing ,"未发现指定的原始模型对象信");

        productPricing.setCreateTime(currentProductPricing.getCreateTime());
        productPricing.setCreateAccount(currentProductPricing.getCreateAccount());
        productPricing.setProduct(product);
        productPricing.setProductSpecification(productSpecification);
        productPricing.setTenantCode(TenantUtils.getTenantCode());
        productPricing.setModifyAccount(SecurityUtils.getUserAccount());
        productPricing.setModifyTime(new Date());
        Set<ProductPricingUnitSpecification> productPricingUnits = productPricing.getProductPricingUnitSpecifications();
        productPricingUnitSpecificationService.updateFormBatch(productPricingUnits,productPricing,productSpecification);
        this.updateValidation(productPricing);
        saveAll.add(productPricing);
      }
      productPricingRepository.saveAll(saveAll);
    }else {
      //全删
      Set<ProductPricing> needDelete = productPricingRepository.findByProductAndProductSpecification(TenantUtils.getTenantCode(), product.getProductCode(),productSpecification.getProductSpecificationCode());
      if (CollectionUtils.isNotEmpty(needDelete)) {
        this.deleteByBatch(needDelete);
      }
    }
  }

  private void preCheck(Set<ProductPricing> productPricings, Product product,ProductSpecification productSpecification) {
    Validate.notNull(product, "未传入商品信息！");
    Validate.notNull(productSpecification, "未传入规格信息！");
  }

  private void createInsertAbleEntity(ProductPricing productPricing) {
    //
    Date now = new Date();
    productPricing.setCreateAccount(SecurityUtils.getUserAccount());
    productPricing.setCreateTime(now);
    productPricing.setModifyAccount(SecurityUtils.getUserAccount());
    productPricing.setModifyTime(now);
    //
    productPricing.setTenantCode(TenantUtils.getTenantCode());

    this.createValidation(productPricing);
  }

  @Transactional
  @Override
  public ProductPricing create(ProductPricing productPricing) {
    ProductPricing current = this.createForm(productPricing);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }
  @Transactional
  @Override
  public ProductPricing createForm(ProductPricing productPricing) {

    this.createInsertAbleEntity(productPricing);

    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================

    this.productPricingRepository.save(productPricing);

    // 返回最终处理的结果，里面带有详细的关联信息
    return productPricing;
  }
  /**
   * 在创建一个新的ProductPricing模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProductPricing productPricing) {
    Validate.notNull(productPricing , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(productPricing.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    productPricing.setId(null);
    Validate.notBlank(productPricing.getCustomerLevel(), "添加信息时，客户级别不能为空！");
    Validate.notBlank(productPricing.getCustomerName(), "添加信息时，客户名称不能为空！");
    Validate.notBlank(productPricing.getLevelName(), "添加信息时，客户级别名称不能为空！");
    Validate.notBlank(productPricing.getCustomerCode(), "添加信息时，客户编码不能为空！");
    Validate.notNull(productPricing.getIsCustomerCode(), "添加信息时，是客户编码不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(productPricing.getCustomerLevel() == null || productPricing.getCustomerLevel().length() < 255 , "客户级别,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productPricing.getCustomerCode() == null || productPricing.getCustomerCode().length() < 255 , "客户编码,在进行添加时填入值超过了限定长度(255)，请检查!");
  }
  @Transactional
  @Override
  public ProductPricing update(ProductPricing productPricing) {
    ProductPricing current = this.updateForm(productPricing);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }
  @Transactional
  @Override
  public ProductPricing updateForm(ProductPricing productPricing) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(productPricing);
    // ===================基本信息
    String currentId = productPricing.getId();
    Optional<ProductPricing> op_currentProductPricing = this.productPricingRepository.findById(currentId);
    ProductPricing currentProductPricing = op_currentProductPricing.orElse(null);
    currentProductPricing = Validate.notNull(currentProductPricing ,"未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentProductPricing.setModifyAccount(SecurityUtils.getUserAccount());
    currentProductPricing.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentProductPricing.setCustomerLevel(productPricing.getCustomerLevel());
    currentProductPricing.setCustomerCode(productPricing.getCustomerCode());
    currentProductPricing.setIsCustomerCode(productPricing.getIsCustomerCode());
    currentProductPricing.setProduct(productPricing.getProduct());

    this.productPricingRepository.saveAndFlush(currentProductPricing);
    return currentProductPricing;
  }
  /**
   * 在更新一个已有的ProductPricing模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ProductPricing productPricing) {
    Validate.isTrue(!StringUtils.isBlank(productPricing.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");

    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(productPricing.getCustomerLevel(), "修改信息时，客户级别不能为空！");
    Validate.notBlank(productPricing.getLevelName(), "修改信息时，客户级别名称不能为空！");
    Validate.notBlank(productPricing.getCustomerCode(), "修改信息时，客户编码不能为空！");
    Validate.notBlank(productPricing.getCustomerName(), "修改信息时，客户名称不能为空！");
    Validate.notNull(productPricing.getIsCustomerCode(), "修改信息时，是客户编码不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(productPricing.getCustomerLevel() == null || productPricing.getCustomerLevel().length() < 255 , "客户级别,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productPricing.getCustomerCode() == null || productPricing.getCustomerCode().length() < 255 , "客户编码,在进行修改时填入值超过了限定长度(255)，请检查!");

    // 关联性判断，关联属性判断，需要满足ManyToOne或者OneToOne，且not null 且是主模型
  }
  @Override
  public Set<ProductPricing> findDetailsByProduct(String product) {
    if(StringUtils.isBlank(product)) {
      return Sets.newHashSet();
    }
    return this.productPricingRepository.findDetailsByProduct(product,TenantUtils.getTenantCode());
  }
  @Override
  public ProductPricing findDetailsById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    return this.productPricingRepository.findDetailsById(id);
  }
  @Override
  public ProductPricing findById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }

    Optional<ProductPricing> op = productPricingRepository.findById(id);
    return op.orElse(null);
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    ProductPricing current = this.findById(id);
    if(current != null) {
      this.productPricingRepository.delete(current);
    }
  }

  @Override
  public List<ProductPricing> findDetailsByCustomerCode(String customerCode) {
    if(StringUtils.isBlank(customerCode)){
      return Lists.newArrayList();
    }
    return productPricingRepository.findByCustomerCodeAndTenantCode(customerCode,TenantUtils.getTenantCode());
  }
}
