package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.product.entity.ProductPricing;
import com.bizunited.empower.business.product.entity.ProductPricingUnit;
import com.bizunited.empower.business.product.repository.ProductPricingUnitRepository;
import com.bizunited.empower.business.product.service.ProductPricingUnitService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

@Service("ProductPricingUnitServiceImpl")
public class ProductPricingUnitServiceImpl implements ProductPricingUnitService {

  @Autowired
  ProductPricingUnitRepository productPricingUnitRepository;

  @Transactional
  @Override
  public void createInsertAbleEntitySet(Set<ProductPricingUnit> productPricingUnits, ProductPricing productPricing) {
    Validate.notNull(productPricing, "未传入客户定价信息！");
    if (CollectionUtils.isNotEmpty(productPricingUnits)) {
      for (ProductPricingUnit productPricingUnit : productPricingUnits) {
        productPricingUnit.setProductPricing(productPricing);
        this.createInsertAbleEntity(productPricingUnit);
      }
      productPricingUnitRepository.saveAll(productPricingUnits);
    }
  }

  @Transactional
  @Override
  public void deleteByBatch(Set<ProductPricingUnit> productPricingUnits) {
    if (CollectionUtils.isNotEmpty(productPricingUnits)) {
      this.productPricingUnitRepository.deleteAll(productPricingUnits);
    }
  }

  @Transactional
  @Override
  public Set<ProductPricingUnit> updateFormBatch(Set<ProductPricingUnit> productPricingUnits, ProductPricing productPricing) {
    // 特殊的前置校验
    this.preCheck(productPricingUnits,productPricing);
    // 分为有ID的，无ID的
    Map<Boolean, List<ProductPricingUnit>> idBooleanMap = productPricingUnits.stream()
        .collect(Collectors.partitioningBy(productPricingUnit -> StringUtils.isNotBlank(productPricingUnit.getId())));
    List<ProductPricingUnit> idTrueList = idBooleanMap.get(true);
    List<ProductPricingUnit> idFalseList = idBooleanMap.get(false);
    // 5.1删去不存在了的
    List<String> ids = idTrueList.stream().map(ProductPricingUnit::getId).filter(StringUtils::isNotEmpty).collect(Collectors.toList());
    Set<ProductPricingUnit> needDelete ;
    if (CollectionUtils.isEmpty(ids)){
      needDelete = productPricingUnitRepository.findByProductPricing(TenantUtils.getTenantCode(), productPricing.getId());
    } else {
      needDelete = productPricingUnitRepository.findByProductPricingAndIdNotIn(TenantUtils.getTenantCode(), productPricing.getId(), ids);
    }
    if (CollectionUtils.isNotEmpty(needDelete)) {
      productPricingUnitRepository.deleteAll(needDelete);
    }
    // 5.2为新增信息赋值
    Set<ProductPricingUnit> saveAll = new HashSet<>(productPricingUnits.size());
    for (ProductPricingUnit productPricingUnit : idFalseList) {
      productPricingUnit.setProductPricing(productPricing);
      this.createInsertAbleEntity(productPricingUnit);
      saveAll.add(productPricingUnit);
    }
    // 5.3 旧信息覆盖
    for (ProductPricingUnit productPricingUnit : idTrueList) {
      Optional<ProductPricingUnit> op_currentProductPricingUnit = this.productPricingUnitRepository.findById(productPricingUnit.getId());
      ProductPricingUnit currentProductPricingUnit = op_currentProductPricingUnit.orElse(null);
      Validate.notNull(currentProductPricingUnit ,"未发现指定的原始模型对象信");

      productPricingUnit.setProductPricing(productPricing);
      productPricingUnit.setTenantCode(TenantUtils.getTenantCode());
      this.updateValidation(productPricingUnit);
      saveAll.add(productPricingUnit);
    }
    productPricingUnitRepository.saveAll(saveAll);
    return saveAll;
  }

  private void updateValidation(ProductPricingUnit productPricingUnit) {
    Validate.isTrue(!StringUtils.isBlank(productPricingUnit.getId()), "修改信息时，当前信息的数据编号（主键）必须有值！");
    Validate.notBlank(productPricingUnit.getUnitCode(), "修改信息时，单位编号不能为空！");
    Validate.notBlank(productPricingUnit.getUnitName(), "修改信息时，单位名称不能为空！");
    Validate.notNull(productPricingUnit.getOrderPrice(), "修改信息时，订货价格不能为空！");
    Validate.isTrue(productPricingUnit.getUnitCode().length() < 64, "单位编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productPricingUnit.getUnitName().length() < 128, "单位名称,在进行添加时填入值超过了限定长度(128)，请检查!");
  }

  private void preCheck(Set<ProductPricingUnit> productPricingUnits, ProductPricing productPricing) {
    Validate.notNull(productPricing, "定价客户信息不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(productPricingUnits),"单位价格信息不能为空");
  }

  private void createInsertAbleEntity(ProductPricingUnit productPricingUnit) {
    productPricingUnit.setTenantCode(TenantUtils.getTenantCode());
    this.createValidation(productPricingUnit);
  }

  private void createValidation(ProductPricingUnit productPricingUnit) {
    Validate.notNull(productPricingUnit , "进行当前操作时，信息对象必须传入!!");
    Validate.isTrue(StringUtils.isBlank(productPricingUnit.getId()), "添加信息时，当前信息的数据编号（主键）不能有值！");
    productPricingUnit.setId(null);
    Validate.notBlank(productPricingUnit.getUnitCode(), "添加信息时，单位编号不能为空！");
    Validate.notBlank(productPricingUnit.getUnitName(), "添加信息时，单位名称不能为空！");
    Validate.notNull(productPricingUnit.getOrderPrice(), "添加信息时，订货价格不能为空！");
    Validate.isTrue(productPricingUnit.getUnitCode().length() < 64, "单位编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productPricingUnit.getUnitName().length() < 128, "单位名称,在进行添加时填入值超过了限定长度(128)，请检查!");
  }
}
