package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.product.entity.ProductPricing;
import com.bizunited.empower.business.product.entity.ProductPricingUnitSpecification;
import com.bizunited.empower.business.product.entity.ProductSpecification;
import com.bizunited.empower.business.product.repository.ProductPricingUnitSpecificationRepository;
import com.bizunited.empower.business.product.service.ProductPriceModifyLogService;
import com.bizunited.empower.business.product.service.ProductPricingUnitSpecificationService;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Sets;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

@Service("ProductPricingUnitSpecificationServiceImpl")
public class ProductPricingUnitSpecificationServiceImpl implements ProductPricingUnitSpecificationService {

  @Autowired
  ProductPricingUnitSpecificationRepository productPricingUnitSpecificationRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private ProductPriceModifyLogService productPriceModifyLogService;

  @Transactional
  @Override
  public void createInsertAbleEntitySet(Set<ProductPricingUnitSpecification> productPricingUnitSpecifications, ProductPricing productPricing, ProductSpecification productSpecification) {
    Validate.notNull(productPricing, "未传入客户定价信息！");
    Validate.notNull(productSpecification, "未传入产品规格信息！");
    Validate.isTrue(CollectionUtils.isNotEmpty(productPricingUnitSpecifications), "未传入规格定价信息!");

    for (ProductPricingUnitSpecification productPricingUnitSpecification : productPricingUnitSpecifications) {
      //绑定规格信息
      productPricingUnitSpecification.setProductPricing(productPricing);
      productPricingUnitSpecification.setProductSpecificationCode(productSpecification.getProductSpecificationCode());
      productPricingUnitSpecification.setProductSpecificationName(productSpecification.getProductSpecificationName());
      this.createInsertAbleEntity(productPricingUnitSpecification);
    }
    productPricingUnitSpecificationRepository.saveAll(productPricingUnitSpecifications);

  }

  @Transactional
  @Override
  public void deleteByBatch(Set<ProductPricingUnitSpecification> productPricingUnitSpecifications) {
    if (CollectionUtils.isNotEmpty(productPricingUnitSpecifications)) {
      this.productPricingUnitSpecificationRepository.deleteAll(productPricingUnitSpecifications);
    }
  }

  private void basicsUpdate(ProductPricingUnitSpecification dbProductPricingUnitSpecification,ProductPricingUnitSpecification productPricingUnitSpecification){
    dbProductPricingUnitSpecification.setProductSpecificationName(productPricingUnitSpecification.getProductSpecificationName());
    dbProductPricingUnitSpecification.setProductSpecificationCode(productPricingUnitSpecification.getProductSpecificationCode());
    dbProductPricingUnitSpecification.setUnitCode(productPricingUnitSpecification.getUnitCode());
    dbProductPricingUnitSpecification.setUnitName(productPricingUnitSpecification.getUnitName());
    dbProductPricingUnitSpecification.setOrderPrice(productPricingUnitSpecification.getOrderPrice());
  }


  @Transactional
  @Override
  public Set<ProductPricingUnitSpecification> updateFormBatch(Set<ProductPricingUnitSpecification> productPricingUnitSpecifications, ProductPricing productPricing, ProductSpecification productSpecification) {
    // 特殊的前置校验
    this.preCheck(productPricingUnitSpecifications, productPricing);
    Validate.notNull(productPricing,"客户定价不能为空");
    Validate.notNull(productSpecification,"规格不能为空");
    productPricingUnitSpecifications = ObjectUtils.defaultIfNull(productPricingUnitSpecifications, Sets.newHashSet());
    Set<ProductPricingUnitSpecification> deletes = Sets.newHashSet();
    Set<ProductPricingUnitSpecification> updates = Sets.newHashSet();
    Set<ProductPricingUnitSpecification> creates = Sets.newHashSet();
    Map<String,ProductPricingUnitSpecification> productPricingUnitSpecificationMap = productPricingUnitSpecifications.stream().collect(Collectors.toMap(ProductPricingUnitSpecification::getId, v->v,(v1, v2)->v2));
    List<ProductPricingUnitSpecification> dbProductPricingUnitSpecifications = productPricingUnitSpecificationRepository.findByProductPricingAndProductSpecificationCode(TenantUtils.getTenantCode(),productPricing.getId(),productSpecification.getProductSpecificationCode());
    nebulaToolkitService.collectionDiscrepancy(productPricingUnitSpecifications,dbProductPricingUnitSpecifications,ProductPricingUnitSpecification::getId,deletes,updates,creates);
    if(!CollectionUtils.isEmpty(deletes)){
      productPricingUnitSpecificationRepository.deleteInBatch(deletes);
    }
    //更新数据
    for (ProductPricingUnitSpecification update : updates) {
      ProductPricingUnitSpecification productPricingUnitSpecification = productPricingUnitSpecificationMap.get(update.getId());
      BigDecimal oldOrderPrice = update.getOrderPrice();
      this.basicsUpdate(update,productPricingUnitSpecification);
      this.updateValidation(update);
      productPricingUnitSpecificationRepository.saveAndFlush(update);
      //保存日志信息
      productPriceModifyLogService.save(update,oldOrderPrice);
    }
    //新增数据
    for (ProductPricingUnitSpecification create : creates) {
      create.setProductPricing(productPricing);
      create.setProductSpecificationCode(productSpecification.getProductSpecificationCode());
      create.setProductSpecificationName(productSpecification.getProductSpecificationName());
      this.createInsertAbleEntity(create);
      productPricingUnitSpecificationRepository.saveAndFlush(create);
    }

    return creates;
  }

  private void updateValidation(ProductPricingUnitSpecification productPricingUnitSpecification) {
    Validate.isTrue(!StringUtils.isBlank(productPricingUnitSpecification.getId()), "修改信息时，当前信息的数据编号（主键）必须有值！");
    Validate.notBlank(productPricingUnitSpecification.getUnitCode(), "修改信息时，单位编号不能为空！");
    Validate.notBlank(productPricingUnitSpecification.getUnitName(), "修改信息时，单位名称不能为空！");
    Validate.notNull(productPricingUnitSpecification.getOrderPrice(), "修改信息时，订货价格不能为空！");
    Validate.isTrue(productPricingUnitSpecification.getUnitCode().length() < 64, "单位编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productPricingUnitSpecification.getUnitName().length() < 128, "单位名称,在进行添加时填入值超过了限定长度(128)，请检查!");
  }

  private void preCheck(Set<ProductPricingUnitSpecification> productPricingUnitSpecifications, ProductPricing productPricing) {
    Validate.notNull(productPricing, "定价客户信息不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(productPricingUnitSpecifications), "单位价格信息不能为空");
  }

  private void createInsertAbleEntity(ProductPricingUnitSpecification productPricingUnitSpecification) {
    productPricingUnitSpecification.setTenantCode(TenantUtils.getTenantCode());
    this.createValidation(productPricingUnitSpecification);
  }

  private void createValidation(ProductPricingUnitSpecification productPricingUnitSpecification) {
    Validate.notNull(productPricingUnitSpecification, "进行当前操作时，信息对象必须传入!!");
    Validate.isTrue(StringUtils.isBlank(productPricingUnitSpecification.getId()), "添加信息时，当前信息的数据编号（主键）不能有值！");
    productPricingUnitSpecification.setId(null);
    Validate.notBlank(productPricingUnitSpecification.getUnitCode(), "添加信息时，单位编号不能为空！");
    Validate.notBlank(productPricingUnitSpecification.getUnitName(), "添加信息时，单位名称不能为空！");
    Validate.notNull(productPricingUnitSpecification.getOrderPrice(), "添加信息时，订货价格不能为空！");
    Validate.isTrue(productPricingUnitSpecification.getUnitCode().length() < 64, "单位编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productPricingUnitSpecification.getUnitName().length() < 128, "单位名称,在进行添加时填入值超过了限定长度(128)，请检查!");
  }

  @Override
  public Set<ProductPricingUnitSpecification> findByProductSpecificationCode(String productSpecificationCode) {
    if (StringUtils.isBlank(productSpecificationCode)) {
      return Sets.newHashSet();
    }
    return this.productPricingUnitSpecificationRepository.findByProductSpecificationCodeAndTenantCode(productSpecificationCode, TenantUtils.getTenantCode());
  }

  @Override
  public Set<ProductPricingUnitSpecification> findByProductSpecificationCodes(List<String> productSpecificationCodes) {
    if (CollectionUtils.isEmpty(productSpecificationCodes)) {
      return Sets.newHashSet();
    }
    return this.productPricingUnitSpecificationRepository.findByProductSpecificationCodeInAndTenantCode(productSpecificationCodes, TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public void saveBatch(Set<ProductPricingUnitSpecification> productPricingUnitSpecifications) {
    Validate.notEmpty(productPricingUnitSpecifications,"保存信息不能为空");
    productPricingUnitSpecificationRepository.saveAll(productPricingUnitSpecifications);
  }
}
