package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.common.util.SecurityUtils;
import com.bizunited.empower.business.product.common.enumerate.ProductImageStatus;
import com.bizunited.empower.business.product.common.enumerate.ShelfStatus;
import com.bizunited.empower.business.product.common.enumerate.SpecificationImageStatus;
import com.bizunited.empower.business.product.dto.ProductDto;
import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.empower.business.product.entity.ProductBrand;
import com.bizunited.empower.business.product.entity.ProductCategory;
import com.bizunited.empower.business.product.entity.ProductDescription;
import com.bizunited.empower.business.product.entity.ProductFile;
import com.bizunited.empower.business.product.entity.ProductMultipleSpecification;
import com.bizunited.empower.business.product.entity.ProductPricing;
import com.bizunited.empower.business.product.entity.ProductShowCategory;
import com.bizunited.empower.business.product.entity.ProductSpecification;
import com.bizunited.empower.business.product.entity.ProductTag;
import com.bizunited.empower.business.product.entity.ProductUnitAndPrice;
import com.bizunited.empower.business.product.entity.ProductUnitSpecificationAndPrice;
import com.bizunited.empower.business.product.repository.ProductRepository;
import com.bizunited.empower.business.product.service.ProductBrandService;
import com.bizunited.empower.business.product.service.ProductCategoryService;
import com.bizunited.empower.business.product.service.ProductDescriptionService;
import com.bizunited.empower.business.product.service.ProductFileService;
import com.bizunited.empower.business.product.service.ProductMultipleSpecificationService;
import com.bizunited.empower.business.product.service.ProductPricingService;
import com.bizunited.empower.business.product.service.ProductService;
import com.bizunited.empower.business.product.service.ProductShowCategoryService;
import com.bizunited.empower.business.product.service.ProductSpecificationService;
import com.bizunited.empower.business.product.service.ProductTagService;
import com.bizunited.empower.business.product.service.ProductUnitAndPriceService;
import com.bizunited.empower.business.product.service.ProductUnitSpecificationAndPriceService;
import com.bizunited.empower.business.product.service.notifier.ProductEventListener;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import static javax.transaction.Transactional.TxType.REQUIRES_NEW;

/**
 * Product业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("ProductServiceImpl")
public class ProductServiceImpl implements ProductService {
  @Autowired
  private ProductRepository productRepository;
  @Autowired
  private ProductUnitSpecificationAndPriceService productUnitSpecificationAndPriceService;
  @Autowired
  private ProductBrandService productBrandService;
  @Autowired
  private ProductTagService productTagService;
  @Autowired
  private ProductCategoryService productCategoryService;
  @Autowired
  private ProductPricingService productPricingService;
  @Autowired
  ProductMultipleSpecificationService productMultipleSpecificationService;
  @Autowired
  ProductSpecificationService productSpecificationService;
  @Autowired
  ProductFileService productFileService;
  @Autowired
  ProductDescriptionService productDescriptionService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private ProductShowCategoryService productShowCategoryService;
  @Autowired
  private ProductUnitAndPriceService productUnitAndPriceService;

  @Autowired
  private RedisMutexService redisMutexService;

  @Autowired(required = false)
  private List<ProductEventListener> productEventListeners;

  /**
   * 商品品牌编号前缀
   */
  private static final String PRODUCT_CODE_PREFIX = "SP";
  /**
   * 商品品牌编号生成所使用的redis分布式锁的前缀
   */
  private static final String PRODUCT_REDIS_LOCK_CODE = "SP_PRODUCT_";


  @Transactional
  @Override
  public Product create(ProductDto productDto) {
    Product current = this.createForm(productDto);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  /**
   * 新增商品
   *
   * @param productDto
   * @return
   */
  @Transactional(REQUIRES_NEW)
  @Override
  public Product createForm(ProductDto productDto) {
    /**
     * 1.新增商品
     * 2.新增商品单位与价格
     * 3.新增商品定价
     * 4.新增商品多规格基本信息
     * 5.新增商品规格信息
     * 6.新增商品文件
     * 7.新增商品描述
     */
    Product product = productDto.getProduct();
    Validate.notNull(product, "未传入商品信息！");
    // 商品基本信息
    this.createInsertAbleEntity(product);
    this.productRepository.save(product);
    // ===============================

    // 单位与价格
    Set<ProductUnitAndPrice> productUnitAndPrices = product.getProductUnitAndPrices();
    productUnitAndPriceService.createInsertAbleEntitySet(productUnitAndPrices, product);
    // 商品规格
    //多规格的键值
    //更改层级  先保存商品规格 ，在规格中 保存商品单位价格和规格客户定价
    Set<ProductMultipleSpecification> multipleSpecifications = product.getProductMultipleSpecifications();
    productMultipleSpecificationService.createInsertAbleEntitySet(multipleSpecifications, product);
    //规格
    Set<ProductSpecification> specifications = product.getProductSpecifications();
    productSpecificationService.createInsertAbleEntitySet(specifications, product);
    // 商品图片
    Set<ProductFile> files = product.getProductFiles();
    productFileService.createInsertAbleEntitySet(files, product);
    // 商品描述
    ProductDescription description = productDto.getProductDescription();
    productDescriptionService.createInsertAbleEntity(description, product);
    // ===============================
    // 返回最终处理的结果，里面带有详细的关联信息

    return product;
  }

  private void createInsertAbleEntity(Product product) {
    //---------时间
    Date now = new Date();
    product.setCreateAccount(SecurityUtils.getUserAccount());
    product.setCreateTime(now);
    product.setModifyAccount(SecurityUtils.getUserAccount());
    product.setModifyTime(now);
    //--------系统自构数据
    String tenantCode = TenantUtils.getTenantCode();
    product.setTenantCode(tenantCode);
    String productCode = this.generateCode(tenantCode);
    product.setProductCode(productCode);
    Product productUQ = productRepository.findByProductCode(productCode, tenantCode);
    Validate.isTrue(Objects.isNull(productUQ), "商品编号重复生成，请重新点击提交，无需更改页面");
    //--------------------
    //由于入参只传了ID
    String brandId = product.getProductBrand().getId();
    Validate.notBlank(brandId, "品牌信息不能为空");
    ProductBrand brand = productBrandService.findById(brandId);
    Validate.notNull(brand, "品牌信息不存在");
    product.setProductBrand(brand);
    String categoryId = product.getProductCategory().getId();
    Validate.notBlank(categoryId, "分类信息不能为空");
    ProductCategory category = productCategoryService.findById(categoryId);
    Validate.notNull(category, "分类信息不存在");
    List<ProductCategory> byPCode = productCategoryService.findByPCode(category.getCode());
    Validate.isTrue(CollectionUtils.isEmpty(byPCode), "请选择商品分类最末级分类");
    product.setProductCategory(category);

    String showCategoryId = product.getProductShowCategory().getId();
    Validate.notBlank(showCategoryId, "前端展示分类信息不能为空");
    ProductShowCategory showCategory = productShowCategoryService.findById(showCategoryId);
    Validate.notNull(showCategory, "前端展示分类信息不存在");
    List<ProductShowCategory> showCategories = productShowCategoryService.findByPCode(showCategory.getCode());
    Validate.isTrue(CollectionUtils.isEmpty(showCategories), "请选择商品前端展示分类最末级分类");
    product.setProductShowCategory(showCategory);

    //新增商品状态设置， 新增商品 图片是必传 所以直接设置为已配置
    product.setProductImageStatus(ProductImageStatus.COMPLETE.getStatus());
    product.setSpecificationImageStatus(SpecificationImageStatus.COMPLETE.getStatus());

    Set<ProductTag> tagsIn = product.getTags();
    if (!CollectionUtils.isEmpty(tagsIn)) {
      Set<ProductTag> tags = new HashSet<>(tagsIn.size());
      for (ProductTag tag : tagsIn) {
        ProductTag byId = productTagService.findById(tag.getId());
        tags.add(byId);
      }
      product.setTags(tags);
    }
    //---------商品基本信息校验
    this.createValidation(product);
  }

  private String generateCode(String tenantCode) {
    String productBrandCode = null;
    String lockCode = StringUtils.join(PRODUCT_REDIS_LOCK_CODE, tenantCode);
    try {
      String atomicNumber = this.redisMutexService.getAndIncrement(lockCode, 1, 6);
      productBrandCode = StringUtils.join(PRODUCT_CODE_PREFIX, atomicNumber);
    } catch (Exception e) {
      throw new RuntimeException("生成商品编码失败", e);
    }
    return productBrandCode;
  }

  /**
   * 在创建一个新的Product模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(Product product) {
    Validate.notNull(product, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(product.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    product.setId(null);
    Validate.notBlank(product.getProductCode(), "添加信息时，商品编码不能为空！");
    Validate.notBlank(product.getProductName(), "添加信息时，商品名称不能为空！");
    Validate.notNull(product.getShelfStatus(), "添加信息时，上架状态不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(product.getProductCode() == null || product.getProductCode().length() < 64, "商品编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(product.getProductName() == null || product.getProductName().length() < 128, "商品名称,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(product.getDefaultWarehouseCode() == null || product.getDefaultWarehouseCode().length() < 255, "默认仓库,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(product.getSearchKeyword() == null || product.getSearchKeyword().length() < 255, "搜索关键字,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(product.getRemark() == null || product.getRemark().length() < 255, "备注信息,在进行添加时填入值超过了限定长度(255)，请检查!");
    Product currentProduct = this.findByProductCode(product.getProductCode());
    Validate.isTrue(currentProduct == null, "商品编码已存在,请检查");
  }

  @Transactional
  @Override
  public Product update(ProductDto productDto) {
    //==================================================== 
    /**
     * 1.修改商品
     * 2.修改商品单位与价格
     *    2.1删去不存在了的
     *    2.2为新增信息赋值
     *    2.3新，旧信息一起保存
     * 3.修改商品定价
     *    3.1删去不存在了的
     *    3.2为新增信息赋值
     *    3.3新，旧信息一起保存
     * 4.修改商品多规格基本信息
     *    4.1 4.2 4.3
     * 5.修改商品规格信息
     *    5.1 规格删除
     *      5.1.1 条码删除
     *    5.2 新增规格
     *      5.2.1 条码新增
     *    5.3 旧规格维护
     *      5.3.1 条码删去不存在的
     *      5.3.2 条码新增
     *      5.3.3 条码保存
     *    5.4 规格保存
     * 6.修改商品文件
     *    6.1 6.2 6.3
     * 7.修改商品描述
     *    7.1 必填只可修改
     */
    //1.修改商品
    Product product = productDto.getProduct();
    Product current = this.updateForm(product);
    //2.修改商品单位与价格
    Set<ProductUnitAndPrice> productUnitAndPrices = product.getProductUnitAndPrices();
    productUnitAndPriceService.updateFormBatch(productUnitAndPrices, current);

    //修改商品价格 移动到 商品规格中去修改
    //4.修改商品多规格基本信息
    Set<ProductMultipleSpecification> multipleSpecifications = product.getProductMultipleSpecifications();
    productMultipleSpecificationService.updateFormBatch(multipleSpecifications, product);
    //5.修改商品规格信息
    Set<ProductSpecification> specifications = product.getProductSpecifications();
    productSpecificationService.updateFormBatch(specifications, product);
    //6.修改商品文件
    Set<ProductFile> files = product.getProductFiles();
    productFileService.updateFormBatch(files, product);
    //7.修改商品描述
    ProductDescription description = productDto.getProductDescription();
    //商品描述允许为空
    if (description != null) {
      String id = description.getId();
      if (StringUtils.isNotBlank(id)) {
        productDescriptionService.updateForm(description);
      } else {
        productDescriptionService.createInsertAbleEntity(description, product);
      }
    } else {
      //如果修改商品描述为空，删除之前的商品描述数据
      ProductDescription productDescription = productDescriptionService.findDetailsByProduct(product.getProductCode());
      if (productDescription != null) {
        productDescriptionService.delete(productDescription);
      }
    }
    if (!CollectionUtils.isEmpty(productEventListeners)) {
      for (ProductEventListener productEventListener : productEventListeners) {
        productEventListener.onUpdate(current);
      }
    }

    //====================================================
    return current;
  }

  @Transactional
  @Override
  public Product updateForm(Product product) {
    this.updateValidation(product);
    // ===================基本信息
    String currentId = product.getId();
    Optional<Product> op_currentProduct = this.productRepository.findById(currentId);
    Product currentProduct = op_currentProduct.orElse(null);
    currentProduct = Validate.notNull(currentProduct, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentProduct.setModifyAccount(SecurityUtils.getUserAccount());
    currentProduct.setModifyTime(now);
    //由于入参只传了ID
    String brandId = product.getProductBrand().getId();
    Validate.notBlank(brandId, "品牌信息不能为空");
    ProductBrand brand = productBrandService.findById(brandId);
    Validate.notNull(brand, "品牌信息不存在");
    currentProduct.setProductBrand(brand);
    String categoryId = product.getProductCategory().getId();
    Validate.notBlank(categoryId, "分类信息不能为空");
    ProductCategory category = productCategoryService.findById(categoryId);
    Validate.notNull(category, "分类信息不存在");
    List<ProductCategory> byPCode = productCategoryService.findByPCode(category.getCode());
    Validate.isTrue(CollectionUtils.isEmpty(byPCode), "请选择商品分类最末级分类");
    currentProduct.setProductCategory(category);

    String showCategoryId = product.getProductShowCategory().getId();
    ProductShowCategory productShowCategory = productShowCategoryService.findById(showCategoryId);
    Validate.notNull(productShowCategory, "前端展示分类信息不存在");
    List<ProductShowCategory> productCategories = productShowCategoryService.findByPCode(productShowCategory.getCode());
    Validate.isTrue(CollectionUtils.isEmpty(productCategories), "请选择商品前端展示分类最末级分类");
    currentProduct.setProductShowCategory(productShowCategory);
    Set<ProductTag> tagsIn = product.getTags();
    if (!CollectionUtils.isEmpty(tagsIn)) {
      Set<ProductTag> tags = new HashSet<>(tagsIn.size());
      for (ProductTag tag : tagsIn) {
        ProductTag byId = productTagService.findById(tag.getId());
        tags.add(byId);
      }
      currentProduct.setTags(tags);
    }
    //新增商品状态设置， 新增商品 图片是必传 所以直接设置为已配置
    currentProduct.setProductImageStatus(ProductImageStatus.COMPLETE.getStatus());
    currentProduct.setSpecificationImageStatus(SpecificationImageStatus.COMPLETE.getStatus());

    // 开始重新赋值——一般属性
    currentProduct.setProductCode(product.getProductCode());
    currentProduct.setProductName(product.getProductName());
    currentProduct.setShelfStatus(product.getShelfStatus());
    currentProduct.setDefaultWarehouseCode(product.getDefaultWarehouseCode());
    currentProduct.setSearchKeyword(product.getSearchKeyword());
    currentProduct.setWarehouseName(product.getWarehouseName());
    currentProduct.setRemark(product.getRemark());


    this.productRepository.saveAndFlush(currentProduct);
    return currentProduct;
  }

  /**
   * 在更新一个已有的Product模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(Product product) {
    Validate.isTrue(!StringUtils.isBlank(product.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");

    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(product.getProductCode(), "修改信息时，商品编码不能为空！");
    Validate.notBlank(product.getProductName(), "修改信息时，商品名称不能为空！");
    Validate.notNull(product.getShelfStatus(), "修改信息时，上架状态不能为空！");

    // 重复性判断，基本属性，需要满足unique = true
    Product currentForProductCode = this.findByProductCode(product.getProductCode());
    Validate.isTrue(currentForProductCode == null || StringUtils.equals(currentForProductCode.getId(), product.getId()), "商品编码已存在,请检查");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(product.getProductCode() == null || product.getProductCode().length() < 64, "商品编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(product.getProductName() == null || product.getProductName().length() < 128, "商品名称,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(product.getDefaultWarehouseCode() == null || product.getDefaultWarehouseCode().length() < 255, "默认仓库,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(product.getSearchKeyword() == null || product.getSearchKeyword().length() < 255, "搜索关键字,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(product.getRemark() == null || product.getRemark().length() < 255, "备注信息,在进行修改时填入值超过了限定长度(255)，请检查!");

    // 关联性判断，关联属性判断，需要满足ManyToOne或者OneToOne，且not null 且是主模型
  }

  @Override
  public Set<Product> findDetailsByProductCategory(String productCategory) {
    if (StringUtils.isBlank(productCategory)) {
      return Sets.newHashSet();
    }
    return this.productRepository.findDetailsByProductCategory(productCategory, TenantUtils.getTenantCode());
  }

  @Override
  public Set<Product> findDetailsByProductBrand(String productBrand) {
    if (StringUtils.isBlank(productBrand)) {
      return Sets.newHashSet();
    }
    return this.productRepository.findDetailsByProductBrand(productBrand, TenantUtils.getTenantCode());
  }

  @Override
  public Set<Product> findDetailsByTags(String tags) {
    if (StringUtils.isBlank(tags)) {
      return Sets.newHashSet();
    }
    return this.productRepository.findDetailsByTags(tags, TenantUtils.getTenantCode());
  }

  @Override
  public Product findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    Product product = this.productRepository.findDetailsById(id);
    Validate.notNull(product, "数据不存在");
    //获取规格
    Set<ProductSpecification> productSpecifications = productSpecificationService.findDetailsByProduct(product.getId());
    product.setProductSpecifications(productSpecifications);
    //获取规格与价格
    Set<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrices = productUnitSpecificationAndPriceService.findDetailsByProduct(product.getId());
    product.setProductUnitSpecificationAndPrices(productUnitSpecificationAndPrices);
    //获取单位与价格
    Set<ProductUnitAndPrice> unitAndPrices = productUnitAndPriceService.findDetailsByProduct(product.getId());
    product.setProductUnitAndPrices(unitAndPrices);
    //获取文件
    Set<ProductFile> files = productFileService.findDetailsByProduct(product.getId());
    product.setProductFiles(files);
    //获取规格键值
    Set<ProductMultipleSpecification> multipleSpecification = productMultipleSpecificationService.findDetailsByProduct(product.getId());
    product.setProductMultipleSpecifications(multipleSpecification);

    //客户定价
    Set<ProductPricing> pricing = productPricingService.findDetailsByProduct(product.getId());
    product.setProductPricings(pricing);

    // 商品分类展示父级名称
    this.categoryDisplayParentName(product);
    return product;
  }

  /**
   * 要展示商品分类的上层节点的所有的名称
   *
   * @param product
   */
  private void categoryDisplayParentName(Product product) {
    ProductCategory category = product.getProductCategory();
    String name = category.getName();
    String parentCode = category.getParentCode();
    do {
      if (StringUtils.isNotBlank(parentCode)) {
        ProductCategory parent = productCategoryService.findByCode(parentCode);
        if (parent != null) {
          name = parent.getName() + ">" + name;
        } else {
          break;
        }
        parentCode = parent.getParentCode();
      } else {
        break;
      }
    } while (true);
    category.setName(name);
  }

  @Override
  public Product findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<Product> op = productRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    Product current = this.findById(id);
    Validate.isTrue(!current.getShelfStatus().equals(ShelfStatus.AVAILABLE_NOW.getStatus()), "上架状态不允许删除");

    Set<ProductFile> productFiles = current.getProductFiles();
    if (!CollectionUtils.isEmpty(productFiles)) {
      productFileService.deleteByBatch(productFiles);
    }
    Set<ProductMultipleSpecification> productMultipleSpecifications = current.getProductMultipleSpecifications();
    if (!CollectionUtils.isEmpty(productMultipleSpecifications)) {
      productMultipleSpecificationService.deleteByBatch(productMultipleSpecifications);
    }
    Set<ProductSpecification> productSpecifications = current.getProductSpecifications();
    if (!CollectionUtils.isEmpty(productSpecifications)) {
      productSpecificationService.deleteByBatch(productSpecifications);
    }
    Set<ProductUnitSpecificationAndPrice> productUnitAndPrices = current.getProductUnitSpecificationAndPrices();
    if (!CollectionUtils.isEmpty(productUnitAndPrices)) {
      productUnitSpecificationAndPriceService.deleteByBatch(productUnitAndPrices);
    }
    Set<ProductPricing> productPricings = current.getProductPricings();
    if (!CollectionUtils.isEmpty(productPricings)) {
      productPricingService.deleteByBatch(productPricings);
    }
    ProductDescription productDescription = productDescriptionService.findByProductCode(current.getProductCode());
    if (Objects.nonNull(productDescription)) {
      productDescriptionService.delete(productDescription);
    }

    if (current != null) {
      this.productRepository.delete(current);
    }
    if (!CollectionUtils.isEmpty(productEventListeners)) {
      for (ProductEventListener productEventListener : productEventListeners) {
        productEventListener.onDelete(current);
      }
    }

  }

  @Override
  public Product findByProductCode(String productCode) {
    if (StringUtils.isBlank(productCode)) {
      return null;
    }
    return this.productRepository.findByProductCode(productCode, TenantUtils.getTenantCode());
  }


  @Transactional
  @Override
  public void updateShelfStatus(Integer shelfStatus, String productCode) {
    Validate.notNull(shelfStatus, "上架状态不能为空");
    Validate.notBlank(productCode, "商品编号不能为空");
    this.productRepository.updateShelfStatus(shelfStatus, productCode, TenantUtils.getTenantCode());
    if (!CollectionUtils.isEmpty(productEventListeners)) {
      for (ProductEventListener productEventListener : productEventListeners) {
        productEventListener.onUpdateShelfStatus(productCode, shelfStatus);
      }
    }
  }

  @Override
  public List<Product> findByTenantCodeAndProductCodeIn(List<String> productCodeList) {
    if (CollectionUtils.isEmpty(productCodeList)) {
      return Lists.newLinkedList();
    }
    return productRepository.findByTenantCodeAndProductCodeIn(TenantUtils.getTenantCode(), productCodeList);
  }


  @Override
  public void updateImageResourceStatus(String productId) {
    Validate.notBlank(productId, "商品ID不能为空");
    Product product = this.findById(productId);
    Validate.notNull(product, "商品信息为空");
    Validate.notEmpty(product.getProductSpecifications(), "商品规格为空");
    //规格图片信息是否全部配置验证
    Integer specificationImageResourceStatus = SpecificationImageStatus.COMPLETE.getStatus();
    for (ProductSpecification productSpecification : product.getProductSpecifications()) {
      if (productSpecification.getProductImageResource() == null) {
        specificationImageResourceStatus = SpecificationImageStatus.UNFINISHED.getStatus();
        break;
      }
    }
    product.setSpecificationImageStatus(specificationImageResourceStatus);
    if (specificationImageResourceStatus.equals(SpecificationImageStatus.UNFINISHED.getStatus())) {
      product.setProductImageStatus(ProductImageStatus.UNFINISHED.getStatus());
    } else {
      //主图是否配置
      if (CollectionUtils.isEmpty(product.getProductFiles())) {
        product.setProductImageStatus(ProductImageStatus.UNFINISHED.getStatus());
      } else {
        product.setProductImageStatus(ProductImageStatus.COMPLETE.getStatus());
      }
    }
    productRepository.saveAndFlush(product);
  }
}
