package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.common.util.SecurityUtils;
import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.empower.business.product.entity.ProductShowCategory;
import com.bizunited.empower.business.product.repository.ProductShowCategoryRepository;
import com.bizunited.empower.business.product.service.ProductShowCategoryService;
import com.bizunited.empower.business.product.service.notifier.ProductShowCategoryEventListener;
import com.bizunited.empower.business.product.vo.ProductShowCategoryVo;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;

import java.lang.Override;
import java.lang.String;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;
import javax.transaction.Transactional;

import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

/**
 * ProductShowCategory业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("ProductShowCategoryServiceImpl")
public class ProductShowCategoryServiceImpl implements ProductShowCategoryService {

  /**
   * 最大层级限制
   */
  final static Integer MAX_LEVEL = 2;

  /**
   * 顶层标识
   */
  final static Integer TOP_LEVEL = 0;

  @Autowired(required = false)
  private List<ProductShowCategoryEventListener> productShowCategoryEventListeners;

  @Autowired
  private RedisMutexService redisMutexService;
  /**
   * 商品品牌编号前缀
   */
  private static final String PRODUCT_CATEGORY_CODE_PREFIX = "PS";
  /**
   * 商品品牌编号生成所使用的redis分布式锁的前缀
   */
  private static final String PRODUCT_CATEGORY_REDIS_LOCK_CODE = "PS_PRODUCT_SHOW_CATEGORY_";

  @Autowired
  private ProductShowCategoryRepository productShowCategoryRepository;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Transactional
  @Override
  public ProductShowCategory create(ProductShowCategory productShowCategory) {
    ProductShowCategory current = this.createForm(productShowCategory);
    //====================================================
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductShowCategory createForm(ProductShowCategory productShowCategory) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    Date now = new Date();
    productShowCategory.setCreateAccount(SecurityUtils.getUserAccount());
    productShowCategory.setCreateTime(now);
    productShowCategory.setModifyAccount(SecurityUtils.getUserAccount());
    productShowCategory.setModifyTime(now);

    // =============系统自构数据==================
    String tenantCode = this.getTenantCode();
    productShowCategory.setTenantCode(tenantCode);
    String productCategoryCode = this.generateCode(tenantCode);
    productShowCategory.setCode(productCategoryCode);
    // 获取父级的flatCode；
    String pFlatCode = this.getParentFlatCode(productShowCategory.getParentCode());
    productShowCategory.setFlatCode(StringUtils.join(pFlatCode, productCategoryCode));
    // ===============================
    this.createValidation(productShowCategory);

    // ===============================
    //  和业务有关的验证填写在这个区域
    // ===============================

    this.productShowCategoryRepository.save(productShowCategory);

    // 返回最终处理的结果，里面带有详细的关联信息
    return productShowCategory;
  }

  private String generateCode(String tenantCode) {
    String lockCode = StringUtils.join(PRODUCT_CATEGORY_REDIS_LOCK_CODE, tenantCode);
    String atomicNumber = this.redisMutexService.getAndIncrement(lockCode, 1, 6);
    return StringUtils.join(PRODUCT_CATEGORY_CODE_PREFIX, atomicNumber);
  }

  private String getTenantCode() {
    String tenantCode = TenantUtils.getTenantCode();
    return tenantCode;
  }

  private String getParentFlatCode(String pCode) {
    String flatCode = "";
    if (StringUtils.isNotBlank(pCode)) {
      ProductShowCategory category = this.findByCode(pCode);
      Validate.notNull(category, "父级分类不存在");
      flatCode = category.getFlatCode();
    }
    return flatCode;
  }

  /**
   * 在创建一个新的ProductCategory模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProductShowCategory productShowCategory) {
    Validate.notNull(productShowCategory, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(productShowCategory.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    productShowCategory.setId(null);
    Validate.notBlank(productShowCategory.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(productShowCategory.getName(), "添加信息时，分类名称不能为空！");
    Validate.notBlank(productShowCategory.getCode(), "添加信息时，分类编码不能为空！");
    Validate.isTrue(productShowCategory.getLevel() != null && productShowCategory.getLevel() < MAX_LEVEL, "添加信息时，层级只有【%d】级", MAX_LEVEL);
    if (productShowCategory.getLevel() != 0) {
      Validate.notBlank(productShowCategory.getParentCode(), "添加信息时，父级编号不能为空！");
      ProductShowCategory productCategoryParent = productShowCategoryRepository.findByCodeAndTenantCode(productShowCategory.getParentCode(), TenantUtils.getTenantCode());
      // 检验父节点是不是末位节点
      Validate.notNull(productCategoryParent, "添加信息时，父节点不存在！");
      Validate.isTrue(productCategoryParent.getLevel() < MAX_LEVEL - 1, "添加信息时，末位节点不允许再添加下级节点");
    }
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(productShowCategory.getTenantCode() == null || productShowCategory.getTenantCode().length() < 255, "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productShowCategory.getName() == null || productShowCategory.getName().length() < 255, "分类名称,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productShowCategory.getCode() == null || productShowCategory.getCode().length() < 64, "分类编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productShowCategory.getParentCode() == null || productShowCategory.getParentCode().length() < 64, "父级编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productShowCategory.getFlatCode() == null || productShowCategory.getFlatCode().length() < 255, "快速编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    ProductShowCategory currentProductCategory = this.findByCode(productShowCategory.getCode());
    Validate.isTrue(currentProductCategory == null, "分类编码已存在,请检查");
    List<ProductShowCategory> list = productShowCategoryRepository.findByNameAndTenantCode(productShowCategory.getName(), TenantUtils.getTenantCode());
    Validate.isTrue(CollectionUtils.isEmpty(list), "添加信息时，分类名称已存在！");
  }

  @Transactional
  @Override
  public ProductShowCategory update(ProductShowCategory productShowCategory) {
    ProductShowCategory current = this.updateForm(productShowCategory);
    //====================================================
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductShowCategory updateForm(ProductShowCategory productShowCategory) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(productShowCategory);
    // ===================基本信息
    String currentId = productShowCategory.getId();
    Optional<ProductShowCategory> op_currentProductCategory = this.productShowCategoryRepository.findById(currentId);
    ProductShowCategory currentProductCategory = op_currentProductCategory.orElse(null);
    currentProductCategory = Validate.notNull(currentProductCategory, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentProductCategory.setModifyAccount(SecurityUtils.getUserAccount());
    currentProductCategory.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentProductCategory.setTenantCode(productShowCategory.getTenantCode());
    currentProductCategory.setName(productShowCategory.getName());
    currentProductCategory.setCode(productShowCategory.getCode());
    currentProductCategory.setLevel(productShowCategory.getLevel());
    currentProductCategory.setParentCode(productShowCategory.getParentCode());
    currentProductCategory.setFlatCode(productShowCategory.getFlatCode());

    this.productShowCategoryRepository.saveAndFlush(currentProductCategory);
    return currentProductCategory;
  }

  /**
   * 在更新一个已有的ProductCategory模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ProductShowCategory productShowCategory) {
    Validate.isTrue(!StringUtils.isBlank(productShowCategory.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    productShowCategory.setTenantCode(TenantUtils.getTenantCode());
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(productShowCategory.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(productShowCategory.getName(), "修改信息时，分类名称不能为空！");
    Validate.notBlank(productShowCategory.getCode(), "修改信息时，分类编码不能为空！");
    if (productShowCategory.getLevel() != 0) {
      Validate.notBlank(productShowCategory.getParentCode(), "修改信息时，父级编号不能为空！");
      // 检验父节点是不是末位节点
      ProductShowCategory productCategoryParent = productShowCategoryRepository.findByCodeAndTenantCode(productShowCategory.getParentCode(), TenantUtils.getTenantCode());
      Validate.notNull(productCategoryParent, "修改信息时，父节点不存在！");
      Validate.isTrue(productCategoryParent.getLevel() < MAX_LEVEL - 1, "修改信息时，末位节点不允许作为父级节点");
    }

    // 重复性判断，基本属性，需要满足unique = true
    ProductShowCategory currentForCode = this.findByCode(productShowCategory.getCode());
    Validate.isTrue(currentForCode != null && StringUtils.equals(currentForCode.getId(), productShowCategory.getId()), "分类编码已存在,请检查");
    // 重复性判断，同租户下名称不能重复
    if (!currentForCode.getName().equals(productShowCategory.getName())) {
      List<ProductShowCategory> list = productShowCategoryRepository.findByNameAndTenantCode(productShowCategory.getName(), TenantUtils.getTenantCode());
      Validate.isTrue(CollectionUtils.isEmpty(list), "修改信息时，分类名称已存在！");
    }
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(productShowCategory.getTenantCode() == null || productShowCategory.getTenantCode().length() < 255, "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productShowCategory.getName() == null || productShowCategory.getName().length() < 255, "分类名称,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productShowCategory.getCode() == null || productShowCategory.getCode().length() < 64, "分类编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productShowCategory.getParentCode() == null || productShowCategory.getParentCode().length() < 64, "父级编号,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productShowCategory.getFlatCode() == null || productShowCategory.getFlatCode().length() < 255, "快速编号,在进行修改时填入值超过了限定长度(255)，请检查!");
  }

  @Override
  public ProductShowCategory findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.productShowCategoryRepository.findDetailsById(id);
  }

  @Override
  public ProductShowCategory findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<ProductShowCategory> op = productShowCategoryRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public String deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    ProductShowCategory current = this.findById(id);
    if (current != null) {
      Set<Product> products = current.getProducts();
      Validate.isTrue(CollectionUtils.isEmpty(products), "已绑定商品，不允许删除！");
      List<ProductShowCategory> child = this.findByPCode(current.getCode());
      Validate.isTrue(CollectionUtils.isEmpty(child), "当前分类下有子分类，不可删除！");
      this.productShowCategoryRepository.delete(current);
      StringBuilder msg = new StringBuilder();
      // 触发事件
      if (!CollectionUtils.isEmpty(productShowCategoryEventListeners)) {
        for (ProductShowCategoryEventListener listener : productShowCategoryEventListeners) {
          String resultMsg = listener.onDelete(current);
          if (StringUtils.isNotBlank(resultMsg)) {
            msg.append(resultMsg);
          }
        }
      }
      return msg.toString();
    }
    return "";
  }

  @Override
  public ProductShowCategory findByCode(String code) {
    if (StringUtils.isBlank(code)) {
      return null;
    }
    return this.productShowCategoryRepository.findByCode(code, TenantUtils.getTenantCode());
  }

  /**
   * 分页查询
   *
   * @param pageable
   * @return
   */
  @Override
  public Page<ProductShowCategory> findByConditions(Pageable pageable) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    Page<ProductShowCategory> rolePage = this.productShowCategoryRepository.findByTenantCodeOrderByCreateTimeDesc(TenantUtils.getTenantCode(), pageable);
    if (CollectionUtils.isEmpty(rolePage.getContent())) {
      return Page.empty(pageable);
    }
    return new PageImpl<>(Lists.newArrayList(rolePage.getContent()), pageable, rolePage.getTotalElements());

  }


  /**
   * 根据父级code查询下一级，不传则查询所有顶层
   *
   * @param pCode
   * @return
   */
  @Override
  public List<ProductShowCategory> findByPCode(String pCode) {
    if (StringUtils.isBlank(pCode)) {
      return this.productShowCategoryRepository.findByLevelAndTenantCodeOrderByCreateTimeDesc(TOP_LEVEL, TenantUtils.getTenantCode());
    }
    return this.productShowCategoryRepository.findByParentCode(pCode, TenantUtils.getTenantCode());
  }

  @Override
  public List<ProductShowCategory> findByCodeList(List<String> codeList) {
    if (CollectionUtils.isEmpty(codeList)) {
      return Lists.newLinkedList();
    }
    List<ProductShowCategory> list = productShowCategoryRepository.findByTenantCodeAndCodeIn(TenantUtils.getTenantCode(), codeList);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    // codeList 的顺序 要和结果List里的一致
    Map<String, ProductShowCategory> map = new HashMap<>(codeList.size());
    for (ProductShowCategory category : list) {
      map.put(category.getCode(), category);
    }
    List<ProductShowCategory> resultList = new ArrayList<>(codeList.size());
    for (String code : codeList) {
      resultList.add(map.get(code));
    }
    return resultList;
  }

  @Override
  public List<ProductShowCategoryVo> findStructureTree() {

    List<ProductShowCategory> all = this.productShowCategoryRepository.findByTenantCode(TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(all)) {
      return Lists.newArrayList();
    }
    /**
     * 将品牌信息组装为父子树
     * 自己的flat - 自己的code = 父的flat
     */
    Collection<ProductShowCategoryVo> productBrandVos = this.nebulaToolkitService.copyCollectionByWhiteList(all, ProductShowCategory.class, ProductShowCategoryVo.class, HashSet.class, ArrayList.class);
    Map<String, Set<ProductShowCategoryVo>> collect = productBrandVos.stream().collect(Collectors.groupingBy((ProductShowCategoryVo vo) -> {
      String flatCode = vo.getFlatCode();
      String brandCode = vo.getCode();
      return flatCode.replace(brandCode, "");
    }, Collectors.toSet()));
    Set<ProductShowCategoryVo> topSet = collect.get("");
    List<ProductShowCategoryVo> topList = topSet.stream().sorted(Comparator.comparing(ProductShowCategoryVo::getCreateTime).reversed()).collect(Collectors.toList());
    for (ProductShowCategoryVo top : topList) {
      // 递归取子
      this.getChildren(top, collect);
    }
    return topList;
  }

  @SuppressWarnings({"unchecked", "rawtypes"})
  private void getChildren(ProductShowCategoryVo top, Map<String, Set<ProductShowCategoryVo>> collect) {
    Set<ProductShowCategoryVo> children = collect.get(top.getFlatCode());
    if (CollectionUtils.isEmpty(children)) {
      return;
    }
    Set<ProductShowCategoryVo> childrenTreeSet = new TreeSet(Comparator.comparing(ProductShowCategoryVo::getCreateTime).reversed());
    children.stream().forEach(productCategoryVo -> childrenTreeSet.add(productCategoryVo));
    for (ProductShowCategoryVo child : childrenTreeSet) {
      this.getChildren(child, collect);
    }
    top.setChildren(childrenTreeSet);
  }
} 
