package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.common.util.SecurityUtils;
import com.bizunited.empower.business.product.common.enumerate.AdjustType;
import com.bizunited.empower.business.product.common.enumerate.PriceType;
import com.bizunited.empower.business.product.dto.SpecificationImageDto;
import com.bizunited.empower.business.product.dto.SpecificationPriceDto;
import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.empower.business.product.entity.ProductBarCodeInfo;
import com.bizunited.empower.business.product.entity.ProductImageResource;
import com.bizunited.empower.business.product.entity.ProductPriceModifyLog;
import com.bizunited.empower.business.product.entity.ProductPricing;
import com.bizunited.empower.business.product.entity.ProductPricingUnitSpecification;
import com.bizunited.empower.business.product.entity.ProductSpecification;

import com.bizunited.empower.business.product.entity.ProductUnit;
import com.bizunited.empower.business.product.entity.ProductUnitSpecificationAndPrice;
import com.bizunited.empower.business.product.repository.ProductSpecificationRepository;
import com.bizunited.empower.business.product.service.ProductActionService;
import com.bizunited.empower.business.product.service.ProductBarCodeInfoService;
import com.bizunited.empower.business.product.service.ProductImageResourceService;
import com.bizunited.empower.business.product.service.ProductPriceModifyLogService;
import com.bizunited.empower.business.product.service.ProductPricingService;
import com.bizunited.empower.business.product.service.ProductPricingUnitSpecificationService;
import com.bizunited.empower.business.product.service.ProductService;
import com.bizunited.empower.business.product.service.ProductSpecificationService;
import com.bizunited.empower.business.product.service.ProductUnitSpecificationAndPriceService;
import com.bizunited.empower.business.product.service.notifier.ProductSpecificationEventListener;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Sets;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * ProductSpecification业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("ProductSpecificationServiceImpl")
public class ProductSpecificationServiceImpl implements ProductSpecificationService {
  @Autowired
  private ProductSpecificationRepository productSpecificationRepository;

  @Autowired
  private ProductBarCodeInfoService productBarCodeInfoService;

  @Autowired
  private ProductUnitSpecificationAndPriceService productUnitSpecificationAndPriceService;
  @Autowired
  private ProductPricingService productPricingService;

  @Autowired
  private ProductPricingUnitSpecificationService productPricingUnitSpecificationService;

  @Autowired
  private RedisMutexService redisMutexService;

  @Autowired
  private ProductService productService;
  @Autowired
  private ProductImageResourceService productImageResourceService;

  @Autowired(required = false)
  private List<ProductSpecificationEventListener> productSpecificationEventListeners;

  @Autowired
  private ProductPriceModifyLogService productPriceModifyLogService;

  /**
   * 商品品牌编号前缀
   */
  private static final String PRODUCT_SPECIFICATION_CODE_PREFIX = "PS";
  /**
   * 商品品牌编号生成所使用的redis分布式锁的前缀
   */
  private static final String PRODUCT_SPECIFICATION_REDIS_LOCK_CODE = "PS_PRODUCT_SPECIFICATION";

  /**
   * 批量修改价格最低价
   */
  private static final BigDecimal UPDATE_MIN_PRICE = new BigDecimal("0.01");


  /**
   * 批量新增商品规格
   *
   * @param specifications
   * @param product
   */
  @Override
  public void createInsertAbleEntitySet(Set<ProductSpecification> specifications, Product product) {
    Validate.notNull(product, "未传入商品信息！");
    if (!CollectionUtils.isEmpty(specifications)) {
      for (ProductSpecification specification : specifications) {
        specification.setProduct(product);
        this.createInsertAbleEntity(specification);
      }
      productSpecificationRepository.saveAll(specifications);

      for (ProductSpecification specification : specifications) {
        Set<ProductBarCodeInfo> productBarCodeInfos = specification.getProductBarCodeInfos();
        productBarCodeInfoService.createInsertAbleEntitySet(productBarCodeInfos, specification);

        // 单位与价格
        Set<ProductUnitSpecificationAndPrice> productUnitAndPrices = specification.getProductUnitSpecificationAndPrices();
        productUnitSpecificationAndPriceService.createInsertAbleEntitySet(productUnitAndPrices, product, specification);
        // 产品定价
        Set<ProductPricing> productPricings = specification.getProductPricings();
        productPricingService.createInsertAbleEntitySet(productPricings, product, specification);

      }

    }
  }

  @Transactional
  @Override
  public void deleteByBatch(Set<ProductSpecification> productSpecifications) {
    if (!CollectionUtils.isEmpty(productSpecifications)) {
      for (ProductSpecification specification : productSpecifications) {
        Set<ProductBarCodeInfo> barCodeInfos = specification.getProductBarCodeInfos();
        if (!CollectionUtils.isEmpty(barCodeInfos)) {
          this.productBarCodeInfoService.deleteByBatch(barCodeInfos);
        }
      }
      this.productSpecificationRepository.deleteAll(productSpecifications);
    }
  }

  @Override
  public ProductSpecification findByTenantCodeAndProductSpecificationCode(String productSpecificationCode) {
    Validate.notBlank(productSpecificationCode, "规格编码必传");
    return productSpecificationRepository.findByTenantCodeAndProductSpecificationCode(TenantUtils.getTenantCode(), productSpecificationCode);
  }

  @Transactional
  @Override
  public void updateFormBatch(Set<ProductSpecification> specifications, Product product) {
    /*5.修改商品规格信息
     *    5.1 规格删除
     *      5.1.1 条码删除
     *    5.2 新增规格
     *      5.2.1 条码新增
     *    5.3 旧规格维护
     *      5.3.1 条码删去不存在的
     *      5.3.2 条码新增
     *      5.3.3 条码保存
     *    5.4 规格保存
     *    5.5 修改商品 规格价格
     *    5.6 修改规格客户定价
     *
     */
    Validate.isTrue(!CollectionUtils.isEmpty(specifications), "修改商品时，商品规格不能为空");
    Map<Boolean, List<ProductSpecification>> idBooleanMap = specifications.stream()
        .collect(Collectors.partitioningBy(productSpecification -> StringUtils.isNotBlank(productSpecification.getId())));
    List<ProductSpecification> idTrueList = idBooleanMap.get(true);
    List<ProductSpecification> idFalseList = idBooleanMap.get(false);
    //规格删除
    List<String> ids = idTrueList.stream().map(ProductSpecification::getId).filter(StringUtils::isNotEmpty).collect(Collectors.toList());
    Set<ProductSpecification> needDeleteSet;
    if (CollectionUtils.isEmpty(ids)) {
      needDeleteSet = productSpecificationRepository.findByProduct(TenantUtils.getTenantCode(), product.getProductCode());
    } else {
      needDeleteSet = productSpecificationRepository.findByProductAndIdNotIn(TenantUtils.getTenantCode(), product.getProductCode(), ids);
    }
    if (!CollectionUtils.isEmpty(needDeleteSet)) {
      //规格删除前修改图片状态
      for (ProductSpecification productSpecification : needDeleteSet) {
       if(productSpecification.getProductImageResource()!=null){
         productImageResourceService.updateUseImage(productSpecification.getProductImageResource());
       }
      }
      this.deleteByBatch(needDeleteSet);
      //当删除商品规格时，触发规格删除事件
      if (!CollectionUtils.isEmpty(productSpecificationEventListeners)) {
        for (ProductSpecificationEventListener productSpecificationEventListener : productSpecificationEventListeners) {
          productSpecificationEventListener.onDelete(needDeleteSet);
        }
      }
    }
    // 5.2 新增规格
    this.createInsertAbleEntitySet(new HashSet<>(idFalseList), product);
    // 5.3 旧信息覆盖
    Set<ProductSpecification> saveAll = new HashSet<>(specifications.size());
    for (ProductSpecification productSpecification : idTrueList) {
      Optional<ProductSpecification> op_currentProductSpecification = this.productSpecificationRepository.findById(productSpecification.getId());
      ProductSpecification currentProductSpecification = op_currentProductSpecification.orElse(null);
      currentProductSpecification = Validate.notNull(currentProductSpecification, "未发现指定的原始模型对象信");


      //验证资源图片是否修改
      if (productSpecification.getProductImageResource()!=null&&StringUtils.isNotBlank(productSpecification.getProductImageResource().getId())&&currentProductSpecification.getProductImageResource()!=null&&currentProductSpecification.getProductImageResource().getId().equals(productSpecification.getProductImageResource().getId())) {
       productSpecification.setProductImageResource(currentProductSpecification.getProductImageResource());
      }else{
        //保存图片到资源库中
        ProductImageResource productImageResource = new ProductImageResource();
        productImageResource.setUseImage(true);
        productImageResource.setProduct(product);
        productImageResource.setFileName(productSpecification.getMainImageName());
        productImageResource.setRelativePath(productSpecification.getMainImagePath());
        productImageResourceService.create(productImageResource);
        //设置规格图片资源
        productSpecification.setProductImageResource(productImageResource);
        //将之前的图片设置为 未使用,这里处理之前老数据可能没有图片资源的情况
        if(currentProductSpecification.getProductImageResource()!=null) {
          productImageResourceService.updateUseImage(currentProductSpecification.getProductImageResource(), null, productSpecification.getId());
        }
      }

      productSpecification.setCreateAccount(currentProductSpecification.getCreateAccount());
      productSpecification.setCreateTime(currentProductSpecification.getCreateTime());
      productSpecification.setProduct(product);
      productSpecification.setModifyAccount(SecurityUtils.getUserAccount());
      productSpecification.setModifyTime(new Date());
      productSpecification.setTenantCode(TenantUtils.getTenantCode());
      Set<ProductBarCodeInfo> productBarCodeInfos = productSpecification.getProductBarCodeInfos();
      productBarCodeInfoService.updateFormBatch(productBarCodeInfos, productSpecification);

      //2.修改商品单位与价格
      Set<ProductUnitSpecificationAndPrice> productUnitAndPrices = productSpecification.getProductUnitSpecificationAndPrices();
      productUnitSpecificationAndPriceService.updateFormBatch(productUnitAndPrices, product, productSpecification);
      //3.修改商品定价
      Set<ProductPricing> productPricings = productSpecification.getProductPricings();
      productPricingService.updateFormBatch(productPricings, product, productSpecification);


      saveAll.add(productSpecification);
    }
    //当修改商品规格时，触发规格修改事件
    if (!CollectionUtils.isEmpty(idTrueList) && !CollectionUtils.isEmpty(productSpecificationEventListeners)) {
      for (ProductSpecificationEventListener productSpecificationEventListener : productSpecificationEventListeners) {
        productSpecificationEventListener.onUpdate(Sets.newHashSet(idTrueList));
      }
    }

    productSpecificationRepository.saveAll(saveAll);
  }

  @Override
  public ProductSpecification findByProductSpecificationCode(String productSpecificationCode) {
    if (StringUtils.isAnyBlank(productSpecificationCode, TenantUtils.getTenantCode())) {
      return null;
    }
    return productSpecificationRepository.findByProductSpecificationCodeAndTenantCode(productSpecificationCode, TenantUtils.getTenantCode());
  }

  @Override
  public Set<String> findProductCodeBySpecificationCodes(Set<String> specCodeSet) {
    if (CollectionUtils.isEmpty(specCodeSet)) {
      return Sets.newHashSet();
    }
    Set<String> productCodeSet = this.productSpecificationRepository.findProductCodeBySpecCodes(TenantUtils.getTenantCode(), specCodeSet);
    return Objects.nonNull(productCodeSet) ? productCodeSet : Sets.newHashSet();
  }

  private void createInsertAbleEntity(ProductSpecification specification) {
    //
    Date now = new Date();
    specification.setCreateAccount(SecurityUtils.getUserAccount());
    specification.setCreateTime(now);
    specification.setModifyAccount(SecurityUtils.getUserAccount());
    specification.setModifyTime(now);
    //
    String tenantCode = TenantUtils.getTenantCode();
    specification.setTenantCode(tenantCode);
    String specificationCode = this.generateCode(tenantCode);
    specification.setProductSpecificationCode(specificationCode);


    //保存图片到资源库中
    ProductImageResource productImageResource = new ProductImageResource();
    productImageResource.setUseImage(true);
    productImageResource.setProduct(specification.getProduct());
    productImageResource.setFileName(specification.getMainImageName());
    productImageResource.setRelativePath(specification.getMainImagePath());
    productImageResourceService.create(productImageResource);
    specification.setProductImageResource(productImageResource);

    this.createValidation(specification);
  }

  private String generateCode(String tenantCode) {
    String specificationCode = null;
    String lockCode = StringUtils.join(PRODUCT_SPECIFICATION_REDIS_LOCK_CODE, tenantCode);
    try {
      String atomicNumber = this.redisMutexService.getAndIncrement(lockCode, 1, 6);
      specificationCode = StringUtils.join(PRODUCT_SPECIFICATION_CODE_PREFIX, atomicNumber);
    } catch (Exception e) {
      throw new RuntimeException("生成规格编码失败", e);
    }
    return specificationCode;
  }

  @Transactional
  @Override
  public ProductSpecification create(ProductSpecification productSpecification) {
    ProductSpecification current = this.createForm(productSpecification);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductSpecification createForm(ProductSpecification productSpecification) {
    this.createInsertAbleEntity(productSpecification);

    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================

    this.productSpecificationRepository.save(productSpecification);

    // 返回最终处理的结果，里面带有详细的关联信息
    return productSpecification;
  }

  /**
   * 在创建一个新的ProductSpecification模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProductSpecification productSpecification) {
    Validate.notNull(productSpecification, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(productSpecification.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    productSpecification.setId(null);
    Validate.notBlank(productSpecification.getProductSpecificationCode(), "添加信息时，规格编号不能为空！");
    Validate.notBlank(productSpecification.getProductSpecificationName(), "添加信息时，规格名称不能为空！");
    Validate.notBlank(productSpecification.getMainImagePath(), "请补充商品规格主图！");
    Validate.notBlank(productSpecification.getMainImageName(), "添加信息时，主图名称不能为空！");
    // ----------------------------------
    ProductSpecification psUQ = productSpecificationRepository.findByTenantCodeAndProductSpecificationCode(TenantUtils.getTenantCode(), productSpecification.getProductSpecificationCode());
    Validate.isTrue(Objects.isNull(psUQ), "商品规格编号重复生成，请重新点击提交，无需更改页面");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(productSpecification.getProductSpecificationCode() == null || productSpecification.getProductSpecificationCode().length() < 64, "规格编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productSpecification.getProductSpecificationName() == null || productSpecification.getProductSpecificationName().length() < 128, "规格名称,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(productSpecification.getMainImagePath() == null || productSpecification.getMainImagePath().length() < 255, "主图路径,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productSpecification.getMainImageName() == null || productSpecification.getMainImageName().length() < 255, "主图名称,在进行添加时填入值超过了限定长度(255)，请检查!");
  }

  @Transactional
  @Override
  public ProductSpecification update(ProductSpecification productSpecification) {
    ProductSpecification current = this.updateForm(productSpecification);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductSpecification updateForm(ProductSpecification productSpecification) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(productSpecification);
    // ===================基本信息
    String currentId = productSpecification.getId();
    Optional<ProductSpecification> op_currentProductSpecification = this.productSpecificationRepository.findById(currentId);
    ProductSpecification currentProductSpecification = op_currentProductSpecification.orElse(null);
    currentProductSpecification = Validate.notNull(currentProductSpecification, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentProductSpecification.setModifyAccount(SecurityUtils.getUserAccount());
    currentProductSpecification.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentProductSpecification.setProductSpecificationCode(productSpecification.getProductSpecificationCode());
    currentProductSpecification.setProductSpecificationName(productSpecification.getProductSpecificationName());
    currentProductSpecification.setMinimumOrderQuantity(productSpecification.getMinimumOrderQuantity());
    currentProductSpecification.setMaximumOrderQuantity(productSpecification.getMaximumOrderQuantity());
    currentProductSpecification.setMainImagePath(productSpecification.getMainImagePath());
    currentProductSpecification.setMainImageName(productSpecification.getMainImageName());
    currentProductSpecification.setProduct(productSpecification.getProduct());

    this.productSpecificationRepository.saveAndFlush(currentProductSpecification);
    return currentProductSpecification;
  }

  /**
   * 在更新一个已有的ProductSpecification模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ProductSpecification productSpecification) {
    Validate.isTrue(!StringUtils.isBlank(productSpecification.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");

    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(productSpecification.getProductSpecificationCode(), "修改信息时，规格编号不能为空！");
    Validate.notBlank(productSpecification.getProductSpecificationName(), "修改信息时，规格名称不能为空！");
    Validate.notBlank(productSpecification.getMainImagePath(), "请补充商品规格主图！");
    Validate.notBlank(productSpecification.getMainImageName(), "修改信息时，主图名称不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(productSpecification.getProductSpecificationCode() == null || productSpecification.getProductSpecificationCode().length() < 64, "规格编号,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productSpecification.getProductSpecificationName() == null || productSpecification.getProductSpecificationName().length() < 128, "规格名称,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(productSpecification.getMainImagePath() == null || productSpecification.getMainImagePath().length() < 255, "主图路径,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productSpecification.getMainImageName() == null || productSpecification.getMainImageName().length() < 255, "主图名称,在进行修改时填入值超过了限定长度(255)，请检查!");

    // 关联性判断，关联属性判断，需要满足ManyToOne或者OneToOne，且not null 且是主模型
  }

  @Override
  public Set<ProductSpecification> findDetailsByProduct(String product) {
    if (StringUtils.isBlank(product)) {
      return Sets.newHashSet();
    }
    Set<ProductSpecification> productSpecifications = this.productSpecificationRepository.findDetailsByProduct(product);
    this.mergePricing(productSpecifications);
    return productSpecifications;
  }

  @Override
  public ProductSpecification findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    ProductSpecification productSpecification = this.productSpecificationRepository.findDetailsById(id);
    productSpecification.setProductPricings(this.mergePricing(productSpecification));
    return productSpecification;
  }

  /**
   * 合并商品规格定价信息
   *
   * @param productSpecification
   * @return
   */
  private Set<ProductPricing> mergePricing(ProductSpecification productSpecification) {
    Set<ProductPricing> productPricings = Sets.newHashSet();
    Set<ProductPricingUnitSpecification> productPricingUnitSpecifications = productPricingUnitSpecificationService.findByProductSpecificationCode(productSpecification.getProductSpecificationCode());
    if (CollectionUtils.isEmpty(productPricingUnitSpecifications)) {
      return productPricings;
    }
    //将规格定价信息放到客户定价模型中
    Map<String, ProductPricing> productPricingMap = new HashMap<>();
    for (ProductPricingUnitSpecification productPricingUnitSpecification : productPricingUnitSpecifications) {
      if (productPricingMap.containsKey(productPricingUnitSpecification.getProductPricing().getId())) {
        //如果已经存在， 添加到当前集合中
        productPricingMap.get(productPricingUnitSpecification.getProductPricing().getId()).getProductPricingUnitSpecifications().add(productPricingUnitSpecification);
      } else {
        //初始化 客户定价信息 保存到map集合中
        productPricingUnitSpecification.getProductPricing().setProductPricingUnitSpecifications(new HashSet<>());
        productPricingUnitSpecification.getProductPricing().getProductPricingUnitSpecifications().add(productPricingUnitSpecification);
        productPricingMap.put(productPricingUnitSpecification.getProductPricing().getId(), productPricingUnitSpecification.getProductPricing());
      }
    }
    return new HashSet<>(productPricingMap.values());
  }

  /**
   * 合并商品规格定价信息
   *
   * @return
   */
  private void mergePricing(Set<ProductSpecification> productSpecifications) {
    if (CollectionUtils.isEmpty(productSpecifications)) {
      return;
    }
    List<String> productSpecificationCodes = productSpecifications.stream().map(ProductSpecification::getProductSpecificationCode).collect(Collectors.toList());

    Set<ProductPricingUnitSpecification> productPricingUnitSpecifications = productPricingUnitSpecificationService.findByProductSpecificationCodes(productSpecificationCodes);
    //将规格定价信息放到客户定价模型中
    Map<String, Map<String, ProductPricing>> productPricingMap = new HashMap<>();
    for (ProductPricingUnitSpecification productPricingUnitSpecification : productPricingUnitSpecifications) {
      String mapKey = productPricingUnitSpecification.getProductSpecificationCode();
      String childKey = productPricingUnitSpecification.getProductPricing().getId();
      if (productPricingMap.containsKey(mapKey)) {
        //如果已经存在， 添加到当前集合中
        Map<String, ProductPricing> child = productPricingMap.get(mapKey);
        if (child.containsKey(childKey)) {
          //如果当前子集map 已经存在,那么累加定价信息
          ProductPricing productPricing = child.get(childKey);
          productPricing.getProductPricingUnitSpecifications().add(productPricingUnitSpecification);

        } else {
          productPricingUnitSpecification.getProductPricing().setProductPricingUnitSpecifications(new HashSet<>());
          productPricingUnitSpecification.getProductPricing().getProductPricingUnitSpecifications().add(productPricingUnitSpecification);
          child.put(childKey, productPricingUnitSpecification.getProductPricing());
        }
      } else {
        //初始化 客户定价信息 保存到map集合中
        Map<String, ProductPricing> child = new HashMap<>();
        productPricingUnitSpecification.getProductPricing().setProductPricingUnitSpecifications(new HashSet<>());
        productPricingUnitSpecification.getProductPricing().getProductPricingUnitSpecifications().add(productPricingUnitSpecification);
        child.put(childKey, productPricingUnitSpecification.getProductPricing());
        productPricingMap.put(mapKey, child);
      }
    }
  }


  @Override
  public ProductSpecification findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<ProductSpecification> op = productSpecificationRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    ProductSpecification current = this.findById(id);
    if (current != null) {
      this.productSpecificationRepository.delete(current);
    }
  }


  @Override
  @Transactional
  public void updateImage(SpecificationImageDto specificationImageDto) {
    Validate.notNull(specificationImageDto, "规格修改对象不能为空");
    Validate.notBlank(specificationImageDto.getImageResourceId(), "图片资源ID不能为空");
    Validate.notEmpty(specificationImageDto.getSpecificationId(), "规格ID不能为空");
    ProductImageResource productImageResource = productImageResourceService.findById(specificationImageDto.getImageResourceId());
    Validate.notNull(productImageResource, "图片资源不存在");
    for (String specificationId : specificationImageDto.getSpecificationId()) {
      //获取规格信息
      ProductSpecification productSpecification = this.findById(specificationId);
      Validate.notNull(productSpecification, "规格信息不存在");
      ProductImageResource oldImageResource = productSpecification.getProductImageResource();
      productSpecification.setMainImagePath(productImageResource.getRelativePath());
      productSpecification.setMainImageName(productImageResource.getFileName());
      productSpecification.setProductImageResource(productImageResource);
      productSpecificationRepository.saveAndFlush(productSpecification);
      //资源使用状态修改
      productImageResourceService.updateUseImage(oldImageResource, productImageResource);
    }
    //商品图片资源状态修改
    productService.updateImageResourceStatus(productImageResource.getProduct().getId());
  }

  @Override
  @Transactional
  public void updatePriceBatch(SpecificationPriceDto specificationPriceDto) {
    Validate.notNull(specificationPriceDto, "修改信息不能为空");
    Validate.notEmpty(specificationPriceDto.getSpcificationId(), "规格集合不能为空");

    List<ProductPriceModifyLog> productPriceModifyLogs = Lists.newArrayList();

    //批量修改规格信息
    for (String id : specificationPriceDto.getSpcificationId()) {
      //价格信息列表
      List<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrices = productUnitSpecificationAndPriceService.findBySpecification(id);
      //获取基本单位
      ProductUnitSpecificationAndPrice productUnitSpecificationAndPrice = productUnitSpecificationAndPrices.stream().filter(ProductUnitSpecificationAndPrice::getBasicUnit).findFirst().get();

      if (PriceType.SELLING_PRICE.getType().equals(specificationPriceDto.getPriceType())) {
        BigDecimal basicsPrice = productUnitSpecificationAndPrice.getSellingPrice();
        //修改销售价
        BigDecimal adjustMoney = getAdjustMoney(basicsPrice, specificationPriceDto.getAdjustMoney(), specificationPriceDto.getAdjustType());
        for (ProductUnitSpecificationAndPrice unitSpecificationAndPrice : productUnitSpecificationAndPrices) {
          //原价格
          BigDecimal oldMoney = unitSpecificationAndPrice.getSellingPrice();
          if(unitSpecificationAndPrice.getBasicUnit()){
            unitSpecificationAndPrice.setSellingPrice(adjustMoney);
          }else{
            //保留两位小数 并且4舍五入
            unitSpecificationAndPrice.setSellingPrice(adjustMoney.multiply(unitSpecificationAndPrice.getConversionRatio()).setScale(2,BigDecimal.ROUND_HALF_UP));
          }
          //日志信息拼装
          this.convertPriceModifyLog(productPriceModifyLogs,specificationPriceDto,unitSpecificationAndPrice,oldMoney, unitSpecificationAndPrice.getSellingPrice());
        }
        if(specificationPriceDto.getUpdateCustomerPrice()){
          //修改客户定价,获取规格客户定价列表
          Set<ProductPricingUnitSpecification> productPricingUnitSpecifications = productPricingUnitSpecificationService.findByProductSpecificationCode(productUnitSpecificationAndPrice.getProductSpecification().getProductSpecificationCode());
          //规格没有客户定价信息
          if(!CollectionUtils.isEmpty(productPricingUnitSpecifications)) {
            //修改客户定价信息
            for (ProductPricingUnitSpecification productPricingUnitSpecification : productPricingUnitSpecifications) {
              //原价格
              BigDecimal oldMoney = productPricingUnitSpecification.getOrderPrice();
              BigDecimal adjustCustomerMoney = this.getAdjustCustomerMoney(basicsPrice, specificationPriceDto.getAdjustMoney(), productPricingUnitSpecification.getOrderPrice(), specificationPriceDto.getAdjustType(),productPricingUnitSpecification.getUnitCode(),productUnitSpecificationAndPrices);
              productPricingUnitSpecification.setOrderPrice(adjustCustomerMoney);

              //日志信息拼装
              this.convertPriceModifyLog(productPriceModifyLogs,specificationPriceDto,productPricingUnitSpecification,oldMoney, productPricingUnitSpecification.getOrderPrice());
            }
            this.productPricingUnitSpecificationService.saveBatch(productPricingUnitSpecifications);
          }
        }

      } else {
        //修改进货价
        BigDecimal adjustMoney = getAdjustMoney(productUnitSpecificationAndPrice.getReferencePurchasePrice(), specificationPriceDto.getAdjustMoney(), specificationPriceDto.getAdjustType());
        for (ProductUnitSpecificationAndPrice unitSpecificationAndPrice : productUnitSpecificationAndPrices) {
          //原价格
          BigDecimal oldMoney = unitSpecificationAndPrice.getReferencePurchasePrice();
          if(unitSpecificationAndPrice.getBasicUnit()){
            unitSpecificationAndPrice.setReferencePurchasePrice(adjustMoney);
          }else{
            //保留两位小数 并且4舍五入
            unitSpecificationAndPrice.setReferencePurchasePrice(adjustMoney.multiply(unitSpecificationAndPrice.getConversionRatio()).setScale(2,BigDecimal.ROUND_HALF_UP));
          }
          //日志信息拼装
          this.convertPriceModifyLog(productPriceModifyLogs,specificationPriceDto,unitSpecificationAndPrice,oldMoney, unitSpecificationAndPrice.getReferencePurchasePrice());
        }
      }
      //批量保存价格修改
      productUnitSpecificationAndPriceService.saveBatch(productUnitSpecificationAndPrices);

    }
    //批量保存修改日志
    productPriceModifyLogService.saveBatch(productPriceModifyLogs);
  }

  /**
   * 价格修改日志组装
   * @param list
   * @param specificationPriceDto
   * @param productPricingUnitSpecification
   * @param oldMoney
   * @param afterMoney
   */
  private void convertPriceModifyLog(List<ProductPriceModifyLog> list,SpecificationPriceDto specificationPriceDto,ProductPricingUnitSpecification productPricingUnitSpecification,BigDecimal oldMoney,BigDecimal afterMoney){

    Validate.notNull(specificationPriceDto,"价格修改参数不能为空");
    Validate.notNull(productPricingUnitSpecification,"价格信息不能为空");
    Validate.notNull(oldMoney,"原本价格不能为空");
    Validate.notNull(afterMoney,"修改后金额不能为空");

    ProductPriceModifyLog productPriceModifyLog = new ProductPriceModifyLog();
    productPriceModifyLog.setProductCode(productPricingUnitSpecification.getProductPricing().getProduct().getProductCode());
    productPriceModifyLog.setProductName(productPricingUnitSpecification.getProductPricing().getProduct().getProductName());
    productPriceModifyLog.setProductSpecificationCode(productPricingUnitSpecification.getProductSpecificationCode());
    productPriceModifyLog.setProductSpecificationName(productPricingUnitSpecification.getProductSpecificationName());
    productPriceModifyLog.setUnitCode(productPricingUnitSpecification.getUnitCode());
    productPriceModifyLog.setUnitName(productPricingUnitSpecification.getUnitName());
    productPriceModifyLog.setCustomerCode(productPricingUnitSpecification.getProductPricing().getCustomerCode());
    productPriceModifyLog.setCustomerName(productPricingUnitSpecification.getProductPricing().getCustomerName());
    productPriceModifyLog.setPriceType(specificationPriceDto.getPriceType());
    productPriceModifyLog.setAdjustMoney(specificationPriceDto.getAdjustMoney());
    productPriceModifyLog.setAdjustType(specificationPriceDto.getAdjustType());
    productPriceModifyLog.setOldMoney(oldMoney);
    productPriceModifyLog.setAfterMoney(afterMoney);
    Date now = new Date();
    productPriceModifyLog.setCreateAccount(SecurityUtils.getUserAccount());
    productPriceModifyLog.setCreateTime(now);
    productPriceModifyLog.setModifyAccount(SecurityUtils.getUserAccount());
    productPriceModifyLog.setModifyTime(now);
    productPriceModifyLog.setTenantCode(TenantUtils.getTenantCode());

    list.add(productPriceModifyLog);
  }

  /**
   * 价格修改日志组装
   * @param list
   * @param specificationPriceDto
   * @param productUnitSpecificationAndPrice
   * @param oldMoney
   * @param afterMoney
   */
  private void convertPriceModifyLog(List<ProductPriceModifyLog> list,SpecificationPriceDto specificationPriceDto,ProductUnitSpecificationAndPrice productUnitSpecificationAndPrice,BigDecimal oldMoney,BigDecimal afterMoney){

    Validate.notNull(specificationPriceDto,"价格修改参数不能为空");
    Validate.notNull(productUnitSpecificationAndPrice,"价格信息不能为空");
    Validate.notNull(oldMoney,"原本价格不能为空");
    Validate.notNull(afterMoney,"修改后金额不能为空");

    ProductPriceModifyLog productPriceModifyLog = new ProductPriceModifyLog();
    productPriceModifyLog.setProductCode(productUnitSpecificationAndPrice.getProduct().getProductCode());
    productPriceModifyLog.setProductName(productUnitSpecificationAndPrice.getProduct().getProductName());
    productPriceModifyLog.setProductSpecificationCode(productUnitSpecificationAndPrice.getProductSpecification().getProductSpecificationCode());
    productPriceModifyLog.setProductSpecificationName(productUnitSpecificationAndPrice.getProductSpecification().getProductSpecificationName());
    productPriceModifyLog.setUnitCode(productUnitSpecificationAndPrice.getProductUnit().getUnitCode());
    productPriceModifyLog.setUnitName(productUnitSpecificationAndPrice.getProductUnit().getUnitName());
    productPriceModifyLog.setPriceType(specificationPriceDto.getPriceType());
    productPriceModifyLog.setAdjustMoney(specificationPriceDto.getAdjustMoney());
    productPriceModifyLog.setAdjustType(specificationPriceDto.getAdjustType());
    productPriceModifyLog.setOldMoney(oldMoney);
    productPriceModifyLog.setAfterMoney(afterMoney);
    Date now = new Date();
    productPriceModifyLog.setCreateAccount(SecurityUtils.getUserAccount());
    productPriceModifyLog.setCreateTime(now);
    productPriceModifyLog.setModifyAccount(SecurityUtils.getUserAccount());
    productPriceModifyLog.setModifyTime(now);
    productPriceModifyLog.setTenantCode(TenantUtils.getTenantCode());

    list.add(productPriceModifyLog);
  }


  /**
   * 获取调整后的金额
   *
   * @param oldMoney    规格原金额
   * @param adjustMoney 调整金额
   * @param adjustType  调整类型
   * @return
   */
  private BigDecimal getAdjustMoney(BigDecimal oldMoney, BigDecimal adjustMoney, Integer adjustType) {
    Validate.notNull(oldMoney,"原本金额不能为空");
    Validate.notNull(adjustMoney,"调整金额不能为空");
    Validate.notNull(adjustType,"调整类型不能为空");
    AdjustType type = AdjustType.getValue(adjustType);
    Validate.notNull(type,"调整类型不存在");
    BigDecimal money = BigDecimal.ZERO;
    switch (type) {
      case MONEY_ADD:
        //金额增加
        money = oldMoney.add(adjustMoney);
        break;
      case SCALE_ADD:
        BigDecimal add = oldMoney.multiply(adjustMoney);
        money = oldMoney.add(add);
        break;
      case MONEY_REDUCE:
        money = oldMoney.subtract(adjustMoney);
        break;
      case SCALE_REDUCE:
        BigDecimal reduce = oldMoney.multiply(adjustMoney);
        money = oldMoney.subtract(reduce);
        break;
    }
    //这里保留两位小数，并且多余的位数四舍五入。
    money = money.setScale(2,BigDecimal.ROUND_HALF_UP);
    Validate.isTrue(money.compareTo(UPDATE_MIN_PRICE)>=0,"调整后金额不能小于等于0");

    return money;
  }


  private BigDecimal getAdjustCustomerMoney(BigDecimal oldMoney, BigDecimal adjustMoney,BigDecimal customerMoney, Integer adjustType,String unitCode,List<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrices) {
    Validate.notNull(oldMoney,"原本金额不能为空");
    Validate.notNull(adjustMoney,"调整金额不能为空");
    Validate.notNull(adjustType,"调整类型不能为空");
    Validate.notNull(customerMoney,"客户金额不能为空");

    Map<String, ProductUnitSpecificationAndPrice> collect = productUnitSpecificationAndPrices.stream().collect(Collectors.toMap(e->e.getProductUnit().getUnitCode(), Function.identity()));
    //增加金额*换算比例
    BigDecimal conversionRatio = collect.get(unitCode).getConversionRatio();
    adjustMoney = adjustMoney.multiply(conversionRatio);

    AdjustType type = AdjustType.getValue(adjustType);
    Validate.notNull(type,"调整类型不存在");
    BigDecimal money = BigDecimal.ZERO;
    switch (type) {
      case MONEY_ADD:
        //金额增加
        money = customerMoney.add(adjustMoney);
        break;
      case SCALE_ADD:
        BigDecimal add = oldMoney.multiply(adjustMoney);
        money = customerMoney.add(add);
        break;
      case MONEY_REDUCE:
        money = customerMoney.subtract(adjustMoney);
        break;
      case SCALE_REDUCE:
        BigDecimal reduce = oldMoney.multiply(adjustMoney);
        money = customerMoney.subtract(reduce);
        break;
    }
    //这里保留两位小数，并且多余的位数四舍五入。
    money = money.setScale(2,BigDecimal.ROUND_HALF_UP);
    Validate.isTrue(money.compareTo(UPDATE_MIN_PRICE)>=0,"调整后金额不能小于等于0");
    return money;
  }

  @Override
  @Transactional
  public void updateCustomerPrice(ProductSpecification productSpecification) {
    Validate.notNull(productSpecification,"规格信息不能为空");
    Validate.notBlank(productSpecification.getId(),"规格ID不能为空");
    Validate.notEmpty(productSpecification.getProductPricings(),"客户定价不能为空");
    //获取规格信息
    ProductSpecification specification = this.findById(productSpecification.getId());
    //修改价格信息
    productPricingService.updateFormBatch(productSpecification.getProductPricings(),specification.getProduct(),specification);
  }
}
