package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.product.common.aspect.LabelPreferentialItems;
import com.bizunited.empower.business.product.dto.BrandAndUnitMapBarCodeDto;
import com.bizunited.empower.business.product.dto.DiscountResultDto;
import com.bizunited.empower.business.product.dto.ProductSpecificationOrderDto;
import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.empower.business.product.entity.ProductBarCodeInfo;
import com.bizunited.empower.business.product.entity.ProductBrand;
import com.bizunited.empower.business.product.entity.ProductCategory;
import com.bizunited.empower.business.product.entity.ProductSpecification;
import com.bizunited.empower.business.product.entity.ProductUnitAndPrice;
import com.bizunited.empower.business.product.entity.ProductUnitSpecificationAndPrice;
import com.bizunited.empower.business.product.repository.ProductCategoryRepository;
import com.bizunited.empower.business.product.repository.ProductRepository;
import com.bizunited.empower.business.product.repository.ProductSpecificationRepository;
import com.bizunited.empower.business.product.service.ProductActionService;
import com.bizunited.empower.business.product.service.ProductSpecificationVoService;
import com.bizunited.empower.business.product.service.handler.DiscountHandler;
import com.bizunited.empower.business.product.service.notifier.ProductCustomerBuyableEventListener;
import com.bizunited.empower.business.product.service.notifier.ProductWarehouseEventListener;
import com.bizunited.empower.business.product.vo.ProductSpecificationVo;
import com.bizunited.empower.business.product.vo.ProductUnitSpecificationAndPriceVo;
import com.bizunited.empower.business.product.vo.ProductUnitVo;
import com.bizunited.empower.business.product.vo.ProductVo;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.script.context.InvokeParams;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service("ProductSpecificationVoServiceImpl")
public class ProductSpecificationVoServiceImpl implements ProductSpecificationVoService {

  @Autowired
  ProductSpecificationRepository productSpecificationRepository;
  @Autowired
  ProductRepository productRepository;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  @Qualifier("VirtualHeadDiscountHandler")
  DiscountHandler discountHandler;

  @Autowired(required = false)
  private ProductWarehouseEventListener productWarehouseEventListener;

  @Autowired
  ProductActionService productActionService;

  @Autowired
  private ProductCategoryRepository productCategoryRepository;

  @Autowired
  private RedisMutexService redisMutexService;

  @Autowired(required = false)
  private ProductCustomerBuyableEventListener productCustomerBuyableEventListener;



  @LabelPreferentialItems
  @Override
  public ProductSpecificationVo findBySpecificationCode(String specificationCode) {
    ProductSpecification ps = productSpecificationRepository.findByTenantCodeAndProductSpecificationCode(TenantUtils.getTenantCode(), specificationCode);
    if (Objects.isNull(ps)) {
      return null;
    }
    Product product = ps.getProduct();
    Validate.notNull(product, "该规格没有上层商品，属于非法数据！");
    ProductSpecificationVo productSpecificationVo = this.nebulaToolkitService.copyObjectByWhiteList(ps, ProductSpecificationVo.class, HashSet.class, ArrayList.class,
        "product", "productBarCodeInfos");
    ProductVo productVo = nebulaToolkitService.copyObjectByWhiteList(product, ProductVo.class, HashSet.class, ArrayList.class,
        "productCategory", "productBrand", "productFiles", "productSpecifications", "productMultipleSpecifications", "tags", "productPricings", "productUnitAndPrices");
    productSpecificationVo.setProduct(productVo);
    return productSpecificationVo;
  }

  /**
   * 分页查询规格信息
   *
   * @param pageable
   * @param conditions
   * @return
   */
  @LabelPreferentialItems("#conditions.getInvokeParam('customerCode')")
  @Override
  public Page<ProductSpecificationVo> findByConditions(Pageable pageable, InvokeParams conditions) {

    /**
     * 把单位与价格表（ProductUnitAndPrice）以外的，所有有关与价格的操作，都视为打折或优惠。
     * 并将最终价格覆盖在单位与价格表（ProductUnitAndPrice）中的销售价格（sellingPrice）
     */
    conditions.putInvokeParam("tenantCode", TenantUtils.getTenantCode());
    Map<String, Object> params = conditions.getInvokeParams();
    String customerCode = (String) params.get("customerCode");
    // 查询数据库
    //这里增加筛选， 可以筛选库存信息大于0 的数据
    //当 存在客户可购清单列表时， 在客户可购清单列表中去筛选大于0的数据，
    //不存在客户可购清单列表时，筛选所有的数据
    String available = (String) params.get("available");
    if (StringUtils.isNotBlank(available) && Boolean.parseBoolean(available)) {
      String productCodeList = (String) params.get("productCodeList");
      List<String> specificationCodes = Lists.newLinkedList();
      if (StringUtils.isNotBlank(productCodeList)) {
        specificationCodes = Lists.newArrayList(productCodeList.split(","));
      }
      if (productWarehouseEventListener != null) {
        specificationCodes = productWarehouseEventListener.findAvailableByProductSpecificationCodes(specificationCodes,customerCode);
        if(CollectionUtils.isEmpty(specificationCodes)){
          //如果返回库存大于0的规格编码列表为空的情况
          //那么直接返回
          return new PageImpl<>(Lists.newArrayList(),pageable,0);
        }
      }
      //替换商品规格编码集合参数
      conditions.putInvokeParam("productCodeList", StringUtils.join(specificationCodes, ","));
    }
    String warehosueCode = (String) params.get("warehosueCode");
    if (StringUtils.isNotBlank(warehosueCode)) {
      String productCodeList = (String) params.get("productCodeList");
      List<String> specificationCodes = Lists.newLinkedList();
      if (StringUtils.isNotBlank(productCodeList)) {
        specificationCodes = Lists.newArrayList(productCodeList.split(","));
      }
      if (productWarehouseEventListener != null) {
        specificationCodes = productWarehouseEventListener.findPracticalByProductSpecificationCodes(specificationCodes,warehosueCode);
        if(CollectionUtils.isEmpty(specificationCodes)){
          //如果返回实际库存大于0的规格编码列表为空的情况
          //那么直接返回
          return new PageImpl<>(Lists.newArrayList(),pageable,0);
        }
      }
      //替换商品规格编码集合参数
      conditions.putInvokeParam("productCodeList", StringUtils.join(specificationCodes, ","));
    }

    Page<ProductSpecification> page = productSpecificationRepository.queryPage(pageable, conditions);
    List<ProductSpecification> content = page.getContent();
    Map<String, List<DiscountResultDto>> puMapDiscountResult = new HashMap<>();
    if (StringUtils.isNotBlank(customerCode)) {
      // 获取打折信息
      List<DiscountResultDto> discountResultDtoList = this.discountedEntry(content, conditions);
      List<DiscountResultDto> finalPrice = productActionService.findProductFinalPrice(discountResultDtoList);
      puMapDiscountResult = finalPrice.stream().collect(Collectors.groupingBy(discountResultDto -> {
        String productCode = discountResultDto.getProductCode();
        String unitCode = discountResultDto.getUnitCode();
        String productSpecificationCode = discountResultDto.getProductSpecificationCode();
        return StringUtils.join(productCode, unitCode, productSpecificationCode);
      }));
    }
    //------数据拷贝------------
    List<ProductSpecificationVo> productSpecificationVos = new ArrayList<>(content.size());
    for (ProductSpecification specification : content) {
      ProductSpecificationVo productSpecificationVo = this.nebulaToolkitService.copyObjectByWhiteList(specification, ProductSpecificationVo.class, HashSet.class, ArrayList.class,
          "product", "productBarCodeInfos","productPricings","productPricings.productPricingUnitSpecifications");
      Product product = specification.getProduct();
      Validate.notNull(product, "数据库信息不正确，规格无商品归属");
      ProductVo productVo = this.nebulaToolkitService.copyObjectByWhiteList(product, ProductVo.class, HashSet.class, ArrayList.class,
          "productCategory", "productBrand", "productFiles", "productSpecifications", "productMultipleSpecifications", "tags", "productPricings", "productUnitAndPrices", "productUnitAndPrices.productUnit");
      Set<ProductUnitSpecificationAndPrice> productUnitAndPrices = product.getProductUnitSpecificationAndPrices();
      //这里拆分价格信息为当前规格的价格信息
      productUnitAndPrices = splitPriceBySpecificationCode(productUnitAndPrices, specification.getProductSpecificationCode());
      Set<ProductUnitSpecificationAndPriceVo> productUnitAndPriceVoSet = (Set<ProductUnitSpecificationAndPriceVo>) this.nebulaToolkitService.copyCollectionByWhiteList(productUnitAndPrices, ProductUnitSpecificationAndPrice.class, ProductUnitSpecificationAndPriceVo.class, HashSet.class, ArrayList.class,
          "productUnit", "product", "productSpecification");
      productVo.setProductUnitSpecificationAndPrices(productUnitAndPriceVoSet);
      productSpecificationVo.setProduct(productVo);
      productSpecificationVos.add(productSpecificationVo);
      //------------覆盖打折结果-------------------
      if (StringUtils.isNotBlank(customerCode)) {
        this.coverDiscountResults(puMapDiscountResult, productUnitAndPriceVoSet);
      }
      //-------------------------------------------------
    }
    //丰富库存信息
    if (productWarehouseEventListener != null) {
      String warehouseCode = (String) params.get("warehouseCode");
      if (StringUtils.isNotBlank(warehouseCode)){
        //如果库存编号不为空
        productWarehouseEventListener.findProductSpecificationWarehouseByWarehouseCode(productSpecificationVos,warehouseCode);
      }else {
        productWarehouseEventListener.findProductSpecificationWarehouse(productSpecificationVos,customerCode);
      }
    }
    return new PageImpl<>(Lists.newArrayList(productSpecificationVos), pageable, page.getTotalElements());
  }

  /**
   * 拆分价格信息
   *
   * @param specificationAndPrices
   * @return
   */
  private Set<ProductUnitSpecificationAndPrice> splitPriceBySpecificationCode(Set<ProductUnitSpecificationAndPrice> specificationAndPrices, String specificationCode) {
    //只保留当前规格的价格信息
    Set<ProductUnitSpecificationAndPrice> splitSpecificationAndPrice = Sets.newHashSet();
    for (ProductUnitSpecificationAndPrice specificationAndPrice : specificationAndPrices) {
      if (specificationAndPrice.getProductSpecification().getProductSpecificationCode().equals(specificationCode)) {
        splitSpecificationAndPrice.add(specificationAndPrice);
      }
    }
    return splitSpecificationAndPrice;
  }

  /**
   * 构造获取最终打折结果的入参
   *
   * @param content
   * @param conditions
   * @return
   */
  private List<DiscountResultDto> discountedEntry(List<ProductSpecification> content, InvokeParams conditions) {
    Map<String, Object> params = conditions.getInvokeParams();
    String customerCode = (String) params.get("customerCode");

    List<DiscountResultDto> discountResultDtos = new ArrayList<>();
    for (ProductSpecification specification : content) {
      DiscountResultDto dto = new DiscountResultDto();
      dto.setCustomerCode(customerCode);
      dto.setProductCode(specification.getProduct().getProductCode());
      discountResultDtos.add(dto);
    }
    return discountResultDtos;
  }

  /**
   * 把打折结果覆盖在交易价格上
   *
   * @param puMapDiscountResult
   * @param productUnitAndPriceVoSet
   */
  private void coverDiscountResults(Map<String, List<DiscountResultDto>> puMapDiscountResult, Set<ProductUnitSpecificationAndPriceVo> productUnitAndPriceVoSet) {
    for (ProductUnitSpecificationAndPriceVo unitAndPriceVo : productUnitAndPriceVoSet) {
      String productCode = unitAndPriceVo.getProduct().getProductCode();
      String unitCode = unitAndPriceVo.getProductUnit().getUnitCode();
      ProductSpecificationVo productSpecification = unitAndPriceVo.getProductSpecification();
      String specificationCode = productSpecification.getProductSpecificationCode();
      String join = StringUtils.join(productCode, unitCode, specificationCode);
      List<DiscountResultDto> resultDtos = puMapDiscountResult.get(join);
      if (!CollectionUtils.isEmpty(resultDtos)) {
        BigDecimal afterDiscountPrice = resultDtos.get(0).getAfterDiscountPrice();
        unitAndPriceVo.setSellingPrice(afterDiscountPrice);
      }
    }
  }

  @Override
  public List<String> findAllSpecificationCode() {
    return this.productSpecificationRepository.findProductSpecificationCodeByTenantCode(TenantUtils.getTenantCode());
  }

  @Override
  public List<ProductSpecificationVo> findBySpecificationCodeList(List<String> specificationCodes) {
    Validate.isTrue(!CollectionUtils.isEmpty(specificationCodes), "规格集合必传");
    List<ProductSpecification> pss = productSpecificationRepository.findByTenantCodeAndProductSpecificationCodeIn(TenantUtils.getTenantCode(), specificationCodes);
    List<ProductSpecificationVo> productSpecificationVos = new ArrayList<>(pss.size());
    for (ProductSpecification ps : pss) {
      Product product = ps.getProduct();
      ProductSpecificationVo productSpecificationVo = this.nebulaToolkitService.copyObjectByWhiteList(ps, ProductSpecificationVo.class, HashSet.class, ArrayList.class,
          "productBarCodeInfos","productUnitSpecificationAndPrices","productUnitSpecificationAndPrices.productUnit","productUnitSpecificationAndPrices.productSpecification");
      ProductVo productVo = nebulaToolkitService.copyObjectByWhiteList(product, ProductVo.class, HashSet.class, ArrayList.class,
          "productUnitAndPrices", "productUnitAndPrices.productUnit", "productUnitSpecificationAndPrices", "productUnitSpecificationAndPrices.productUnit","productUnitSpecificationAndPrices.productSpecification", "productBrand","productCategory");
      productSpecificationVo.setProduct(productVo);
      productSpecificationVos.add(productSpecificationVo);
    }
    return productSpecificationVos;
  }

  @Override
  public List<ProductSpecificationVo> findBySpecificationCodesAndCustomerCode(List<String> specificationCodes, String customerCode) {
    if (CollectionUtils.isEmpty(specificationCodes)){
      return Lists.newArrayList();
    }
    List<ProductSpecificationVo> productSpecificationVoList = this.findBySpecificationCodeList(specificationCodes);
    if (productWarehouseEventListener != null) {
      productWarehouseEventListener.findProductSpecificationWarehouse(productSpecificationVoList,customerCode);
    }
    return productSpecificationVoList;
  }

  @Override
  public void validateOrderQuantity(List<ProductSpecificationOrderDto> productSpecificationOrderDtoList) {
    Validate.isTrue(!CollectionUtils.isEmpty(productSpecificationOrderDtoList), "入参集合不能为空");
    List<String> specificationCodes = productSpecificationOrderDtoList.stream().map(ProductSpecificationOrderDto::getProductSpecificationCode).collect(Collectors.toList());
    Validate.isTrue(!CollectionUtils.isEmpty(specificationCodes), "规格集合必传");
    List<ProductSpecification> pss = productSpecificationRepository.findByTenantCodeAndProductSpecificationCodeIn(TenantUtils.getTenantCode(), specificationCodes);
    Map<String, List<ProductSpecification>> specificationCodeMap = pss.stream().collect(Collectors.groupingBy(ProductSpecification::getProductSpecificationCode));
    for (ProductSpecificationOrderDto dto : productSpecificationOrderDtoList) {
      String productSpecificationCode = dto.getProductSpecificationCode();
      List<ProductSpecification> list = specificationCodeMap.get(productSpecificationCode);
      Validate.isTrue(!CollectionUtils.isEmpty(list), "入参中【%s】，数据库中不存在", productSpecificationCode);
      ProductSpecification specification = list.get(0);
      // 限定
      BigDecimal maximumOrderQuantity = specification.getMaximumOrderQuantity();
      // 起定
      BigDecimal minimumOrderQuantity = specification.getMinimumOrderQuantity();

      // 入参转基本
      BigDecimal quantity = this.conversionUnit(dto, specification);
      if (Objects.nonNull(minimumOrderQuantity)) {
        Validate.isTrue(quantity.compareTo(minimumOrderQuantity) > -1, "【%s】【%s】，订货量少于起订量%s", specification.getProduct().getProductName(), specification.getProductSpecificationName(), minimumOrderQuantity.stripTrailingZeros().toPlainString());
      }
      if (Objects.nonNull(maximumOrderQuantity)) {
        Validate.isTrue(quantity.compareTo(maximumOrderQuantity) <= 0, "【%s】【%s】，订货量超过限定量%s", specification.getProduct().getProductName(), specification.getProductSpecificationName(), maximumOrderQuantity.stripTrailingZeros().toPlainString());
      }
    }
  }

  /**
   * TODO 修改 转换率 / 单位，记得修改redis的内容
   *
   * @param productSpecificationCode
   * @param unitCode
   * @return
   */
  @Override
  public BigDecimal findConvertRate(String productSpecificationCode, String unitCode) {
    BigDecimal conversionRatio = BigDecimal.ZERO;
    Validate.notNull(productSpecificationCode, "出库商品规格必填！");
    Validate.notNull(unitCode, "出库商品规格单位必填！");
    // 从redis内直接拿到了转换率
    String mapName = StringUtils.join(TenantUtils.getTenantCode(), productSpecificationCode);
    String tenantSpecificationUnit = StringUtils.join(mapName, unitCode);
    String conversionRatioStr = this.redisMutexService.getMCode(mapName, tenantSpecificationUnit);
    if (StringUtils.isNotBlank(conversionRatioStr)) {
      // 直接获取到了 单位转换率
      conversionRatio = new BigDecimal(conversionRatioStr);
      Validate.notNull(conversionRatio, "商品规格【%s】单位未指定转换率", productSpecificationCode);
    } else {
      List<ProductUnitSpecificationAndPrice> productUnitAndPricesList = this.getProductUnitAndPrices(productSpecificationCode);
      Validate.isTrue(!CollectionUtils.isEmpty(productUnitAndPricesList), "出库商品规格【%s】,上层商品未指定单位", productSpecificationCode);
      Boolean exist = false;
      for (ProductUnitSpecificationAndPrice unitAndPrice : productUnitAndPricesList) {
        conversionRatio = unitAndPrice.getConversionRatio();
        // 将 单位转换率 放入 redis
        long randomMilliseconds = this.getRandomRange(10, 5);
        this.redisMutexService.setMCode(mapName, tenantSpecificationUnit, conversionRatio.toString(), randomMilliseconds);
        if (unitCode.equals(unitAndPrice.getProductUnit().getUnitCode()) && unitAndPrice.getProductSpecification().getProductSpecificationCode().equals(productSpecificationCode)) {
          exist = true;
          Validate.notNull(conversionRatio, "商品规格【%s】单位未指定转换率", productSpecificationCode);
          break;
        }
      }
      Validate.isTrue(exist, "入参单位，在商品规格【%s】的上层商品中不存在", productSpecificationCode);
    }
    Validate.isTrue(BigDecimal.ZERO.compareTo(conversionRatio) < 1, "转换率非法，小于或等于0");
    return conversionRatio;
  }

  @Override
  public List<Pair<String, String>> findSpecAndBasicUnitByBrandCodeList(List<String> brandCodeList) {
    if (CollectionUtils.isEmpty(brandCodeList)) {
      return Lists.newLinkedList();
    }
    Set<Product> products = productRepository.findByProductBrandCodeList(brandCodeList, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(products)) {
      return Lists.newLinkedList();
    }
    List<Pair<String, String>> pairs = this.collectionSpecifications(products);
    return pairs;
  }

  private List<Pair<String, String>> collectionSpecifications(Set<Product> products) {
    List<Pair<String, String>> pairs = new ArrayList<>();
    for (Product product : products) {
      Set<ProductSpecification> specifications = product.getProductSpecifications();
      Set<ProductUnitSpecificationAndPrice> productUnitAndPrices = product.getProductUnitSpecificationAndPrices();
      // 获取 基本单位
      String basicUnitCode = "";
      for (ProductUnitSpecificationAndPrice unitAndPrice : productUnitAndPrices) {
        if (unitAndPrice.getBasicUnit()) {
          basicUnitCode = unitAndPrice.getProductUnit().getUnitCode();
        }
      }
      // 收集规格
      for (ProductSpecification specification : specifications) {
        Pair<String, String> pair = new ImmutablePair<>(specification.getProductSpecificationCode(), basicUnitCode);
        pairs.add(pair);
      }
    }
    return pairs;
  }

  @Override
  public List<Pair<String, String>> findSpecAndBasicUnitByCategoryCodeList(List<String> categoryCodeList) {
    if (CollectionUtils.isEmpty(categoryCodeList)) {
      return Lists.newLinkedList();
    }
    Set<Product> products = productRepository.findByProductCategoryCodeList(categoryCodeList, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(products)) {
      return Lists.newLinkedList();
    }
    List<Pair<String, String>> pairs = this.collectionSpecifications(products);
    return pairs;
  }

  @Override
  public List<ProductSpecificationVo> findByBrandCodeList(List<String> brandCodeList) {
    if (CollectionUtils.isEmpty(brandCodeList)) {
      return Lists.newLinkedList();
    }
    Set<Product> products = productRepository.findByProductBrandCodeList(brandCodeList, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(products)) {
      return Lists.newLinkedList();
    }
    return this.getProductSpecificationVos(products);
  }

  private List<ProductSpecificationVo> getProductSpecificationVos(Set<Product> products) {
    if (CollectionUtils.isEmpty(products)) {
      return Lists.newArrayList();
    }
    List<ProductSpecificationVo> productSpecificationVos = new ArrayList<>();
    for (Product product : products) {
      Set<ProductSpecification> productSpecifications = product.getProductSpecifications();
      if (CollectionUtils.isEmpty(productSpecifications)) {
        continue;
      }
      Set<ProductSpecificationVo> productSpecificationVo = (Set<ProductSpecificationVo>) this.nebulaToolkitService.copyCollectionByWhiteList(productSpecifications, ProductSpecification.class, ProductSpecificationVo.class, HashSet.class, ArrayList.class,
          "productBarCodeInfos");
      ProductVo productVo = nebulaToolkitService.copyObjectByWhiteList(product, ProductVo.class, HashSet.class, ArrayList.class,
          "productUnitAndPrices", "productUnitAndPrices.productUnit", "productBrand", "productUnitSpecificationAndPrices", "productUnitSpecificationAndPrices.productUnit");
      ProductUnitVo basicUnit = null;
      if (!CollectionUtils.isEmpty(productVo.getProductUnitSpecificationAndPrices())) {
        for (ProductUnitSpecificationAndPriceVo unitAndPrice : productVo.getProductUnitSpecificationAndPrices()) {
          if (unitAndPrice.getBasicUnit()) {
            basicUnit = unitAndPrice.getProductUnit();
          }
        }
      }
      for (ProductSpecificationVo vo : productSpecificationVo) {
        vo.setProduct(productVo);
        vo.setBasicUnit(basicUnit);
      }
      productSpecificationVos.addAll(productSpecificationVo);
    }
    return productSpecificationVos;
  }

  @Override
  public List<ProductSpecificationVo> findByCategoryCodeList(List<String> categoryCodeList) {
    if (CollectionUtils.isEmpty(categoryCodeList)) {
      return Lists.newLinkedList();
    }
    //通过规格编码获取所有的规格编码信息
    List<ProductCategory> productCategories = productCategoryRepository.findByTenantCodeAndCodeIn(TenantUtils.getTenantCode(), categoryCodeList);
    if (CollectionUtils.isEmpty(productCategories)) {
      return Lists.newLinkedList();
    }
    //根据获取到的分类信息 通过like% flatCode 的方式 查询当前分类及子集的商品信息
    Set<Product> productSets = Sets.newHashSet();
    for (ProductCategory productCategory : productCategories) {
      Set<Product> productSet = this.productRepository.findByProductCategoryFlatCodeAndTenantCode(productCategory.getFlatCode(),TenantUtils.getTenantCode());
      productSets.addAll(productSet);
    }
    return this.getProductSpecificationVos(productSets);
  }

  @Override
  public BrandAndUnitMapBarCodeDto findBrandAndUnitMapBarCodeBySpecCodes(List<String> specificationCodes) {
    BrandAndUnitMapBarCodeDto external = new BrandAndUnitMapBarCodeDto();
    if (CollectionUtils.isEmpty(specificationCodes)) {
      return external;
    }
    List<ProductSpecification> specificationList = this.productSpecificationRepository.findByTenantCodeAndProductSpecificationCodeIn(TenantUtils.getTenantCode(), specificationCodes);
    for (ProductSpecification specification : specificationList) {
      Product product = specification.getProduct();
      ProductBrand productBrand = product.getProductBrand();
      //这里商品规格条码允许为空
      Set<ProductBarCodeInfo> productBarCodeInfos = specification.getProductBarCodeInfos();
      Map<String, ProductBarCodeInfo> productBarCodeInfoMap = new HashMap<>();
      if(!CollectionUtils.isEmpty(productBarCodeInfos)){
        productBarCodeInfoMap = productBarCodeInfos.stream().collect(Collectors.toMap(ProductBarCodeInfo::getUnitCode, Function.identity()));
      }
      //通过商品单位价格信息 获取到商品全单位信息
      Set<ProductUnitAndPrice> productUnitAndPrices = product.getProductUnitAndPrices();
      for (ProductUnitAndPrice productUnitAndPrice : productUnitAndPrices) {
        BrandAndUnitMapBarCodeDto internal = new BrandAndUnitMapBarCodeDto();
        internal.setBrandCode(productBrand.getBrandCode());
        internal.setBrandName(productBrand.getBrandName());
        //通过商品单位获取规格下对应的条码信息 可能为空
        if(productBarCodeInfoMap.containsKey(productUnitAndPrice.getProductUnit().getUnitCode())){
          internal.setBarCode(productBarCodeInfoMap.get(productUnitAndPrice.getProductUnit().getUnitCode()).getBarCode());
        }
        internal.setProductSpecificationCode(specification.getProductSpecificationCode());
        internal.setProductSpecificationName(specification.getProductSpecificationName());
        internal.setUnitCode(productUnitAndPrice.getProductUnit().getUnitCode());
        external.setBrandAndUnitMapBarCodeDto(internal);
      }

    }
    return external;
  }

  /**
   * 获取 该规格的单位集合
   *
   * @param productSpecificationCode
   * @return
   */
  private List<ProductUnitSpecificationAndPrice> getProductUnitAndPrices(String productSpecificationCode) {
    List<ProductUnitSpecificationAndPrice> productUnitAndPricesList;
    ProductSpecification ps = productSpecificationRepository.findByTenantCodeAndProductSpecificationCode(TenantUtils.getTenantCode(), productSpecificationCode);
    Validate.notNull(ps, "出库商品规格【%s】不存在", productSpecificationCode);
    Validate.notNull(ps.getProduct(), "出库商品规格【%s】,上层商品不存在", productSpecificationCode);
    Set<ProductUnitSpecificationAndPrice> productUnitAndPrices = ps.getProduct().getProductUnitSpecificationAndPrices();
    productUnitAndPricesList = productUnitAndPrices.stream().collect(Collectors.toList());
    return productUnitAndPricesList;
  }

  private long getRandomRange(int max, int min) {
    int minute = 60000;
    max = max * minute;
    min = min * minute;
    return (long) (Math.random() * (max - min) + min);
  }

  /**
   * 将入参单位的出库值转为基本单位数量
   *
   * @param dto
   * @return
   */
  private BigDecimal conversionUnit(ProductSpecificationOrderDto dto, ProductSpecification specification) {
    // 入参数量转基本单位数量
    Validate.notNull(specification, "出库商品规格【%s】不存在", dto.getProductSpecificationCode());
    Validate.notNull(specification.getProduct(), "出库商品规格【%s】,上层商品不存在", specification.getProductSpecificationName());
    BigDecimal quantity = dto.getOrderQuantity();
    Validate.notNull(quantity, "出库商品规格【%s】【出库数量】必填！", specification.getProductSpecificationName());
    Validate.notNull(dto.getUnitCode(), "出库商品规格【%s】【单位】必填！", specification.getProductSpecificationName());
    Set<ProductUnitSpecificationAndPrice> productUnitAndPrices = specification.getProduct().getProductUnitSpecificationAndPrices();
    Validate.isTrue(!CollectionUtils.isEmpty(productUnitAndPrices), "出库商品规格【%s】,上层商品【%s】未指定单位", specification.getProductSpecificationName(), specification.getProduct().getProductName());
    Boolean exist = false;
    for (ProductUnitSpecificationAndPrice unitAndPrice : productUnitAndPrices) {
      if (dto.getUnitCode().equals(unitAndPrice.getProductUnit().getUnitCode()) && dto.getProductSpecificationCode().equals(unitAndPrice.getProductSpecification().getProductSpecificationCode())) {
        exist = true;
        BigDecimal conversionRatio = unitAndPrice.getConversionRatio();
        Validate.notNull(conversionRatio, "商品规格【%s】单位未指定转换率", dto.getProductSpecificationCode());
        quantity = quantity.multiply(conversionRatio).setScale(4, RoundingMode.HALF_UP);
      }
    }
    Validate.isTrue(exist, "入参单位，在商品规格【%s】的上层商品中不存在", dto.getProductSpecificationCode());
    String quantityStr = quantity.toString();
    Validate.isTrue(quantityStr.endsWith("000"), "出库数量转基本单位数量时有小数！");
    return quantity;
  }

  /**
   * 该私有方法用于完成指定商品规格的单位转换率计算。</br>
   * 从业务的角度出发，sourceUnitCode为输入的用户购买商品规格的单位编号，targetUnitCode为（营销活动中）设定的商品规格的单位编号
   *
   * @param productSpecificationCode
   * @param sourceUnitCode
   * @param targetUnitCode
   * @return
   */
  @Override
  public float transformationUnitRate(String productSpecificationCode, String sourceUnitCode, String targetUnitCode) {
    if (StringUtils.isAnyBlank(productSpecificationCode, sourceUnitCode, targetUnitCode)) {
      return 1.0f;
    }

    // 如果两个单位相同，则返回1.0f
    if (StringUtils.equals(sourceUnitCode, targetUnitCode)) {
      return 1.0f;
    }

    BigDecimal sourceUnitRate = this.findConvertRate(productSpecificationCode, sourceUnitCode);
    BigDecimal targetUnitRate = this.findConvertRate(productSpecificationCode, targetUnitCode);
    // 得到的转换率只保留2为
    if (targetUnitRate.floatValue() == 0.0f) {
      // 只要设置正确，基本上不会出现这种情况
      return 1.0f;
    }
    return sourceUnitRate.divide(targetUnitRate, 6, RoundingMode.HALF_UP).floatValue();
  }

  @Override
  public ProductSpecificationVo findByBarCode(String barCode, String customerCode) {
    if (StringUtils.isBlank(barCode)) {
      return null;
    }
    ProductSpecification productSpecification = this.productSpecificationRepository.findByTenantCodeAndBarCode(TenantUtils.getTenantCode(), barCode);
    if (productSpecification == null) {
      return null;
    }
    //如果客户编码为空 ，那么直接返回
    if(StringUtils.isBlank(customerCode)){
      return nebulaToolkitService.copyObjectByWhiteList(productSpecification,ProductSpecificationVo.class,HashSet.class,ArrayList.class,"product","product.productBrand","product.productCategory","product.productShowCategory","productUnitSpecificationAndPrices","productPricings","productBarCodeInfos","productUnitSpecificationAndPrices.productUnit");
    }
    //验证是否在允消清单
    if (productCustomerBuyableEventListener != null) {
      //如果商品不在允消清单中
      if (!productCustomerBuyableEventListener.isCustomerBuyable(customerCode, productSpecification.getProductSpecificationCode())) {
        return null;
      }
    }
    //丰富 客户定价信息
    // 获取打折信息
    productActionService.coverDiscountResults(customerCode,productSpecification.getProductUnitSpecificationAndPrices());

    //丰富库存信息
    ProductSpecificationVo productSpecificationVo = nebulaToolkitService.copyObjectByWhiteList(productSpecification,ProductSpecificationVo.class,HashSet.class,ArrayList.class,"product","product.productBrand","product.productCategory","product.productShowCategory","productUnitSpecificationAndPrices","productPricings","productBarCodeInfos","productUnitSpecificationAndPrices.productUnit");
    if(productWarehouseEventListener!=null){
      productWarehouseEventListener.findProductSpecificationWarehouse(productSpecificationVo,customerCode);
    }
    return productSpecificationVo;
  }

  @Override
  public BigDecimal convertProductBasicUnitQuantity(String productCode, String unitCode, BigDecimal quantity) {
    Validate.notBlank(productCode,"商品编码不能为空");
    Validate.notNull(quantity,"数量不能为空");
    Validate.notNull(unitCode,"单位编码不能为空");
    //通过商品编码获取商品信息， 通过商品信息获取基础单位编码
    Product product = productRepository.findByProductCode(productCode,TenantUtils.getTenantCode());
    Validate.notNull(product,"商品信息不存在");
    Validate.notEmpty(product.getProductUnitAndPrices(),"商品单位价格信息不存在");
    //获取基础单位
    ProductUnitAndPrice productUnitAndPrice = product.getProductUnitAndPrices().stream().filter(ProductUnitAndPrice::getBasicUnit).findFirst().get();
    Validate.notNull(productUnitAndPrice,"基础单位信息不存在");
    return this.covertProductUnitQuantity(productCode,unitCode,productUnitAndPrice.getProductUnit().getUnitCode(),quantity);
  }

  @Override
  public BigDecimal covertProductUnitQuantity(String productCode, String sourceUnitCode, String targetUnitCode, BigDecimal quantity) {
    Validate.notBlank(productCode,"商品编码不能为空");
    Validate.notBlank(sourceUnitCode,"源单位编码不能为空");
    Validate.notNull(quantity,"数量不能为空");
    Validate.notNull(targetUnitCode,"目标单位编码不能为空");
    //通过商品编码 获取商品单位 转换信息
    //通过转换比例 转换数量  计算方式为： 源单位转换比例 ÷ 目标单位转换比例 × 数量 = 转换后数量
    Product product = productRepository.findByProductCode(productCode,TenantUtils.getTenantCode());
    Validate.notNull(product,"商品信息不存在");
    Validate.notEmpty(product.getProductUnitAndPrices(),"商品单位价格信息不存在");
    Set<ProductUnitAndPrice> productUnitAndPrices = product.getProductUnitAndPrices();
    //这里转换为Map<unitCode,转换比例> 方便计算
    Map<String, BigDecimal> map = productUnitAndPrices.stream().collect(Collectors.toMap(e -> e.getProductUnit().getUnitCode(), ProductUnitAndPrice::getConversionRatio));
    return covertUnitQuantity(map,sourceUnitCode,targetUnitCode,quantity);
  }

  @Override
  public BigDecimal convertSpecificationBasicUnitQuantity(String specificationCode, String unitCode, BigDecimal quantity) {
    Validate.notBlank(specificationCode,"规格编码不能为空");
    Validate.notNull(quantity,"数量不能为空");
    Validate.notNull(unitCode,"单位编码不能为空");
    //通过商品编码获取商品信息， 通过商品信息获取基础单位编码
    ProductSpecification productSpecification = productSpecificationRepository.findByTenantCodeAndProductSpecificationCode(TenantUtils.getTenantCode(),specificationCode);
    Validate.notNull(productSpecification,"商品规格信息不存在");
    Validate.notEmpty(productSpecification.getProductUnitSpecificationAndPrices(),"商品规格单位价格信息不存在");
    //获取基础单位
    ProductUnitSpecificationAndPrice productUnitSpecificationAndPrice = productSpecification.getProductUnitSpecificationAndPrices().stream().filter(ProductUnitSpecificationAndPrice::getBasicUnit).findFirst().get();
    Validate.notNull(productUnitSpecificationAndPrice,"基础单位信息不存在");
    return this.convertSpecificationUnitQuantity(specificationCode,unitCode,productUnitSpecificationAndPrice.getProductUnit().getUnitCode(),quantity);
  }

  @Override
  public BigDecimal convertSpecificationUnitQuantity(String specificationCode, String sourceUnitCode, String targetUnitCode, BigDecimal quantity) {
    Validate.notBlank(specificationCode,"商品编码不能为空");
    Validate.notBlank(sourceUnitCode,"源单位编码不能为空");
    Validate.notNull(quantity,"数量不能为空");
    Validate.notNull(targetUnitCode,"目标单位编码不能为空");
    //通过商品编码 获取商品单位 转换信息
    //通过转换比例 转换数量  计算方式为： 源单位转换比例 ÷ 目标单位转换比例 × 数量 = 转换后数量
    ProductSpecification productSpecification = productSpecificationRepository.findByTenantCodeAndProductSpecificationCode(TenantUtils.getTenantCode(),specificationCode);
    Validate.notNull(productSpecification,"商品规格信息不存在");
    Validate.notEmpty(productSpecification.getProductUnitSpecificationAndPrices(),"规格单位价格信息不存在");
    Set<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrices = productSpecification.getProductUnitSpecificationAndPrices();
    //这里转换为Map<unitCode,转换比例> 方便计算
    Map<String, BigDecimal> map = productUnitSpecificationAndPrices.stream().collect(Collectors.toMap(e -> e.getProductUnit().getUnitCode(), ProductUnitSpecificationAndPrice::getConversionRatio));
    return covertUnitQuantity(map,sourceUnitCode,targetUnitCode,quantity);
  }

  /**
   * 单位数量转换 算法：（源单位转换比例 ÷ 目标单位转换比例） × 数量 = 转换后数量
   * @param map
   * @param sourceUnitCode
   * @param targetUnitCode
   * @param quantity
   * @return
   */
  private BigDecimal covertUnitQuantity(Map<String,BigDecimal> map,String sourceUnitCode, String targetUnitCode, BigDecimal quantity){
    //这里验证单位是否存在
    Validate.isTrue(map.containsKey(sourceUnitCode),"该商品不存在源单位信息");
    Validate.isTrue(map.containsKey(targetUnitCode),"该商品不存在目标单位信息");
    //通过公式计算数量
    BigDecimal conversion = map.get(sourceUnitCode).divide(map.get(targetUnitCode),2,BigDecimal.ROUND_HALF_UP);
    return quantity.multiply(conversion);
  }

}
