package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.product.entity.ProductTag;
import com.bizunited.empower.business.product.repository.ProductTagRepository;
import com.bizunited.empower.business.product.service.ProductTagService;
import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * ProductTag业务模型的服务层接口实现
 * @author saturn
 */
@Service("ProductTagServiceImpl")
public class ProductTagServiceImpl implements ProductTagService {

  /**
   * 顶层标识
   */
  final static Integer TOP_LEVEL = 0;
  @Autowired
  private ProductTagRepository productTagRepository;
  @Autowired
  private RedisMutexService redisMutexService;
  /**
   * 商品品牌编号前缀
   */
  private static final String PRODUCT_TAG_CODE_PREFIX = "PT";
  /**
   * 商品品牌编号生成所使用的redis分布式锁的前缀
   */
  private static final String PRODUCT_TAG_REDIS_LOCK_CODE = "PT_PRODUCT_TAG_";

  @Transactional
  @Override
  public ProductTag create(ProductTag productTag) { 
    ProductTag current = this.createForm(productTag);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public ProductTag createForm(ProductTag productTag) { 
   /* 
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */
    Date now = new Date();
    productTag.setCreateAccount(SecurityUtils.getUserAccount());
    productTag.setCreateTime(now);
    productTag.setModifyAccount(SecurityUtils.getUserAccount());
    productTag.setModifyTime(now);
    // =============系统自构数据==================
    String tenantCode = this.getTenantCode();
    productTag.setTenantCode(tenantCode);
    String productTagCode = this.generateCode(tenantCode);
    productTag.setTagCode(productTagCode);
    // ===============================
    this.createValidation(productTag);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.productTagRepository.save(productTag);
    // TODO manyToMany关联商品信息目前版本下还需要开发人员自行处理
    
    // 返回最终处理的结果，里面带有详细的关联信息
    return productTag;
  }

  private String generateCode(String tenantCode) {
    String productTagCode = null;
    String lockCode = StringUtils.join(PRODUCT_TAG_REDIS_LOCK_CODE , tenantCode);
    try {
      this.redisMutexService.lock(PRODUCT_TAG_REDIS_LOCK_CODE);
      String atomicNumber = this.redisMutexService.getAndIncrement(lockCode, 1,6);
      productTagCode = StringUtils.join(PRODUCT_TAG_CODE_PREFIX , atomicNumber);
    } catch (Exception e) {
      throw new RuntimeException("生成商品标签编码失败",e);
    }finally {
      this.redisMutexService.unlock(PRODUCT_TAG_REDIS_LOCK_CODE);
    }
    return productTagCode;
  }

  private String getTenantCode() {
    String tenantCode = TenantUtils.getTenantCode();
    return tenantCode;
  }

  /**
   * 在创建一个新的ProductTag模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProductTag productTag) { 
    Validate.notNull(productTag , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(productTag.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    productTag.setId(null);
    Validate.notBlank(productTag.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(productTag.getTagName(), "添加信息时，标签名不能为空！");
    Validate.notBlank(productTag.getTagCode(), "添加信息时，标签编号不能为空！");
    // 同名标签不能存入
    List<ProductTag> list = productTagRepository.findByTagNameAndTenantCode(productTag.getTagName(), TenantUtils.getTenantCode());
    Validate.isTrue(CollectionUtils.isEmpty(list),"添加信息时，标签名称已存在！");

    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(productTag.getTenantCode() == null || productTag.getTenantCode().length() < 255 , "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productTag.getTagName() == null || productTag.getTagName().length() < 255 , "标签名,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productTag.getTagCode() == null || productTag.getTagCode().length() < 64 , "标签编号,在进行添加时填入值超过了限定长度(64)，请检查!");
  }
  @Transactional
  @Override
  public ProductTag update(ProductTag productTag) { 
    ProductTag current = this.updateForm(productTag);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public ProductTag updateForm(ProductTag productTag) { 
    /* 
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     * 
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    
    this.updateValidation(productTag);
    // ===================基本信息
    String currentId = productTag.getId();
    Optional<ProductTag> op_currentProductTag = this.productTagRepository.findById(currentId);
    ProductTag currentProductTag = op_currentProductTag.orElse(null);
    currentProductTag = Validate.notNull(currentProductTag ,"未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentProductTag.setModifyAccount(SecurityUtils.getUserAccount());
    currentProductTag.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentProductTag.setTenantCode(productTag.getTenantCode());
    currentProductTag.setTagName(productTag.getTagName());
    currentProductTag.setTagCode(productTag.getTagCode());
    currentProductTag.setLevel(productTag.getLevel());
    currentProductTag.setParentCode(productTag.getParentCode());
    currentProductTag.setFlatCode(productTag.getFlatCode());
    
    this.productTagRepository.saveAndFlush(currentProductTag);
    return currentProductTag;
  }
  /**
   * 在更新一个已有的ProductTag模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ProductTag productTag) { 
    Validate.isTrue(!StringUtils.isBlank(productTag.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    productTag.setTenantCode(TenantUtils.getTenantCode());
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(productTag.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(productTag.getTagName(), "修改信息时，标签名不能为空！");
    Validate.notBlank(productTag.getTagCode(), "修改信息时，标签编号不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(productTag.getTenantCode() == null || productTag.getTenantCode().length() < 255 , "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productTag.getTagName() == null || productTag.getTagName().length() < 255 , "标签名,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productTag.getTagCode() == null || productTag.getTagCode().length() < 64 , "标签编号,在进行修改时填入值超过了限定长度(64)，请检查!");
    // 重复性判断，同租户下名称不能重复
    Optional<ProductTag> byId = productTagRepository.findById(productTag.getId());
    Validate.isTrue(byId.isPresent(), "修改信息时，修改对象在数据库中不存在！");
    ProductTag currentForCode = byId.get();
    if (!currentForCode.getTagName().equals(productTag.getTagName())) {
      List<ProductTag> list = productTagRepository.findByTagNameAndTenantCode(productTag.getTagName(), TenantUtils.getTenantCode());
      Validate.isTrue(CollectionUtils.isEmpty(list),"修改信息时，名称已存在！");
    }
  }
  @Override
  public Set<ProductTag> findDetailsByProducts(String products) {
    if(StringUtils.isBlank(products)) { 
      return Sets.newHashSet();
    }
    return this.productTagRepository.findDetailsByProducts(products);
  }
  @Override
  public ProductTag findDetailsById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.productTagRepository.findDetailsById(id);
  }
  @Override
  public ProductTag findById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    Optional<ProductTag> op = productTagRepository.findById(id);
    return op.orElse(null); 
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    ProductTag current = this.findById(id);
    if(current != null) { 
      this.productTagRepository.delete(current);
    }
  }

  /**
   * 分页查询
   * @param tagName
   * @param pageable
   * @return
   */
  @Override
  public Page<ProductTag> findByConditions(String tagName, Pageable pageable) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    Page<ProductTag> rolePage = this.productTagRepository.findByTenantCodeAndTagNameContainingOrderByCreateTimeDescTagCode(TenantUtils.getTenantCode(),tagName,pageable);
    if(CollectionUtils.isEmpty(rolePage.getContent())) {
      return Page.empty(pageable);
    }
    return new PageImpl<>(Lists.newArrayList(rolePage.getContent()), pageable, rolePage.getTotalElements());
  }

  /**
   * 批量启用
   * @param ids
   * @return
   */
  @Override
  public Set<ProductTag> enables(String[] ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(Arrays.asList(ids)),"入参不能为空！");
    Set<ProductTag> set = this.productTagRepository.findByIdIn(ids);
    set.stream().forEach(tag->{
      tag.setTstatus(NormalStatusEnum.ENABLE.getStatus());
      productTagRepository.save(tag);
    });
    return set;
  }

  /**
   * 批量禁用
   * @param ids
   * @return
   */
  @Override
  public Set<ProductTag> disables(String[] ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(Arrays.asList(ids)),"入参不能为空！");
    Set<ProductTag> set = this.productTagRepository.findByIdIn(ids);
    set.stream().forEach(tag->{
      tag.setTstatus(NormalStatusEnum.DISABLE.getStatus());
      productTagRepository.save(tag);
    });
    return set;
  }

  /**
   * 根据父级code查询下一级，不传就查询顶层品牌
   * @param pCode
   * @return
   */
  @Override
  public List<ProductTag> findByPCode(String pCode) {
    if(StringUtils.isBlank(pCode)) {
      return this.productTagRepository.findByLevelAndTenantCodeOrderByTagCode(TOP_LEVEL,TenantUtils.getTenantCode());
    }
    return this.productTagRepository.findByParentCode(pCode,TenantUtils.getTenantCode());

  }

  @Override
  public void deleteByBatch(Set<ProductTag> tags) {
    if (CollectionUtils.isNotEmpty(tags)){
      this.productTagRepository.deleteAll(tags);
    }
  }

  @Override
  public List<ProductTag> findByEnable() {
    return this.productTagRepository.findByTstatusAndTenantCode(NormalStatusEnum.ENABLE.getStatus(),TenantUtils.getTenantCode());
  }

}
