package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.product.dto.ProductSpecificationAndUnitPriceDto;
import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.empower.business.product.entity.ProductUnit;
import com.bizunited.empower.business.product.entity.ProductUnitAndPrice;
import com.bizunited.empower.business.product.optimize.vo.ProductFlatVo;
import com.bizunited.empower.business.product.optimize.vo.ProductSpecificationFlatVo;
import com.bizunited.empower.business.product.optimize.vo.ProductUnitAndPriceFlatVo;
import com.bizunited.empower.business.product.optimize.service.ProductFlatService;
import com.bizunited.empower.business.product.repository.ProductSpecificationRepository;
import com.bizunited.empower.business.product.repository.ProductUnitAndPriceRepository;
import com.bizunited.empower.business.product.repository.ProductUnitRepository;
import com.bizunited.empower.business.product.service.ProductUnitAndPriceService;
import com.bizunited.empower.business.product.service.ProductUnitService;
import com.bizunited.empower.business.product.service.notifier.ProductUnitAndPriceEventListener;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * ProductUnitAndPrice业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("ProductUnitAndPriceServiceImpl")
public class ProductUnitAndPriceServiceImpl implements ProductUnitAndPriceService {
  @Autowired
  private ProductUnitAndPriceRepository productUnitAndPriceRepository;
  @Autowired
  private ProductUnitRepository productUnitRepository;
  @Autowired
  private ProductSpecificationRepository productSpecificationRepository;
  @Autowired
  private ProductUnitService productUnitService;
  @Autowired
  private ProductFlatService productFlatService;


  @Autowired(required = false)
  private List<ProductUnitAndPriceEventListener> productUnitAndPriceEventListener;

  /**
   * 批量新增商品单位与价格
   * @param productUnitAndPrices
   * @param product
   * @return
   */
  @Transactional
  @Override
  public Set<ProductUnitAndPrice> createInsertAbleEntitySet(Set<ProductUnitAndPrice> productUnitAndPrices, Product product) {
    Validate.notNull(product, "未传入商品信息！");
    if (!CollectionUtils.isEmpty(productUnitAndPrices)) {
      int basic = 0;
      for (ProductUnitAndPrice unitAndPrice : productUnitAndPrices) {
        unitAndPrice.setProduct(product);
        this.createInsertAbleEntity(unitAndPrice);
        //只能有一个基本单位
        Boolean basicUnit = unitAndPrice.getBasicUnit();
        if (basicUnit) {
          basic++;
        }
      }
      Validate.isTrue(basic == 1, "基本单位必须设定");
      productUnitAndPriceRepository.saveAll(productUnitAndPrices);
      return productUnitAndPrices;
    } else {
      return Sets.newHashSet();
    }
  }

  @Transactional
  @Override
  public void deleteByBatch(Set<ProductUnitAndPrice> productUnitAndPrices) {
    if (!CollectionUtils.isEmpty(productUnitAndPrices)) {
      this.productUnitAndPriceRepository.deleteAll(productUnitAndPrices);
    }
  }

  @Transactional
  @Override
  public Set<ProductUnitAndPrice> updateFormBatch(Set<ProductUnitAndPrice> productUnitAndPrices, Product product) {
    /* 2.修改商品单位与价格
     *    特殊的前置校验
     *    分为有ID的，无ID的
     *    2.1删去不存在了的
     *    2.2为新增信息赋值
     *    2.3新，旧信息一起保存
     */
    // 特殊的前置校验
    this.preCheck(productUnitAndPrices);
    // 分为有ID的，无ID的
    Map<Boolean, List<ProductUnitAndPrice>> idBooleanMap = productUnitAndPrices.stream()
        .collect(Collectors.partitioningBy(productUnitAndPrice -> StringUtils.isNotBlank(productUnitAndPrice.getId())));
    List<ProductUnitAndPrice> idTrueList = idBooleanMap.get(true);
    List<ProductUnitAndPrice> idFalseList = idBooleanMap.get(false);
    // 2.1删去不存在了的
    List<String> ids = idTrueList.stream().map(ProductUnitAndPrice::getId).filter(StringUtils::isNotEmpty).collect(Collectors.toList());
    Set<ProductUnitAndPrice> needDelete ;
    if(CollectionUtils.isEmpty(ids)){
      needDelete = productUnitAndPriceRepository.findByProduct(TenantUtils.getTenantCode(), product.getProductCode());
    }else {
      needDelete = productUnitAndPriceRepository.findByProductAndIdNotIn(TenantUtils.getTenantCode(), product.getProductCode(), ids);
    }
    if (!CollectionUtils.isEmpty(needDelete)) {
      productUnitAndPriceRepository.deleteAll(needDelete);
      //触发单位删除事件
      if(!CollectionUtils.isEmpty(productUnitAndPriceEventListener)){
        for (ProductUnitAndPriceEventListener unitAndPriceEventListener : productUnitAndPriceEventListener) {
          unitAndPriceEventListener.onDelete(needDelete);
        }
      }

    }
    Set<ProductUnitAndPrice> saveAll = new HashSet<>(productUnitAndPrices.size());
    // 2.2为新增信息赋值
    for (ProductUnitAndPrice unitAndPrice : idFalseList) {
      unitAndPrice.setProduct(product);
      this.createInsertAbleEntity(unitAndPrice);
      saveAll.add(unitAndPrice);
    }
    // 2.3 旧信息覆盖
    Set<Pair<ProductUnitAndPrice,ProductUnitAndPrice>> updates = Sets.newHashSet();
    for (ProductUnitAndPrice unitAndPrice : idTrueList) {
      unitAndPrice.setProduct(product);
      unitAndPrice.setTenantCode(TenantUtils.getTenantCode());
      this.updateValidation(unitAndPrice);
      saveAll.add(unitAndPrice);
      ProductUnitAndPrice oldBdProductUnitAndPrice = this.findDetailsById(unitAndPrice.getId());
      Validate.notNull(oldBdProductUnitAndPrice,"根据id【%s】，未能获取到商品单位与价格信息，请联系管理员",unitAndPrice.getId());
      Pair<ProductUnitAndPrice,ProductUnitAndPrice> pair = Pair.of(oldBdProductUnitAndPrice,unitAndPrice);
      updates.add(pair);
    }
    //修改单位信息触发 修改事件
    if(!CollectionUtils.isEmpty(idTrueList)&&!CollectionUtils.isEmpty(productUnitAndPriceEventListener)){
      for (ProductUnitAndPriceEventListener unitAndPriceEventListener : productUnitAndPriceEventListener) {
        unitAndPriceEventListener.onUpdate(updates);
      }
    }
    productUnitAndPriceRepository.saveAll(saveAll);
    return productUnitAndPrices;
  }

  @Override
  public ProductUnitAndPriceFlatVo findBasicBySpecificationCode(String specificationCode) {
    Validate.notBlank(specificationCode, "商品规格必须传入！");
    Optional<ProductFlatVo> productFlatDtoOptional = productFlatService.findBySpecificationCode(specificationCode);
    if (!productFlatDtoOptional.isPresent()) {
      return null;
    }
    Map<String, ProductUnitAndPriceFlatVo> unitAndPriceMap = productFlatDtoOptional.get().getProductUnitAndPriceMap();
    for (ProductUnitAndPriceFlatVo dto : unitAndPriceMap.values()) {
      if (dto.getBasicUnit()) {
        return dto;
      }
    }
    return null;
  }

  @Override
  public ProductUnitAndPriceFlatVo findBySpecificationCodeAndUnitCode(String specificationCode, String unitCode) {
    if (StringUtils.isBlank(specificationCode)) {
      return null;
    }
    if (StringUtils.isBlank(unitCode)) {
      return null;
    }
    Optional<ProductFlatVo> productFlatDto = productFlatService.findBySpecificationCode(specificationCode);
    if (!productFlatDto.isPresent()) {
      return null;
    }
    Map<String, ProductUnitAndPriceFlatVo> unitAndPriceMap = productFlatDto.get().getProductUnitAndPriceMap();
    return unitAndPriceMap.get(unitCode);
  }

  @Override
  public ProductSpecificationAndUnitPriceDto findPriceBySpecUnitPairList(List<Pair<String, String>> pairList) {
    ProductSpecificationAndUnitPriceDto dto = new ProductSpecificationAndUnitPriceDto();
    if (CollectionUtils.isEmpty(pairList)) {
      return dto;
    }
    /**
     * Pair：左：规格，右：单位
     * 1. 根据入参，获取 规格code集合，单位code集合
     * 2. 根据规格code集合获取规格信息
     * 3. 根据单位code集合获取单位信息
     * 4. 将查询结果收集为Map
     * 5. 拼装结果DTO
     */
    //根据入参，获取 规格code集合，单位code集合
    List<String> specificationCodeList = new ArrayList<>(pairList.size());
    List<String> unitCodeList =  new ArrayList<>(pairList.size());
    for (Pair<String, String> pair : pairList) {
      specificationCodeList.add(pair.getLeft());
      unitCodeList.add(pair.getRight());
    }
    // 查询信息 , 就当是把数据加缓存了，下面的方法就从缓存获取数据
    List<ProductFlatVo> productFlatVoList = productFlatService.findBySpecificationCodeList(specificationCodeList);
    // 拼装结果DTO
    for (Pair<String, String> pair : pairList) {
      String left = pair.getLeft();
      String right = pair.getRight();
      Optional<ProductFlatVo> productFlatDtoOptional = productFlatService.findBySpecificationCode(left);
      if (!productFlatDtoOptional.isPresent()) {
        break;
      }
      ProductFlatVo productFlatVo = productFlatDtoOptional.get();
      Map<String, ProductSpecificationFlatVo> productSpecificationMap = productFlatVo.getProductSpecificationMap();
      ProductSpecificationAndUnitPriceDto internal = new ProductSpecificationAndUnitPriceDto();
      internal.setProductSpecificationCode(left);
      ProductSpecificationFlatVo productSpecificationFlatVo = productSpecificationMap.get(left);
      if (Objects.isNull(productSpecificationFlatVo)) {
        internal.setErrorPair(true);
      } else {
        internal.setProductSpecificationName(productSpecificationFlatVo.getProductSpecificationName());
      }
      internal.setUnitCode(right);
      Map<String, ProductUnitAndPriceFlatVo> productUnitAndPriceMap = productFlatVo.getProductUnitAndPriceMap();
      ProductUnitAndPriceFlatVo unitAndPriceFlatDto = productUnitAndPriceMap.get(right);
      if (Objects.isNull(unitAndPriceFlatDto)) {
        internal.setErrorPair(true);
      } else {
        internal.setUnitName(unitAndPriceFlatDto.getUnitName());
        internal.setSellingPrice(unitAndPriceFlatDto.getSellingPrice());
      }
      dto.setProductSpecificationAndUnitPriceDto(internal);
    }
    return dto;
  }

  /**
   * 基本单位只能有一个
   * @param productUnitAndPrices
   */
  private void preCheck(Set<ProductUnitAndPrice> productUnitAndPrices) {
    Validate.isTrue(!CollectionUtils.isEmpty(productUnitAndPrices), "修改商品时，必须传入商品单位与价格信息");
    int basic = 0;
    for (ProductUnitAndPrice unitAndPrice : productUnitAndPrices) {
      //只能有一个基本单位
      Boolean basicUnit = unitAndPrice.getBasicUnit();
      if (basicUnit) {
        basic++;
      }
    }
    Validate.isTrue(basic == 1, "基本单位必须设定");
  }

  private ProductUnitAndPrice createInsertAbleEntity(ProductUnitAndPrice unitAndPrice) {
    unitAndPrice.setTenantCode(TenantUtils.getTenantCode());
    String id = unitAndPrice.getProductUnit().getId();
    Validate.notBlank(id, "需要选择单位");
    ProductUnit byId = productUnitService.findById(id);
    Validate.notNull(byId, "单位不存在");
    unitAndPrice.setProductUnit(byId);
    this.createValidation(unitAndPrice);
    return unitAndPrice;
  }

  @Transactional
  @Override
  public ProductUnitAndPrice create(ProductUnitAndPrice productUnitAndPrice) {
    ProductUnitAndPrice current = this.createForm(productUnitAndPrice);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductUnitAndPrice createForm(ProductUnitAndPrice productUnitAndPrice) {

    this.createInsertAbleEntity(productUnitAndPrice);

    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================

    this.productUnitAndPriceRepository.save(productUnitAndPrice);

    // 返回最终处理的结果，里面带有详细的关联信息
    return productUnitAndPrice;
  }

  /**
   * 在创建一个新的ProductUnitAndPrice模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProductUnitAndPrice productUnitAndPrice) {
    Validate.notNull(productUnitAndPrice, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(productUnitAndPrice.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    productUnitAndPrice.setId(null);
    Validate.notNull(productUnitAndPrice.getBasicUnit(), "添加信息时，是基本单位不能为空！");
    Validate.notNull(productUnitAndPrice.getAllowPurchase(), "添加信息时，允许购买不能为空！");
    Validate.notNull(productUnitAndPrice.getDefaultPurchase(), "添加信息时，默认购买不能为空！");
    Validate.notNull(productUnitAndPrice.getConversionRatio(), "添加信息时，换算比例不能为空！");
    Validate.notNull(productUnitAndPrice.getReferencePurchasePrice(), "添加信息时，参考进货价不能为空！");
    Validate.notNull(productUnitAndPrice.getSellingPrice(), "添加信息时，销售价不能为空！");
    BigDecimal referencePurchasePrice = productUnitAndPrice.getReferencePurchasePrice();
    Validate.isTrue(referencePurchasePrice.compareTo(BigDecimal.ZERO) > 0, "参考进货价不能小于等于零！");
    BigDecimal sellingPrice = productUnitAndPrice.getSellingPrice();
    Validate.isTrue(sellingPrice.compareTo(BigDecimal.ZERO) > 0, "销售价不能小于等于零！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）

  }

  @Transactional
  @Override
  public ProductUnitAndPrice update(ProductUnitAndPrice productUnitAndPrice) {
    ProductUnitAndPrice current = this.updateForm(productUnitAndPrice);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductUnitAndPrice updateForm(ProductUnitAndPrice productUnitAndPrice) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(productUnitAndPrice);
    // ===================基本信息
    String currentId = productUnitAndPrice.getId();
    Optional<ProductUnitAndPrice> op_currentProductUnitAndPrice = this.productUnitAndPriceRepository.findById(currentId);
    ProductUnitAndPrice currentProductUnitAndPrice = op_currentProductUnitAndPrice.orElse(null);
    currentProductUnitAndPrice = Validate.notNull(currentProductUnitAndPrice, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    // 开始重新赋值——一般属性
    currentProductUnitAndPrice.setBasicUnit(productUnitAndPrice.getBasicUnit());
    currentProductUnitAndPrice.setAllowPurchase(productUnitAndPrice.getAllowPurchase());
    currentProductUnitAndPrice.setDefaultPurchase(productUnitAndPrice.getDefaultPurchase());
    currentProductUnitAndPrice.setConversionRatio(productUnitAndPrice.getConversionRatio());
    currentProductUnitAndPrice.setReferencePurchasePrice(productUnitAndPrice.getReferencePurchasePrice());
    currentProductUnitAndPrice.setSellingPrice(productUnitAndPrice.getSellingPrice());
    currentProductUnitAndPrice.setProduct(productUnitAndPrice.getProduct());
    currentProductUnitAndPrice.setProductUnit(productUnitAndPrice.getProductUnit());

    this.productUnitAndPriceRepository.saveAndFlush(currentProductUnitAndPrice);
    return currentProductUnitAndPrice;
  }

  /**
   * 在更新一个已有的ProductUnitAndPrice模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ProductUnitAndPrice productUnitAndPrice) {
    String id = productUnitAndPrice.getProductUnit().getId();
    Validate.notBlank(id, "需要选择单位");
    ProductUnit byId = productUnitService.findById(id);
    Validate.notNull(byId, "单位不存在");
    productUnitAndPrice.setProductUnit(byId);
    Validate.isTrue(!StringUtils.isBlank(productUnitAndPrice.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    // 基础信息判断，基本属性，需要满足not null
    Validate.notNull(productUnitAndPrice.getBasicUnit(), "修改信息时，是基本单位不能为空！");
    Validate.notNull(productUnitAndPrice.getAllowPurchase(), "修改信息时，允许购买不能为空！");
    Validate.notNull(productUnitAndPrice.getDefaultPurchase(), "修改信息时，默认购买不能为空！");
    Validate.notNull(productUnitAndPrice.getConversionRatio(), "修改信息时，换算比例不能为空！");
    Validate.notNull(productUnitAndPrice.getReferencePurchasePrice(), "修改信息时，参考进货价不能为空！");
    Validate.notNull(productUnitAndPrice.getSellingPrice(), "修改信息时，销售价不能为空！");
    BigDecimal referencePurchasePrice = productUnitAndPrice.getReferencePurchasePrice();
    Validate.isTrue(referencePurchasePrice.compareTo(BigDecimal.ZERO) > 0, "参考进货价不能小于等于零！");
    BigDecimal sellingPrice = productUnitAndPrice.getSellingPrice();
    Validate.isTrue(sellingPrice.compareTo(BigDecimal.ZERO) > 0, "销售价不能小于等于零！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true

    // 关联性判断，关联属性判断，需要满足ManyToOne或者OneToOne，且not null 且是主模型
  }

  @Override
  public Set<ProductUnitAndPrice> findDetailsByProduct(String product) {
    if (StringUtils.isBlank(product)) {
      return Sets.newHashSet();
    }
    return this.productUnitAndPriceRepository.findDetailsByProduct(product);
  }

  @Override
  public Set<ProductUnitAndPrice> findDetailsByProductUnit(String productUnit) {
    if (StringUtils.isBlank(productUnit)) {
      return Sets.newHashSet();
    }
    return this.productUnitAndPriceRepository.findDetailsByProductUnit(productUnit,TenantUtils.getTenantCode());
  }

  @Override
  public ProductUnitAndPrice findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.productUnitAndPriceRepository.findDetailsById(id);
  }

  @Override
  public ProductUnitAndPrice findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    Optional<ProductUnitAndPrice> op = productUnitAndPriceRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    ProductUnitAndPrice current = this.findById(id);
    if (current != null) {
      this.productUnitAndPriceRepository.delete(current);
    }
  }
} 
