package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.product.entity.ProductUnit;
import com.bizunited.empower.business.product.optimize.vo.ProductFlatVo;
import com.bizunited.empower.business.product.optimize.vo.ProductUnitAndPriceFlatVo;
import com.bizunited.empower.business.product.optimize.service.ProductFlatService;
import com.bizunited.empower.business.product.repository.ProductUnitRepository;
import com.bizunited.empower.business.product.service.ProductUnitService;
import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * ProductUnit业务模型的服务层接口实现
 * @author saturn
 */
@Service("ProductUnitServiceImpl")
public class ProductUnitServiceImpl implements ProductUnitService { 
  @Autowired
  private ProductUnitRepository productUnitRepository;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private ProductFlatService productFlatService;
  /**
   * 商品品牌编号前缀
   */
  private static final String PRODUCT_UNIT_CODE_PREFIX = "PU";
  /**
   * 商品品牌编号生成所使用的redis分布式锁的前缀
   */
  private static final String PRODUCT_UNIT_REDIS_LOCK_CODE = "PU_PRODUCT_UNIT_";

  @Transactional
  @Override
  public ProductUnit create(ProductUnit productUnit) { 
    ProductUnit current = this.createForm(productUnit);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public ProductUnit createForm(ProductUnit productUnit) { 
   /* 
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */
    Date now = new Date();
    productUnit.setCreateAccount(SecurityUtils.getUserAccount());
    productUnit.setCreateTime(now);
    productUnit.setModifyAccount(SecurityUtils.getUserAccount());
    productUnit.setModifyTime(now);
    // =============系统自构数据==================
    String tenantCode = TenantUtils.getTenantCode();
    productUnit.setTenantCode(tenantCode);
    String productUnitCode = this.generateCode(tenantCode);
    productUnit.setUnitCode(productUnitCode);
    // ===============================
    this.createValidation(productUnit);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.productUnitRepository.save(productUnit);
    // 返回最终处理的结果，里面带有详细的关联信息
    return productUnit;
  }

  private String generateCode(String tenantCode) {
    String productUnitCode = null;
    String lockCode = StringUtils.join(PRODUCT_UNIT_REDIS_LOCK_CODE , tenantCode);
    try {
      this.redisMutexService.lock(PRODUCT_UNIT_REDIS_LOCK_CODE);
      String atomicNumber = this.redisMutexService.getAndIncrement(lockCode, 1,6);
      productUnitCode = StringUtils.join(PRODUCT_UNIT_CODE_PREFIX , atomicNumber);
    } catch (Exception e) {
      throw new RuntimeException("生成商品分类编码失败",e);
    }finally {
      this.redisMutexService.unlock(PRODUCT_UNIT_REDIS_LOCK_CODE);
    }
    return productUnitCode;
  }

  /**
   * 在创建一个新的ProductUnit模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProductUnit productUnit) { 
    Validate.notNull(productUnit , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(productUnit.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    productUnit.setId(null);
    Validate.notBlank(productUnit.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(productUnit.getUnitCode(), "添加信息时，单位编号不能为空！");
    Validate.notBlank(productUnit.getUnitName(), "添加信息时，单位名称不能为空！");
    // 同名标签不能存入
    List<ProductUnit> current = this.productUnitRepository.findByUnitNameAndTenantCode(productUnit.getUnitName(),productUnit.getTenantCode());
    Validate.isTrue(CollectionUtils.isEmpty(current), "添加信息时，单位名称已存在！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(productUnit.getTenantCode() == null || productUnit.getTenantCode().length() < 255 , "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productUnit.getUnitCode() == null || productUnit.getUnitCode().length() < 64 , "单位编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productUnit.getUnitName() == null || productUnit.getUnitName().length() < 128 , "单位名称,在进行添加时填入值超过了限定长度(128)，请检查!");
  }
  @Transactional
  @Override
  public ProductUnit update(ProductUnit productUnit) { 
    ProductUnit current = this.updateForm(productUnit);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public ProductUnit updateForm(ProductUnit productUnit) { 
    /* 
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     * 
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    
    this.updateValidation(productUnit);
    // ===================基本信息
    String currentId = productUnit.getId();
    Optional<ProductUnit> op_currentProductUnit = this.productUnitRepository.findById(currentId);
    ProductUnit currentProductUnit = op_currentProductUnit.orElse(null);
    currentProductUnit = Validate.notNull(currentProductUnit ,"未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentProductUnit.setModifyAccount(SecurityUtils.getUserAccount());
    currentProductUnit.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentProductUnit.setTenantCode(productUnit.getTenantCode());
    currentProductUnit.setUnitCode(productUnit.getUnitCode());
    currentProductUnit.setUnitName(productUnit.getUnitName());
    
    this.productUnitRepository.saveAndFlush(currentProductUnit);
    return currentProductUnit;
  }
  /**
   * 在更新一个已有的ProductUnit模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ProductUnit productUnit) { 
    Validate.isTrue(!StringUtils.isBlank(productUnit.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(productUnit.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(productUnit.getUnitCode(), "修改信息时，单位编号不能为空！");
    Validate.notBlank(productUnit.getUnitName(), "修改信息时，单位名称不能为空！");
    // 重复性判断，同租户下名称不能重复
    Optional<ProductUnit> byId = productUnitRepository.findById(productUnit.getId());
    Validate.isTrue(byId.isPresent(), "修改信息时，修改对象在数据库中不存在！");
    ProductUnit currentForCode = byId.get();
    if (!currentForCode.getUnitName().equals(productUnit.getUnitName())) {
      List<ProductUnit> list = productUnitRepository.findByUnitNameAndTenantCode(productUnit.getUnitName(), TenantUtils.getTenantCode());
      Validate.isTrue(CollectionUtils.isEmpty(list),"修改信息时，名称已存在！");
    }
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(productUnit.getTenantCode() == null || productUnit.getTenantCode().length() < 255 , "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productUnit.getUnitCode() == null || productUnit.getUnitCode().length() < 64 , "单位编号,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productUnit.getUnitName() == null || productUnit.getUnitName().length() < 128 , "单位名称,在进行修改时填入值超过了限定长度(128)，请检查!");
  } 
  @Override
  public ProductUnit findDetailsById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.productUnitRepository.findDetailsById(id);
  }
  @Override
  public ProductUnit findById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    Optional<ProductUnit> op = productUnitRepository.findById(id);
    return op.orElse(null); 
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    ProductUnit current = this.findById(id);
    if(current != null) {
      Validate.isTrue(CollectionUtils.isEmpty(current.getProductUnitSpecificationAndPrices()), "单位已经使用，禁止删除！");
      this.productUnitRepository.delete(current);
    }
  }

  /**
   * 分页查询
   * @param pageable
   * @param unitName
   * @return
   */
  @Override
  public Page<ProductUnit> findByConditions(Pageable pageable, String unitName) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    Page<ProductUnit> rolePage = this.productUnitRepository.findByTenantCodeAndUnitNameContainingOrderByCreateTimeDescUnitCode(TenantUtils.getTenantCode(),unitName,pageable);
    if(CollectionUtils.isEmpty(rolePage.getContent())) {
      return Page.empty(pageable);
    }
    return new PageImpl<>(Lists.newArrayList(rolePage.getContent()), pageable, rolePage.getTotalElements());

  }

  /**
   * 批量启用
   * @param ids
   * @return
   */
  @Override
  public Set<ProductUnit> enables(String[] ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(Arrays.asList(ids)),"入参不能为空！");
    Set<ProductUnit> set = this.productUnitRepository.findByIdIn(ids);
    set.stream().forEach(unit->{
      unit.setTstatus(NormalStatusEnum.ENABLE.getStatus());
      productUnitRepository.save(unit);
    });
    return set;
  }

  /**
   * 批量禁用
   * @param ids
   * @return
   */
  @Override
  public Set<ProductUnit> disables(String[] ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(Arrays.asList(ids)),"入参不能为空！");
    Set<ProductUnit> set = this.productUnitRepository.findByIdIn(ids);
    set.stream().forEach(unit->{
      unit.setTstatus(NormalStatusEnum.DISABLE.getStatus());
      productUnitRepository.save(unit);
    });
    return set;
  }

  @Override
  public ProductUnit findByProductCodeAndUnitCode(String productCode, String unitCode) {
    if(StringUtils.isAnyBlank(productCode, unitCode)) {
      return null;
    }
    ProductFlatVo productFlatVo = productFlatService.findByProductCode(productCode);
    if (productFlatVo == null) {
      return null;
    }
    Map<String, ProductUnitAndPriceFlatVo> productUnitAndPriceMap = productFlatVo.getProductUnitAndPriceMap();
    ProductUnitAndPriceFlatVo flatDto = productUnitAndPriceMap.get(unitCode);
    if (flatDto == null) {
      return null;
    }
    ProductUnit unit = new ProductUnit();
    unit.setUnitCode(flatDto.getUnitCode());
    unit.setUnitName(flatDto.getUnitName());
    return unit;
  }

  @Override
  public List<ProductUnit> findByEnable() {
    return productUnitRepository.findByTstatusAndTenantCode(NormalStatusEnum.ENABLE.getStatus(),TenantUtils.getTenantCode());
  }
}
