package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.product.dto.DiscountResultDto;
import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.empower.business.product.entity.ProductSpecification;
import com.bizunited.empower.business.product.entity.ProductUnit;
import com.bizunited.empower.business.product.entity.ProductUnitSpecificationAndPrice;
import com.bizunited.empower.business.product.repository.ProductSpecificationRepository;
import com.bizunited.empower.business.product.repository.ProductUnitRepository;
import com.bizunited.empower.business.product.repository.ProductUnitSpecificationAndPriceRepository;
import com.bizunited.empower.business.product.service.ProductActionService;
import com.bizunited.empower.business.product.service.ProductPriceModifyLogService;
import com.bizunited.empower.business.product.service.ProductUnitService;
import com.bizunited.empower.business.product.service.ProductUnitSpecificationAndPriceService;
import com.bizunited.empower.business.product.vo.ProductSpecificationAndUnitPriceVo;
import com.bizunited.empower.business.product.vo.ProductUnitSpecificationAndPriceVo;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * ProductUnitAndPrice业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("ProductUnitSpecificationAndPriceServiceImpl")
public class ProductUnitSpecificationAndPriceServiceImpl implements ProductUnitSpecificationAndPriceService {
  @Autowired
  private ProductUnitSpecificationAndPriceRepository productUnitSpecificationAndPriceRepository;
  @Autowired
  ProductUnitRepository productUnitRepository;
  @Autowired
  ProductSpecificationRepository productSpecificationRepository;
  @Autowired
  private ProductUnitService productUnitService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private ProductActionService productActionService;
  @Autowired
  private ProductPriceModifyLogService productPriceModifyLogService;

  /**
   * 批量新增商品单位与价格
   *
   * @param productUnitSpecificationAndPrices
   * @param product
   * @return
   */
  @Transactional
  @Override
  public Set<ProductUnitSpecificationAndPrice> createInsertAbleEntitySet(Set<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrices, Product product, ProductSpecification productSpecification) {
    Validate.notNull(product, "未传入商品信息！");
    Validate.notNull(productSpecification, "未传入规格信息！");
    if (CollectionUtils.isEmpty(productUnitSpecificationAndPrices)) {
      return Sets.newHashSet();
    }
    int basic = 0;
    for (ProductUnitSpecificationAndPrice unitAndPrice : productUnitSpecificationAndPrices) {
      unitAndPrice.setProduct(product);
      unitAndPrice.setProductSpecification(productSpecification);
      this.createInsertAbleEntity(unitAndPrice);
      //每个规格下 只有一个基本单位
      Boolean basicUnit = unitAndPrice.getBasicUnit();
      if (basicUnit) {
        basic++;
      }
    }
    Validate.isTrue(basic == 1, "基本单位必须设定");
    productUnitSpecificationAndPriceRepository.saveAll(productUnitSpecificationAndPrices);
    return productUnitSpecificationAndPrices;

  }

  @Transactional
  @Override
  public void deleteByBatch(Set<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrices) {
    Validate.isTrue(!CollectionUtils.isEmpty(productUnitSpecificationAndPrices), "删除数据不能为空");
    this.productUnitSpecificationAndPriceRepository.deleteAll(productUnitSpecificationAndPrices);
  }

  @Transactional
  @Override
  public Set<ProductUnitSpecificationAndPrice> updateFormBatch(Set<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrices, Product product, ProductSpecification productSpecification) {
    /* 2.修改商品单位与价格
     *    特殊的前置校验
     *    分为有ID的，无ID的
     *    2.1删去不存在了的
     *    2.2为新增信息赋值
     *    2.3新，旧信息一起保存
     */
    // 特殊的前置校验
    this.preCheck(productUnitSpecificationAndPrices);
    Validate.notNull(product, "商品信息不能为空");
    Validate.notNull(productSpecification, "规格不能为空");
    // 分为有ID的，无ID的
    productUnitSpecificationAndPrices = ObjectUtils.defaultIfNull(productUnitSpecificationAndPrices, Sets.newHashSet());
    Set<ProductUnitSpecificationAndPrice> deletes = Sets.newHashSet();
    Set<ProductUnitSpecificationAndPrice> updates = Sets.newHashSet();
    Set<ProductUnitSpecificationAndPrice> creates = Sets.newHashSet();
    Map<String, ProductUnitSpecificationAndPrice> productUnitSpecificationAndPriceMap = productUnitSpecificationAndPrices.stream().collect(Collectors.toMap(ProductUnitSpecificationAndPrice::getId, v -> v, (v1, v2) -> v2));
    List<ProductUnitSpecificationAndPrice> dbProductUnitSpecificationAndPrices = productUnitSpecificationAndPriceRepository.findByProductAndProductSpecification(TenantUtils.getTenantCode(), product.getProductCode(), productSpecification.getProductSpecificationCode());
    nebulaToolkitService.collectionDiscrepancy(productUnitSpecificationAndPrices, dbProductUnitSpecificationAndPrices, ProductUnitSpecificationAndPrice::getId, deletes, updates, creates);
    //删除数据
    //解决遗留问题:当在同一事物中 先使用delete 方法删除 在使用save方法保存时，
    //jpa 会先执行insert 在执行delete，该方法会产生唯一索引异常，
    // 解决方法为：使用deleteInBatch方法或者delete之后手动调用flush方法
    if (!CollectionUtils.isEmpty(deletes)) {
      productUnitSpecificationAndPriceRepository.deleteInBatch(deletes);
    }
    //更新数据
    for (ProductUnitSpecificationAndPrice update : updates) {
      //原价格信息
      BigDecimal oldSellingPrice = update.getSellingPrice();
      BigDecimal oldReferencePurchasePrice = update.getReferencePurchasePrice();
      ProductUnitSpecificationAndPrice productUnitSpecificationAndPrice = productUnitSpecificationAndPriceMap.get(update.getId());
      this.basicsUpdate(update, productUnitSpecificationAndPrice);
      this.updateValidation(update);
      productUnitSpecificationAndPriceRepository.saveAndFlush(update);
      //修改日志保存
      productPriceModifyLogService.save(update,oldSellingPrice,oldReferencePurchasePrice);

    }
    //新增数据
    for (ProductUnitSpecificationAndPrice create : creates) {
      create.setProduct(product);
      create.setProductSpecification(productSpecification);
      this.createInsertAbleEntity(create);
      productUnitSpecificationAndPriceRepository.saveAndFlush(create);
    }

    return productUnitSpecificationAndPrices;
  }

  private void basicsUpdate(ProductUnitSpecificationAndPrice dbProductUnitSpecificationAndPrice, ProductUnitSpecificationAndPrice productUnitSpecificationAndPrice) {
    dbProductUnitSpecificationAndPrice.setBasicUnit(productUnitSpecificationAndPrice.getBasicUnit());
    dbProductUnitSpecificationAndPrice.setSellingPrice(productUnitSpecificationAndPrice.getSellingPrice());
    dbProductUnitSpecificationAndPrice.setAllowPurchase(productUnitSpecificationAndPrice.getAllowPurchase());
    dbProductUnitSpecificationAndPrice.setConversionRatio(productUnitSpecificationAndPrice.getConversionRatio());
    dbProductUnitSpecificationAndPrice.setDefaultPurchase(productUnitSpecificationAndPrice.getDefaultPurchase());
    dbProductUnitSpecificationAndPrice.setReferencePurchasePrice(productUnitSpecificationAndPrice.getReferencePurchasePrice());
  }

  @Override
  public ProductUnitSpecificationAndPrice findBasicBySpecificationCode(String specificationCode) {
    if (StringUtils.isBlank(specificationCode)) {
      return null;
    }
    return productUnitSpecificationAndPriceRepository.findBasicBySpecificationCode(TenantUtils.getTenantCode(), specificationCode);
  }

  @Override
  public ProductUnitSpecificationAndPrice findBySpecificationCodeAndUnitCode(String specificationCode, String unitCode) {
    if (StringUtils.isAnyBlank(specificationCode, unitCode)) {
      return null;
    }
    return productUnitSpecificationAndPriceRepository.findBySpecificationCodeAndUnitCode(TenantUtils.getTenantCode(), specificationCode, unitCode);
  }

  @Override
  public ProductUnitSpecificationAndPrice findBySpecificationCodeAndUnitCodeAndCustomerCode(String specificationCode, String unitCode, String customerCode) {
    ProductUnitSpecificationAndPrice productUnitSpecificationAndPrice = this.findBySpecificationCodeAndUnitCode(specificationCode, unitCode);
    if (productUnitSpecificationAndPrice == null) {
      return null;
    }
    if (StringUtils.isBlank(customerCode)) {
      return productUnitSpecificationAndPrice;
    }
    /**
     *     根据现有责任链 获取商品折扣后价格
     *     1.初始化参数
     *     2.查询客户的商品 折扣后所有价格
     *     3.根据当前商品编码+规格编码+单位编码获取价格并且返回
     */
    List<DiscountResultDto> discountResultDtoList = Lists.newArrayList(new DiscountResultDto(customerCode, productUnitSpecificationAndPrice.getProduct().getProductCode()));
    List<DiscountResultDto> finalPrice = productActionService.findProductFinalPrice(discountResultDtoList);
    Map<String, List<DiscountResultDto>> puMapDiscountResult = finalPrice.stream().collect(Collectors.groupingBy(discountResultDto -> {
      String productCode = discountResultDto.getProductCode();
      String dtoUnitCode = discountResultDto.getUnitCode();
      String productSpecificationCode = discountResultDto.getProductSpecificationCode();
      return StringUtils.join(productCode, dtoUnitCode, productSpecificationCode);
    }));
    String key = StringUtils.join(productUnitSpecificationAndPrice.getProduct().getProductCode(), unitCode, specificationCode);
    List<DiscountResultDto> list = puMapDiscountResult.get(key);
    if (!CollectionUtils.isEmpty(list)) {
      BigDecimal afterDiscountPrice = list.get(0).getAfterDiscountPrice();
      productUnitSpecificationAndPrice.setSellingPrice(afterDiscountPrice);
    }
    return productUnitSpecificationAndPrice;
  }

  @Override
  public ProductSpecificationAndUnitPriceVo findPriceBySpecUnitPairList(List<Pair<String, String>> pairList) {
    ProductSpecificationAndUnitPriceVo dto = new ProductSpecificationAndUnitPriceVo();
    if (CollectionUtils.isEmpty(pairList)) {
      return dto;
    }
    /**
     * Pair：左：规格，右：单位
     * 1. 根据入参，获取 规格code集合，单位code集合
     * 2. 根据规格code集合获取规格信息
     * 3. 根据单位code集合获取单位信息
     * 4. 将查询结果收集为Map
     * 5. 拼装结果DTO
     */
    //根据入参，获取 规格code集合，单位code集合
    List<String> specificationCodeList = new ArrayList<>(pairList.size());
    List<String> unitCodeList = new ArrayList<>(pairList.size());
    for (Pair<String, String> pair : pairList) {
      specificationCodeList.add(pair.getLeft());
      unitCodeList.add(pair.getRight());
    }
    // 查询数据库信息
    List<ProductSpecification> specificationList = productSpecificationRepository.findByTenantCodeAndProductSpecificationCodeIn(TenantUtils.getTenantCode(), specificationCodeList);
    List<ProductUnit> unitList = productUnitRepository.findByTenantCodeAndUnitCodeIn(TenantUtils.getTenantCode(), unitCodeList);
    List<Map<String, Object>> codeList = productUnitSpecificationAndPriceRepository.findSpecAndUnitAndPriceBySpecificationCodeIn(TenantUtils.getTenantCode(), specificationCodeList);
    // 将查询结果收集为Map
    Map<String, String> specMap = new HashMap<>(pairList.size());
    Map<String, String> unitMap = new HashMap<>(pairList.size());
    Map<String, Map<String, Object>> pairMap = new HashMap<>(pairList.size());
    for (ProductSpecification specification : specificationList) {
      specMap.put(specification.getProductSpecificationCode(), specification.getProductSpecificationName());
    }
    for (ProductUnit unit : unitList) {
      unitMap.put(unit.getUnitCode(), unit.getUnitName());
    }
    for (Map<String, Object> map : codeList) {
      String productSpecificationCode = (String) map.get("product_specification_code");
      String unitCode = (String) map.get("unit_code");
      pairMap.put(StringUtils.join(productSpecificationCode, unitCode), map);
    }
    // 拼装结果DTO
    for (Pair<String, String> pair : pairList) {
      String left = pair.getLeft();
      String right = pair.getRight();
      ProductSpecificationAndUnitPriceVo internal = new ProductSpecificationAndUnitPriceVo();
      internal.setProductSpecificationCode(left);
      internal.setProductSpecificationName(specMap.get(left));
      internal.setUnitCode(right);
      internal.setUnitName(unitMap.get(right));
      Map<String, Object> obj = pairMap.get(StringUtils.join(left, right));
      internal.setErrorPair(Objects.isNull(obj));
      BigDecimal sellingPrice = (BigDecimal) obj.get("selling_price");
      internal.setSellingPrice(sellingPrice);
      dto.setProductSpecificationAndUnitPriceDto(internal);
    }
    return dto;
  }

  /**
   * 基本单位只能有一个
   *
   * @param productUnitSpecificationAndPrices
   */
  private void preCheck(Set<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrices) {
    Validate.isTrue(!CollectionUtils.isEmpty(productUnitSpecificationAndPrices), "修改商品时，必须传入商品单位与价格信息");
    int basic = 0;
    for (ProductUnitSpecificationAndPrice unitAndPrice : productUnitSpecificationAndPrices) {
      //只能有一个基本单位
      Boolean basicUnit = unitAndPrice.getBasicUnit();
      if (basicUnit) {
        basic++;
      }
    }
    Validate.isTrue(basic == 1, "基本单位必须设定");
  }

  private ProductUnitSpecificationAndPrice createInsertAbleEntity(ProductUnitSpecificationAndPrice unitAndPrice) {
    unitAndPrice.setTenantCode(TenantUtils.getTenantCode());
    String id = unitAndPrice.getProductUnit().getId();
    Validate.notBlank(id, "需要选择单位");
    ProductUnit byId = productUnitService.findById(id);
    Validate.notNull(byId, "单位不存在");
    unitAndPrice.setProductUnit(byId);
    this.createValidation(unitAndPrice);
    return unitAndPrice;
  }

  @Transactional
  @Override
  public ProductUnitSpecificationAndPrice create(ProductUnitSpecificationAndPrice productUnitSpecificationAndPrice) {
    ProductUnitSpecificationAndPrice current = this.createForm(productUnitSpecificationAndPrice);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductUnitSpecificationAndPrice createForm(ProductUnitSpecificationAndPrice productUnitSpecificationAndPrice) {

    this.createInsertAbleEntity(productUnitSpecificationAndPrice);

    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================

    this.productUnitSpecificationAndPriceRepository.save(productUnitSpecificationAndPrice);

    // 返回最终处理的结果，里面带有详细的关联信息
    return productUnitSpecificationAndPrice;
  }

  /**
   * 在创建一个新的ProductUnitAndPrice模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProductUnitSpecificationAndPrice productUnitSpecificationAndPrice) {
    Validate.notNull(productUnitSpecificationAndPrice, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(productUnitSpecificationAndPrice.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    productUnitSpecificationAndPrice.setId(null);
    Validate.notNull(productUnitSpecificationAndPrice.getBasicUnit(), "添加信息时，是基本单位不能为空！");
    Validate.notNull(productUnitSpecificationAndPrice.getAllowPurchase(), "添加信息时，允许购买不能为空！");
    Validate.notNull(productUnitSpecificationAndPrice.getDefaultPurchase(), "添加信息时，默认购买不能为空！");
    Validate.notNull(productUnitSpecificationAndPrice.getConversionRatio(), "添加信息时，换算比例不能为空！");
    Validate.notNull(productUnitSpecificationAndPrice.getReferencePurchasePrice(), "添加信息时，参考进货价不能为空！");
    Validate.notNull(productUnitSpecificationAndPrice.getSellingPrice(), "添加信息时，销售价不能为空！");
    BigDecimal referencePurchasePrice = productUnitSpecificationAndPrice.getReferencePurchasePrice();
    Validate.isTrue(referencePurchasePrice.compareTo(BigDecimal.ZERO) > 0, "参考进货价不能小于等于零！");
    BigDecimal sellingPrice = productUnitSpecificationAndPrice.getSellingPrice();
    Validate.isTrue(sellingPrice.compareTo(BigDecimal.ZERO) > 0, "销售价不能小于等于零！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）

  }

  @Transactional
  @Override
  public ProductUnitSpecificationAndPrice update(ProductUnitSpecificationAndPrice productUnitSpecificationAndPrice) {
    ProductUnitSpecificationAndPrice current = this.updateForm(productUnitSpecificationAndPrice);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public ProductUnitSpecificationAndPrice updateForm(ProductUnitSpecificationAndPrice productUnitSpecificationAndPrice) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(productUnitSpecificationAndPrice);
    // ===================基本信息
    String currentId = productUnitSpecificationAndPrice.getId();
    Optional<ProductUnitSpecificationAndPrice> op_currentProductUnitAndPrice = this.productUnitSpecificationAndPriceRepository.findById(currentId);
    ProductUnitSpecificationAndPrice currentProductUnitSpecificationAndPrice = op_currentProductUnitAndPrice.orElse(null);
    currentProductUnitSpecificationAndPrice = Validate.notNull(currentProductUnitSpecificationAndPrice, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    // 开始重新赋值——一般属性
    currentProductUnitSpecificationAndPrice.setBasicUnit(productUnitSpecificationAndPrice.getBasicUnit());
    currentProductUnitSpecificationAndPrice.setAllowPurchase(productUnitSpecificationAndPrice.getAllowPurchase());
    currentProductUnitSpecificationAndPrice.setDefaultPurchase(productUnitSpecificationAndPrice.getDefaultPurchase());
    currentProductUnitSpecificationAndPrice.setConversionRatio(productUnitSpecificationAndPrice.getConversionRatio());
    currentProductUnitSpecificationAndPrice.setReferencePurchasePrice(productUnitSpecificationAndPrice.getReferencePurchasePrice());
    currentProductUnitSpecificationAndPrice.setSellingPrice(productUnitSpecificationAndPrice.getSellingPrice());
    currentProductUnitSpecificationAndPrice.setProduct(productUnitSpecificationAndPrice.getProduct());
    currentProductUnitSpecificationAndPrice.setProductUnit(productUnitSpecificationAndPrice.getProductUnit());

    this.productUnitSpecificationAndPriceRepository.saveAndFlush(currentProductUnitSpecificationAndPrice);
    return currentProductUnitSpecificationAndPrice;
  }

  /**
   * 在更新一个已有的ProductUnitAndPrice模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ProductUnitSpecificationAndPrice productUnitSpecificationAndPrice) {
    String id = productUnitSpecificationAndPrice.getProductUnit().getId();
    Validate.notBlank(id, "需要选择单位");
    ProductUnit byId = productUnitService.findById(id);
    Validate.notNull(byId, "单位不存在");
    productUnitSpecificationAndPrice.setProductUnit(byId);
    Validate.isTrue(!StringUtils.isBlank(productUnitSpecificationAndPrice.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    // 基础信息判断，基本属性，需要满足not null
    Validate.notNull(productUnitSpecificationAndPrice.getBasicUnit(), "修改信息时，是基本单位不能为空！");
    Validate.notNull(productUnitSpecificationAndPrice.getAllowPurchase(), "修改信息时，允许购买不能为空！");
    Validate.notNull(productUnitSpecificationAndPrice.getDefaultPurchase(), "修改信息时，默认购买不能为空！");
    Validate.notNull(productUnitSpecificationAndPrice.getConversionRatio(), "修改信息时，换算比例不能为空！");
    Validate.notNull(productUnitSpecificationAndPrice.getReferencePurchasePrice(), "修改信息时，参考进货价不能为空！");
    Validate.notNull(productUnitSpecificationAndPrice.getSellingPrice(), "修改信息时，销售价不能为空！");
    BigDecimal referencePurchasePrice = productUnitSpecificationAndPrice.getReferencePurchasePrice();
    Validate.isTrue(referencePurchasePrice.compareTo(BigDecimal.ZERO) > 0, "参考进货价不能小于等于零！");
    BigDecimal sellingPrice = productUnitSpecificationAndPrice.getSellingPrice();
    Validate.isTrue(sellingPrice.compareTo(BigDecimal.ZERO) > 0, "销售价不能小于等于零！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true

    // 关联性判断，关联属性判断，需要满足ManyToOne或者OneToOne，且not null 且是主模型
  }

  @Override
  public Set<ProductUnitSpecificationAndPrice> findDetailsByProduct(String product) {
    if (StringUtils.isBlank(product)) {
      return Sets.newHashSet();
    }
    return this.productUnitSpecificationAndPriceRepository.findDetailsByProduct(product);
  }

  @Override
  public Set<ProductUnitSpecificationAndPrice> findDetailsByProductUnit(String productUnit) {
    if (StringUtils.isBlank(productUnit)) {
      return Sets.newHashSet();
    }
    return this.productUnitSpecificationAndPriceRepository.findDetailsByProductUnit(productUnit, TenantUtils.getTenantCode());
  }

  @Override
  public ProductUnitSpecificationAndPrice findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.productUnitSpecificationAndPriceRepository.findDetailsById(id);
  }

  @Override
  public ProductUnitSpecificationAndPrice findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<ProductUnitSpecificationAndPrice> op = productUnitSpecificationAndPriceRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    ProductUnitSpecificationAndPrice current = this.findById(id);
    if (current != null) {
      this.productUnitSpecificationAndPriceRepository.delete(current);
    }
  }

  @Override
  public ProductUnitSpecificationAndPrice splitMinProice(Set<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrices, String unitCode) {
    /**
     * 1. 根据 单位编码 筛选出 同一单位 不同规格的 价格信息
     * 2. 获取不同规格最低的价格
     */
    if (CollectionUtils.isEmpty(productUnitSpecificationAndPrices)) {
      return null;
    }
    ProductUnitSpecificationAndPrice specificationAndPrices = null;
    for (ProductUnitSpecificationAndPrice productUnitSpecificationAndPrice : productUnitSpecificationAndPrices) {
      boolean isReplace = false;
      if (StringUtils.isNotBlank(unitCode)) {
        if (productUnitSpecificationAndPrice.getProductUnit().getUnitCode().equals(unitCode)) {
          isReplace = true;
        }
      } else {
        if (productUnitSpecificationAndPrice.getBasicUnit()) {
          isReplace = true;
        }
      }
      if (isReplace) {
        if (specificationAndPrices == null) {
          specificationAndPrices = productUnitSpecificationAndPrice;
        } else {
          if (productUnitSpecificationAndPrice.getSellingPrice().compareTo(specificationAndPrices.getSellingPrice()) < 0) {
            specificationAndPrices = productUnitSpecificationAndPrice;
          }
        }
      }

    }
    return specificationAndPrices;
  }

  @Override
  public Set<ProductUnitSpecificationAndPriceVo> markMinPrices(Set<ProductUnitSpecificationAndPriceVo> productUnitSpecificationAndPrices) {
    if (CollectionUtils.isEmpty(productUnitSpecificationAndPrices)) {
      return null;
    }
    ProductUnitSpecificationAndPriceVo productUnitSpecificationAndPrice = productUnitSpecificationAndPrices.stream().min(Comparator.comparing(ProductUnitSpecificationAndPriceVo::getSellingPrice)).get();
    productUnitSpecificationAndPrice.setMinPrice(true);

    return productUnitSpecificationAndPrices;
  }

  @Override
  public List<ProductUnitSpecificationAndPrice> findBySpecification(String specificationId) {
    if(StringUtils.isBlank(specificationId)){
      return Lists.newArrayList();
    }

    return productUnitSpecificationAndPriceRepository.findByProductSpecificationId(specificationId);
  }

  @Override
  @Transactional
  public void saveBatch(List<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrices) {
    Validate.notEmpty(productUnitSpecificationAndPrices,"保存信息不能为空");
    productUnitSpecificationAndPriceRepository.saveAll(productUnitSpecificationAndPrices);
  }
}
