package com.bizunited.empower.business.product.service.internal;

import com.bizunited.empower.business.common.service.SignService;
import com.bizunited.empower.business.common.util.ResponseModelUtils;
import com.bizunited.empower.business.product.common.aspect.LabelPreferentialItems;
import com.bizunited.empower.business.product.dto.DiscountResultDto;
import com.bizunited.empower.business.product.entity.Product;
import com.bizunited.empower.business.product.entity.ProductCategory;
import com.bizunited.empower.business.product.entity.ProductFile;
import com.bizunited.empower.business.product.entity.ProductMultipleSpecification;
import com.bizunited.empower.business.product.entity.ProductPricing;
import com.bizunited.empower.business.product.entity.ProductSpecification;
import com.bizunited.empower.business.product.entity.ProductUnitAndPrice;
import com.bizunited.empower.business.product.entity.ProductUnitSpecificationAndPrice;
import com.bizunited.empower.business.product.feign.ProductFeignClient;
import com.bizunited.empower.business.product.repository.ProductRepository;
import com.bizunited.empower.business.product.service.ProductActionService;
import com.bizunited.empower.business.product.service.ProductCategoryService;
import com.bizunited.empower.business.product.service.ProductFileService;
import com.bizunited.empower.business.product.service.ProductMultipleSpecificationService;
import com.bizunited.empower.business.product.service.ProductPricingService;
import com.bizunited.empower.business.product.service.ProductSpecificationService;
import com.bizunited.empower.business.product.service.ProductUnitAndPriceService;
import com.bizunited.empower.business.product.service.ProductUnitSpecificationAndPriceService;
import com.bizunited.empower.business.product.service.ProductVoService;
import com.bizunited.empower.business.product.service.notifier.ProductCustomerEventListener;
import com.bizunited.empower.business.product.service.notifier.ProductWarehouseEventListener;
import com.bizunited.empower.business.product.vo.ProductCategoryVo;

import com.bizunited.empower.business.product.vo.ProductUnitAndPriceVo;
import com.bizunited.empower.business.product.vo.ProductUnitSpecificationAndPriceVo;
import com.bizunited.empower.business.product.vo.ProductVo;
import com.bizunited.empower.business.product.vo.SaasProductVo;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.script.context.InvokeParams;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;

@Service("ProductVoServiceImpl")
public class ProductVoServiceImpl implements ProductVoService {

  @Autowired
  ProductRepository productRepository;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private ProductCategoryService productCategoryService;

  @Autowired
  ProductActionService productActionService;
  @Autowired
  private ProductFeignClient feignClient;

  @Autowired
  private SignService signService;

  @Autowired
  private ProductUnitSpecificationAndPriceService productUnitSpecificationAndPriceService;
  @Autowired
  private ProductSpecificationService productSpecificationService;
  @Autowired
  private ProductUnitAndPriceService productUnitAndPriceService;
  @Autowired
  private ProductFileService productFileService;
  @Autowired
  private ProductMultipleSpecificationService productMultipleSpecificationService;
  @Autowired
  private ProductPricingService productPricingService;

  @Autowired(required = false)
  private ProductWarehouseEventListener productWarehouseEventListener;

  @Autowired(required = false)
  private ProductCustomerEventListener productCustomerEventListener;

  @Override
  @LabelPreferentialItems("#conditions.getInvokeParam('customerCode')")
  public Page<ProductVo> findByConditions(Pageable pageable, InvokeParams conditions) {

    conditions.putInvokeParam("tenantCode", TenantUtils.getTenantCode());
    Page<Product> page = productRepository.queryPage(pageable, conditions);
    List<Product> productList = page.getContent();
    //如果查询结果为空 那么直接返回
    if (CollectionUtils.isEmpty(productList)) {
      return Page.empty();
    }
    Map<String, Object> params = conditions.getInvokeParams();
    String customerCode = (String) params.get("customerCode");
    Map<String, List<DiscountResultDto>> puMapDiscountResult = new HashMap<>();
    if (StringUtils.isNotBlank(customerCode)) {
      // 获取打折信息
      List<DiscountResultDto> discountResultDtoList = this.discountedEntry(productList, conditions);
      puMapDiscountResult = this.convertPrice(discountResultDtoList);
    }
    //------数据拷贝------------ TODO 这里可以改用更简便的中考方式，多层对象使用 xxx.xxxx的方式进行指定
    List<ProductVo> productVoList = new ArrayList<>(productList.size());
    for (Product product : productList) {
      ProductVo productVo = this.nebulaToolkitService.copyObjectByWhiteList(product, ProductVo.class, LinkedHashSet.class, ArrayList.class,
          "productCategory", "productShowCategory", "productBrand", "productFiles", "productSpecifications", "productSpecifications.productBarCodeInfos", "productMultipleSpecifications", "tags", "productPricings", "productUnitAndPrices", "productUnitAndPrices.productUnit");

      Set<ProductUnitSpecificationAndPrice> productUnitAndPrices = product.getProductUnitSpecificationAndPrices();

      if (productUnitAndPrices != null) {
        Set<ProductUnitSpecificationAndPriceVo> productUnitAndPriceVoSet = (Set<ProductUnitSpecificationAndPriceVo>) this.nebulaToolkitService.copyCollectionByWhiteList(productUnitAndPrices, ProductUnitSpecificationAndPrice.class, ProductUnitSpecificationAndPriceVo.class, LinkedHashSet.class, ArrayList.class,
            "productUnit", "product", "productSpecification");
        //------------覆盖打折结果-------------------
        if (StringUtils.isNotBlank(customerCode)) {
          this.coverDiscountResults(puMapDiscountResult, productUnitAndPriceVoSet);
        }
        //这里小程序新页面需要所有价格信息，那么标记出最低的价格信息，并且吧所有的价格信息返回
        this.productUnitSpecificationAndPriceService.markMinPrices(productUnitAndPriceVoSet);
        productVo.setProductUnitSpecificationAndPrices(productUnitAndPriceVoSet);
        //-------------------------------------------------
      }

      productVoList.add(productVo);

    }
    //这里通知库存 丰富商品库存信息
    if (productWarehouseEventListener != null) {
      productVoList = productWarehouseEventListener.findProductWarehouse(productVoList, customerCode);
    }

    return new PageImpl<>(Lists.newArrayList(productVoList), pageable, page.getTotalElements());
  }

  @Override
  public Page<ProductVo> findImageResourceByConditions(Pageable pageable, InvokeParams conditions) {
    conditions.putInvokeParam("tenantCode", TenantUtils.getTenantCode());
    Page<Product> page = productRepository.queryPage(pageable, conditions);
    List<Product> productList = page.getContent();
    //如果查询结果为空 那么直接返回
    if(CollectionUtils.isEmpty(productList)){
      return Page.empty();
    }
    Collection<ProductVo> productVos = this.nebulaToolkitService.copyCollectionByWhiteList(productList,Product.class, ProductVo.class, LinkedHashSet.class, ArrayList.class,
        "productCategory", "productShowCategory", "productBrand","productImageResources", "productFiles","productFiles.productImageResource","productSpecifications", "productSpecifications.productImageResource");
    //丰富vo返回信息
    for (ProductVo productVo : productVos) {
      //资源库图片数量
      if(!CollectionUtils.isEmpty(productVo.getProductImageResources())){
        productVo.setImageResourceCount(productVo.getProductImageResources().size());
      }
      //规格数量
      if(!CollectionUtils.isEmpty(productVo.getProductSpecifications())){
        productVo.setSpecificationCount(productVo.getProductSpecifications().size());
      }
      //已经配置规格数量
      long count = productVo.getProductSpecifications().stream().filter(e -> e.getProductImageResource() != null).count();
      productVo.setSpecificationImageResourceCount(Long.valueOf(count).intValue());

      // 商品分类展示父级名称
      this.categoryDisplayParent(productVo);
    }

    return new PageImpl<>(Lists.newArrayList(productVos), pageable, page.getTotalElements());
  }

  /**
   * 把打折结果覆盖在交易价格上
   *
   * @param puMapDiscountResult
   * @param productUnitAndPriceVoSet
   */
  private void coverDiscountResults(Map<String, List<DiscountResultDto>> puMapDiscountResult, Set<ProductUnitSpecificationAndPriceVo> productUnitAndPriceVoSet) {
    for (ProductUnitSpecificationAndPriceVo unitAndPriceVo : productUnitAndPriceVoSet) {
      String productCode = unitAndPriceVo.getProduct().getProductCode();
      String unitCode = unitAndPriceVo.getProductUnit().getUnitCode();
      String productSpecificationCode = unitAndPriceVo.getProductSpecification().getProductSpecificationCode();
      String join = StringUtils.join(productCode, unitCode, productSpecificationCode);
      List<DiscountResultDto> resultDtos = puMapDiscountResult.get(join);
      if (!CollectionUtils.isEmpty(resultDtos)) {
        BigDecimal afterDiscountPrice = resultDtos.get(0).getAfterDiscountPrice();
        unitAndPriceVo.setSellingPrice(afterDiscountPrice);
      }
    }
  }

  /**
   * 构造获取最终打折结果的入参
   *
   * @param productList
   * @param conditions
   * @return
   */
  private List<DiscountResultDto> discountedEntry(List<Product> productList, InvokeParams conditions) {
    Map<String, Object> params = conditions.getInvokeParams();
    String customerCode = (String) params.get("customerCode");

    List<DiscountResultDto> discountResultDtos = new ArrayList<>();
    for (Product product : productList) {
      DiscountResultDto dto = new DiscountResultDto();
      dto.setCustomerCode(customerCode);
      dto.setProductCode(product.getProductCode());
      discountResultDtos.add(dto);
    }
    return discountResultDtos;
  }

  @Override
  public ProductVo findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    ProductVo productVo = this.findDetailsByProductId(id);
    //丰富库存信息
    if (productWarehouseEventListener != null) {
      productVo = productWarehouseEventListener.findWarehouse(productVo, null);
    }
    //丰富客户定价中的商品客户分类和销售区域
    if(productCustomerEventListener!=null){
      productCustomerEventListener.setCustomerCategoryAndSalesArea(productVo);
    }
    return productVo;
  }

  private ProductVo findDetailsByProductId(String id) {
    Product product = this.productRepository.findDetailsById(id);
    Validate.notNull(product, "数据不存在");
    //获取规格
    Set<ProductSpecification> productSpecifications = productSpecificationService.findDetailsByProduct(product.getId());
    product.setProductSpecifications(productSpecifications);
    //获取规格与价格
    Set<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrices = productUnitSpecificationAndPriceService.findDetailsByProduct(product.getId());
    product.setProductUnitSpecificationAndPrices(productUnitSpecificationAndPrices);
    //获取单位与价格
    Set<ProductUnitAndPrice> unitAndPrices = productUnitAndPriceService.findDetailsByProduct(product.getId());
    product.setProductUnitAndPrices(unitAndPrices);
    //获取文件
    Set<ProductFile> files = productFileService.findDetailsByProduct(product.getId());
    product.setProductFiles(files);
    //获取规格键值
    Set<ProductMultipleSpecification> multipleSpecification = productMultipleSpecificationService.findDetailsByProduct(product.getId());
    product.setProductMultipleSpecifications(multipleSpecification);

    //客户定价
    Set<ProductPricing> pricing = productPricingService.findDetailsByProduct(product.getId());
    product.setProductPricings(pricing);

    //这里使用黑名单或者白名单 需要放行或者禁止的数量差不多，所以改回白名单
    ProductVo productVo = this.nebulaToolkitService.copyObjectByWhiteList(product, ProductVo.class, HashSet.class, ArrayList.class,
        "productCategory", "productShowCategory", "productBrand", "productFiles","productFiles.productImageResource", "productMultipleSpecifications", "tags",
        "productSpecifications", "productSpecifications.productImageResource","productSpecifications.productBarCodeInfos", "productSpecifications.productPricingVo",
        "productSpecifications.productPricingVo.productPricingUnitSpecifications", "productPricings", "productPricings.productPricingUnitSpecifications", "productPricings.productSpecification",
        "productUnitAndPrices", "productUnitAndPrices.productUnit", "productUnitAndPrices.productSpecification", "productUnitSpecificationAndPrices", "productUnitSpecificationAndPrices.product",
        "productUnitSpecificationAndPrices.productUnit", "productUnitSpecificationAndPrices.productSpecification");
    Set<ProductUnitAndPriceVo> productUnitAndPriceSet = productVo.getProductUnitAndPrices();
    Validate.notNull(productUnitAndPriceSet, "数据不存在");
    Set<ProductUnitAndPriceVo> productUnitAndPriceTreeSet = new TreeSet<>(Comparator.comparing(ProductUnitAndPriceVo::getConversionRatio));
    productUnitAndPriceSet.stream().forEach(productUnitAndPriceVo -> productUnitAndPriceTreeSet.add(productUnitAndPriceVo));
    productVo.setProductUnitAndPrices(productUnitAndPriceTreeSet);

    // 商品分类展示父级名称
    this.categoryDisplayParent(productVo);
    return productVo;
  }

  @Override
  public ProductVo findDetailsByIdAndCustomerCode(String id, String customerCode) {
    /**
     * 1.根据ID获取商品信息
     * 2.验证是否存在客户编码
     * 3.通过客户编码获取打折信息
     */
    ProductVo productVo = this.findDetailsByProductId(id);
    if (StringUtils.isBlank(customerCode)) {
      return productVo;
    }
    List<DiscountResultDto> discountResultDtoList = Lists.newArrayList(new DiscountResultDto(customerCode, productVo.getProductCode()));
    Map<String, List<DiscountResultDto>> puMapDiscountResult = this.convertPrice(discountResultDtoList);
    this.coverDiscountResults(puMapDiscountResult, productVo.getProductUnitSpecificationAndPrices());
    //丰富库存信息
    if (productWarehouseEventListener != null) {
      productVo = productWarehouseEventListener.findWarehouse(productVo, customerCode);
    }

    return productVo;
  }


  /**
   * 转换价格信息 转为map集合  key为 productCode+unitCode+specificationCode
   *
   * @return
   */
  private Map<String, List<DiscountResultDto>> convertPrice(List<DiscountResultDto> discountResultDtoList) {
    Validate.notEmpty(discountResultDtoList, "价格查询参数不能为空");
    List<DiscountResultDto> finalPrice = productActionService.findProductFinalPrice(discountResultDtoList);
    Validate.notEmpty(finalPrice, "价格信息为空");
    return finalPrice.stream().collect(Collectors.groupingBy(discountResultDto -> {
      String productCode = discountResultDto.getProductCode();
      String dtoUnitCode = discountResultDto.getUnitCode();
      String productSpecificationCode = discountResultDto.getProductSpecificationCode();
      return StringUtils.join(productCode, dtoUnitCode, productSpecificationCode);
    }));
  }

  @LabelPreferentialItems
  @Override
  public ProductVo findDetailsByProductCode(String code) {
    return this.findDetailsByProductCodeAndCustomerCode(code, null);
  }

  @Override
  @LabelPreferentialItems("#customerCode")
  public ProductVo findDetailsByProductCodeAndCustomerCode(String code, String customerCode) {
    if (StringUtils.isBlank(code)) {
      return null;
    }
    // fixme
    Product product = this.productRepository.findDetailsByProductCode(code, TenantUtils.getTenantCode());
    if (Objects.isNull(product)) {
      return null;
    }
    //获取规格
    Set<ProductSpecification> productSpecifications = productSpecificationService.findDetailsByProduct(product.getId());
    product.setProductSpecifications(productSpecifications);
    //获取规格与价格
    Set<ProductUnitSpecificationAndPrice> productUnitSpecificationAndPrices = productUnitSpecificationAndPriceService.findDetailsByProduct(product.getId());
    product.setProductUnitSpecificationAndPrices(productUnitSpecificationAndPrices);
    //获取单位与价格
    Set<ProductUnitAndPrice> unitAndPrices = productUnitAndPriceService.findDetailsByProduct(product.getId());
    product.setProductUnitAndPrices(unitAndPrices);
    //获取文件
    Set<ProductFile> files = productFileService.findDetailsByProduct(product.getId());
    product.setProductFiles(files);

    //这里使用黑名单或者白名单 需要放行或者禁止的数量差不多，所以改回白名单
    ProductVo productVo = this.nebulaToolkitService.copyObjectByWhiteList(product, ProductVo.class, HashSet.class, ArrayList.class, "productCategory", "productShowCategory", "productBrand", "productFiles", "productMultipleSpecifications", "tags",
        "productSpecifications", "productSpecifications.productBarCodeInfos", "productSpecifications.productPricingVo",
        "productSpecifications.productPricingVo.productPricingUnitSpecifications", "productPricings", "productPricings.productPricingUnitSpecifications", "productPricings.productSpecification",
        "productUnitAndPrices", "productUnitAndPrices.productUnit", "productUnitAndPrices.productSpecification", "productUnitSpecificationAndPrices",
        "productUnitSpecificationAndPrices.productUnit", "productUnitSpecificationAndPrices.productSpecification");
    //获取库存信息
    if (productWarehouseEventListener != null) {
      productVo = productWarehouseEventListener.findWarehouse(productVo, customerCode);
    }

    return productVo;
  }

  @Override
  public List<ProductVo> findUnitByProductCodeList(List<String> productCodes) {
    if (CollectionUtils.isEmpty(productCodes)) {
      return Lists.newLinkedList();
    }
    List<Product> list = productRepository.findByTenantCodeAndProductCodeIn(TenantUtils.getTenantCode(), productCodes);
    List<ProductVo> listVo = (List<ProductVo>) this.nebulaToolkitService.copyCollectionByWhiteList(list, Product.class, ProductVo.class, HashSet.class, ArrayList.class,
        "productUnitAndPrices", "productUnitAndPrices.productUnit", "productUnitSpecificationAndPrices", "productUnitSpecificationAndPrices.productSpecification", "productUnitSpecificationAndPrices.productUnit");
    return listVo;
  }

  private void categoryDisplayParent(ProductVo product) {
    ProductCategoryVo productCategory = product.getProductCategory();
    String parentCode = productCategory.getParentCode();
    if (StringUtils.isBlank(parentCode)) {
      return;
    }
    LinkedList<ProductCategory> linkedList = new LinkedList<>();
    do {
      ProductCategory parentCategory = productCategoryService.findByCode(parentCode);
      linkedList.push(parentCategory);
      parentCode = parentCategory.getParentCode();
    } while (StringUtils.isNotBlank(parentCode));

    ProductCategory poll = linkedList.poll();
    ProductCategoryVo topVo = this.nebulaToolkitService.copyObjectByWhiteList(poll, ProductCategoryVo.class, HashSet.class, ArrayList.class);
    ProductCategoryVo childVo;
    ProductCategoryVo parentVo = topVo;
    while (!CollectionUtils.isEmpty(linkedList)) {
      ProductCategory child = linkedList.poll();
      childVo = this.nebulaToolkitService.copyObjectByWhiteList(child, ProductCategoryVo.class, HashSet.class, ArrayList.class);
      parentVo.setChildren(Sets.newHashSet(childVo));
      parentVo = childVo;
    }
    parentVo.setChildren(Sets.newHashSet(productCategory));
    product.setProductCategory(topVo);
  }


  @Override
  public SaasProductVo findSaasProductByBarCode(String barCode) {
    if (StringUtils.isBlank(barCode)) {
      return null;
    }
    String sign = signService.sign(barCode);
    ResponseModel responseModel = feignClient.findByBarCode(signService.getCurrentAppId(), sign, barCode);
    return ResponseModelUtils.getSuccessData(responseModel, SaasProductVo.class);
  }

  @Override
  public ProductVo findByBarCode(String barCode) {
    if (StringUtils.isBlank(barCode)) {
      return null;
    }
    Product product = productRepository.findByBarCodeAndTenantCode(barCode, TenantUtils.getTenantCode());
    if(product==null){
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(product, ProductVo.class, HashSet.class, ArrayList.class, "productCategory", "productShowCategory", "productBrand", "productFiles", "productMultipleSpecifications", "tags",
        "productSpecifications", "productSpecifications.productBarCodeInfos", "productSpecifications.productPricingVo",
        "productSpecifications.productPricingVo.productPricingUnitSpecifications", "productPricings", "productPricings.productPricingUnitSpecifications", "productPricings.productSpecification",
        "productUnitAndPrices", "productUnitAndPrices.productUnit", "productUnitAndPrices.productSpecification", "productUnitSpecificationAndPrices",
        "productUnitSpecificationAndPrices.productUnit", "productUnitSpecificationAndPrices.productSpecification");
  }

  @Override
  public List<ProductVo> findBySpecificationCodes(List<String> specificationCodes) {
    if(CollectionUtils.isEmpty(specificationCodes)){
      return Lists.newArrayList();
    }
    List<Product> products = productRepository.findBySpecificationCodes(specificationCodes,TenantUtils.getTenantCode());
    if(CollectionUtils.isEmpty(products)){
      return Lists.newArrayList();
    }
    return Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(products,Product.class, ProductVo.class, HashSet.class, ArrayList.class, "productCategory", "productShowCategory", "productBrand", "productFiles", "productMultipleSpecifications", "tags",
        "productSpecifications", "productSpecifications.productBarCodeInfos", "productSpecifications.productPricingVo",
        "productSpecifications.productPricingVo.productPricingUnitSpecifications", "productPricings", "productPricings.productPricingUnitSpecifications", "productPricings.productSpecification",
        "productUnitAndPrices", "productUnitAndPrices.productUnit", "productUnitAndPrices.productSpecification", "productUnitSpecificationAndPrices",
        "productUnitSpecificationAndPrices.productUnit", "productUnitSpecificationAndPrices.productSpecification"));
  }
}
