package com.bizunited.empower.business.purchase.service.internal;

import com.bizunited.empower.business.purchase.common.enums.PurchaseOrderStatus;
import com.bizunited.empower.business.purchase.entity.PurchaseOrder;
import com.bizunited.empower.business.purchase.entity.PurchaseOrderProduct;
import com.bizunited.empower.business.purchase.repository.PurchaseOrderProductRepository;
import com.bizunited.empower.business.purchase.service.PurchaseOrderProductService;
import com.bizunited.empower.business.purchase.service.PurchaseOrderService;
import com.bizunited.empower.business.purchase.vo.PurchaseOrderProductVo;
import com.bizunited.empower.business.warehouse.service.WarehouseProductsEnterService;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Sets;

import java.lang.Override;
import java.lang.String;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import javax.transaction.Transactional;

import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

/**
 * PurchaseOrderProduct业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("PurchaseOrderProductServiceImpl")
public class PurchaseOrderProductServiceImpl implements PurchaseOrderProductService {
  @Autowired
  private PurchaseOrderService purchaseOrderService;
  @Autowired
  private PurchaseOrderProductRepository purchaseOrderProductRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private WarehouseProductsEnterService warehouseProductsEnterService;

  @Transactional
  @Override
  public PurchaseOrderProduct create(PurchaseOrderProduct purchaseOrderProduct) {
    PurchaseOrderProduct current = this.createForm(purchaseOrderProduct);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public PurchaseOrderProduct createForm(PurchaseOrderProduct purchaseOrderProduct) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    Date now = new Date();
    purchaseOrderProduct.setCreateAccount(SecurityUtils.getUserAccount());
    purchaseOrderProduct.setCreateTime(now);
    purchaseOrderProduct.setModifyAccount(SecurityUtils.getUserAccount());
    purchaseOrderProduct.setModifyTime(now);

    this.createValidation(purchaseOrderProduct);

    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================

    this.purchaseOrderProductRepository.save(purchaseOrderProduct);

    // 返回最终处理的结果，里面带有详细的关联信息
    return purchaseOrderProduct;
  }

  /**
   * 在创建一个新的PurchaseOrderProduct模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(PurchaseOrderProduct purchaseOrderProduct) {
    Validate.notNull(purchaseOrderProduct, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(purchaseOrderProduct.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    purchaseOrderProduct.setId(null);
    Validate.notBlank(purchaseOrderProduct.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(purchaseOrderProduct.getProductCode(), "添加信息时，商品编码不能为空！");
    Validate.notBlank(purchaseOrderProduct.getProductName(), "添加信息时，商品名称不能为空！");
    Validate.notBlank(purchaseOrderProduct.getMainImagePath(), "添加信息时，主图路径不能为空！");
    Validate.notBlank(purchaseOrderProduct.getProductSpecificationCode(), "添加信息时，商品（规格）编码不能为空！");
    Validate.notBlank(purchaseOrderProduct.getProductSpecificationName(), "添加信息时，商品（规格）名称不能为空！");
    Validate.notBlank(purchaseOrderProduct.getUnitCode(), "添加信息时，单位不能为空！");
    Validate.notNull(purchaseOrderProduct.getQuantity(), "添加信息时，采购数量不能为空！");
    Validate.notNull(purchaseOrderProduct.getPrice(), "添加信息时，进货单价不能为空！");
    Validate.notNull(purchaseOrderProduct.getAmount(), "添加信息时，小计不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(purchaseOrderProduct.getExtend1() == null || purchaseOrderProduct.getExtend1().length() < 255, "扩展字段1,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend2() == null || purchaseOrderProduct.getExtend2().length() < 255, "扩展字段2,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend3() == null || purchaseOrderProduct.getExtend3().length() < 255, "扩展字段3,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend4() == null || purchaseOrderProduct.getExtend4().length() < 255, "扩展字段4,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend5() == null || purchaseOrderProduct.getExtend5().length() < 255, "扩展字段5,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend6() == null || purchaseOrderProduct.getExtend6().length() < 255, "扩展字段6,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend7() == null || purchaseOrderProduct.getExtend7().length() < 255, "扩展字段7,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getTenantCode() == null || purchaseOrderProduct.getTenantCode().length() < 255, "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getProductCode() == null || purchaseOrderProduct.getProductCode().length() < 64, "商品编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getProductName() == null || purchaseOrderProduct.getProductName().length() < 128, "商品名称,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getMainImagePath() == null || purchaseOrderProduct.getMainImagePath().length() < 255, "主图路径,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getBrandCode() == null || purchaseOrderProduct.getBrandCode().length() < 64, "品牌编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getBrandName() == null || purchaseOrderProduct.getBrandName().length() < 1024, "品牌名称,在进行添加时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getProductSpecificationCode() == null || purchaseOrderProduct.getProductSpecificationCode().length() < 128, "商品（规格）编码,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getProductSpecificationName() == null || purchaseOrderProduct.getProductSpecificationName().length() < 128, "商品（规格）名称,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getUnitCode() == null || purchaseOrderProduct.getUnitCode().length() < 255, "单位,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getCategoryCode() == null || purchaseOrderProduct.getCategoryCode().length() < 255, "分类编码,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getCategoryName() == null || purchaseOrderProduct.getCategoryName().length() < 255, "分类名称,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getCategoryFlatCode() == null || purchaseOrderProduct.getCategoryFlatCode().length() < 255, "快速编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getBarCode() == null || purchaseOrderProduct.getBarCode().length() < 64, "条形码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getRemark() == null || purchaseOrderProduct.getRemark().length() < 255, "备注信息,在进行添加时填入值超过了限定长度(255)，请检查!");
  }

  @Transactional
  @Override
  public PurchaseOrderProduct update(PurchaseOrderProduct purchaseOrderProduct) {
    PurchaseOrderProduct current = this.updateForm(purchaseOrderProduct);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public PurchaseOrderProduct updateForm(PurchaseOrderProduct purchaseOrderProduct) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(purchaseOrderProduct);
    // ===================基本信息
    String currentId = purchaseOrderProduct.getId();
    Optional<PurchaseOrderProduct> op_currentPurchaseOrderProduct = this.purchaseOrderProductRepository.findById(currentId);
    PurchaseOrderProduct currentPurchaseOrderProduct = op_currentPurchaseOrderProduct.orElse(null);
    currentPurchaseOrderProduct = Validate.notNull(currentPurchaseOrderProduct, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentPurchaseOrderProduct.setModifyAccount(SecurityUtils.getUserAccount());
    currentPurchaseOrderProduct.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentPurchaseOrderProduct.setExtend1(purchaseOrderProduct.getExtend1());
    currentPurchaseOrderProduct.setExtend2(purchaseOrderProduct.getExtend2());
    currentPurchaseOrderProduct.setExtend3(purchaseOrderProduct.getExtend3());
    currentPurchaseOrderProduct.setExtend4(purchaseOrderProduct.getExtend4());
    currentPurchaseOrderProduct.setExtend5(purchaseOrderProduct.getExtend5());
    currentPurchaseOrderProduct.setExtend6(purchaseOrderProduct.getExtend6());
    currentPurchaseOrderProduct.setExtend7(purchaseOrderProduct.getExtend7());
    currentPurchaseOrderProduct.setExtend8(purchaseOrderProduct.getExtend8());
    currentPurchaseOrderProduct.setExtend9(purchaseOrderProduct.getExtend9());
    currentPurchaseOrderProduct.setExtend10(purchaseOrderProduct.getExtend10());
    currentPurchaseOrderProduct.setExtend11(purchaseOrderProduct.getExtend11());
    currentPurchaseOrderProduct.setTenantCode(purchaseOrderProduct.getTenantCode());
    currentPurchaseOrderProduct.setProductCode(purchaseOrderProduct.getProductCode());
    currentPurchaseOrderProduct.setProductName(purchaseOrderProduct.getProductName());
    currentPurchaseOrderProduct.setMainImagePath(purchaseOrderProduct.getMainImagePath());
    currentPurchaseOrderProduct.setBrandCode(purchaseOrderProduct.getBrandCode());
    currentPurchaseOrderProduct.setBrandName(purchaseOrderProduct.getBrandName());
    currentPurchaseOrderProduct.setProductSpecificationCode(purchaseOrderProduct.getProductSpecificationCode());
    currentPurchaseOrderProduct.setProductSpecificationName(purchaseOrderProduct.getProductSpecificationName());
    currentPurchaseOrderProduct.setUnitCode(purchaseOrderProduct.getUnitCode());
    currentPurchaseOrderProduct.setCategoryCode(purchaseOrderProduct.getCategoryCode());
    currentPurchaseOrderProduct.setCategoryName(purchaseOrderProduct.getCategoryName());
    currentPurchaseOrderProduct.setCategoryFlatCode(purchaseOrderProduct.getCategoryFlatCode());
    currentPurchaseOrderProduct.setBarCode(purchaseOrderProduct.getBarCode());
    currentPurchaseOrderProduct.setQuantity(purchaseOrderProduct.getQuantity());
    currentPurchaseOrderProduct.setPrice(purchaseOrderProduct.getPrice());
    currentPurchaseOrderProduct.setAmount(purchaseOrderProduct.getAmount());
    currentPurchaseOrderProduct.setRemark(purchaseOrderProduct.getRemark());
    currentPurchaseOrderProduct.setPurchaseOrder(purchaseOrderProduct.getPurchaseOrder());

    this.purchaseOrderProductRepository.saveAndFlush(currentPurchaseOrderProduct);
    return currentPurchaseOrderProduct;
  }

  /**
   * 在更新一个已有的PurchaseOrderProduct模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(PurchaseOrderProduct purchaseOrderProduct) {

    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(purchaseOrderProduct.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(purchaseOrderProduct.getProductCode(), "修改信息时，商品编码不能为空！");
    Validate.notBlank(purchaseOrderProduct.getProductName(), "修改信息时，商品名称不能为空！");
    Validate.notBlank(purchaseOrderProduct.getMainImagePath(), "修改信息时，主图路径不能为空！");
    Validate.notBlank(purchaseOrderProduct.getProductSpecificationCode(), "修改信息时，商品（规格）编码不能为空！");
    Validate.notBlank(purchaseOrderProduct.getProductSpecificationName(), "修改信息时，商品（规格）名称不能为空！");
    Validate.notBlank(purchaseOrderProduct.getUnitCode(), "修改信息时，单位不能为空！");
    Validate.notNull(purchaseOrderProduct.getQuantity(), "修改信息时，采购数量不能为空！");
    Validate.notNull(purchaseOrderProduct.getPrice(), "修改信息时，进货单价不能为空！");
    Validate.notNull(purchaseOrderProduct.getAmount(), "修改信息时，小计不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(purchaseOrderProduct.getExtend1() == null || purchaseOrderProduct.getExtend1().length() < 255, "扩展字段1,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend2() == null || purchaseOrderProduct.getExtend2().length() < 255, "扩展字段2,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend3() == null || purchaseOrderProduct.getExtend3().length() < 255, "扩展字段3,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend4() == null || purchaseOrderProduct.getExtend4().length() < 255, "扩展字段4,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend5() == null || purchaseOrderProduct.getExtend5().length() < 255, "扩展字段5,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend6() == null || purchaseOrderProduct.getExtend6().length() < 255, "扩展字段6,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend7() == null || purchaseOrderProduct.getExtend7().length() < 255, "扩展字段7,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getTenantCode() == null || purchaseOrderProduct.getTenantCode().length() < 255, "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getProductCode() == null || purchaseOrderProduct.getProductCode().length() < 64, "商品编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getProductName() == null || purchaseOrderProduct.getProductName().length() < 128, "商品名称,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getMainImagePath() == null || purchaseOrderProduct.getMainImagePath().length() < 255, "主图路径,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getBrandCode() == null || purchaseOrderProduct.getBrandCode().length() < 64, "品牌编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getBrandName() == null || purchaseOrderProduct.getBrandName().length() < 1024, "品牌名称,在进行修改时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getProductSpecificationCode() == null || purchaseOrderProduct.getProductSpecificationCode().length() < 128, "商品（规格）编码,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getProductSpecificationName() == null || purchaseOrderProduct.getProductSpecificationName().length() < 128, "商品（规格）名称,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getUnitCode() == null || purchaseOrderProduct.getUnitCode().length() < 255, "单位,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getCategoryCode() == null || purchaseOrderProduct.getCategoryCode().length() < 255, "分类编码,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getCategoryName() == null || purchaseOrderProduct.getCategoryName().length() < 255, "分类名称,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getCategoryFlatCode() == null || purchaseOrderProduct.getCategoryFlatCode().length() < 255, "快速编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getBarCode() == null || purchaseOrderProduct.getBarCode().length() < 64, "条形码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getRemark() == null || purchaseOrderProduct.getRemark().length() < 255, "备注信息,在进行修改时填入值超过了限定长度(255)，请检查!");

    // 关联性判断，关联属性判断，需要满足ManyToOne或者OneToOne，且not null 且是主模型
  }

  @Override
  public Set<PurchaseOrderProduct> findDetailsByPurchaseOrder(String purchaseOrder) {
    if (StringUtils.isBlank(purchaseOrder)) {
      return Sets.newHashSet();
    }
    return this.purchaseOrderProductRepository.findDetailsByPurchaseOrder(purchaseOrder);
  }

  @Override
  public PurchaseOrderProduct findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.purchaseOrderProductRepository.findDetailsById(id);
  }

  @Override
  public PurchaseOrderProduct findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<PurchaseOrderProduct> op = purchaseOrderProductRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    PurchaseOrderProduct current = this.findById(id);
    if (current != null) {
      this.purchaseOrderProductRepository.delete(current);
    }
  }

  @Override
  @Transactional
  public void save(Set<PurchaseOrderProduct> purchaseOrderProducts, PurchaseOrder purchaseOrder) {
    /**
     * 1.保存数据便捷校验
     * 2.工具类拆分数据
     * 3.写入数据库
     */
    this.saveValidation(purchaseOrderProducts, purchaseOrder);
    purchaseOrderProducts = ObjectUtils.defaultIfNull(purchaseOrderProducts, Sets.newHashSet());
    Set<PurchaseOrderProduct> deletes = Sets.newHashSet();
    Set<PurchaseOrderProduct> updates = Sets.newHashSet();
    Set<PurchaseOrderProduct> creates = Sets.newHashSet();
    Map<String, PurchaseOrderProduct> purchaseOrderProductMap = purchaseOrderProducts.stream().collect(Collectors.toMap(PurchaseOrderProduct::getId, v -> v, (v1, v2) -> v2));
    List<PurchaseOrderProduct> dbPurchaseOrderProduct = purchaseOrderProductRepository.findByOrderId(purchaseOrder.getId());
    nebulaToolkitService.collectionDiscrepancy(purchaseOrderProducts, dbPurchaseOrderProduct, PurchaseOrderProduct::getId, deletes, updates, creates);
    //删除数据
    //解决遗留问题:当在同一事物中 先使用delete 方法删除 在使用save方法保存时，
    //jpa 会先执行insert 在执行delete，该方法会产生唯一索引异常，
    // 解决方法为：使用deleteInBatch方法或者delete之后手动调用flush方法
    if (!CollectionUtils.isEmpty(deletes)) {
      purchaseOrderProductRepository.deleteInBatch(deletes);
    }
    //更新数据
    for (PurchaseOrderProduct update : updates) {
      PurchaseOrderProduct purchaseOrderProduct = purchaseOrderProductMap.get(update.getId());
      this.basicsUpdate(update, purchaseOrderProduct);
      purchaseOrderProductRepository.saveAndFlush(update);
    }
    //新增数据
    Date now = new Date();
    String userAccount = SecurityUtils.getUserAccount();
    for (PurchaseOrderProduct create : creates) {
      create.setCreateAccount(userAccount);
      create.setCreateTime(now);
      create.setModifyAccount(userAccount);
      create.setModifyTime(now);
      create.setPurchaseOrder(purchaseOrder);
      create.setId(null);
      purchaseOrderProductRepository.saveAndFlush(create);
    }

  }

  /**
   * 更新基础数据
   *
   * @param dbPurchaseOrderProduct
   * @param purchaseOrderProduct
   */
  private void basicsUpdate(PurchaseOrderProduct dbPurchaseOrderProduct, PurchaseOrderProduct purchaseOrderProduct) {
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    dbPurchaseOrderProduct.setModifyAccount(SecurityUtils.getUserAccount());
    dbPurchaseOrderProduct.setModifyTime(now);
    // 开始重新赋值——一般属性
    dbPurchaseOrderProduct.setExtend1(purchaseOrderProduct.getExtend1());
    dbPurchaseOrderProduct.setExtend2(purchaseOrderProduct.getExtend2());
    dbPurchaseOrderProduct.setExtend3(purchaseOrderProduct.getExtend3());
    dbPurchaseOrderProduct.setExtend4(purchaseOrderProduct.getExtend4());
    dbPurchaseOrderProduct.setExtend5(purchaseOrderProduct.getExtend5());
    dbPurchaseOrderProduct.setExtend6(purchaseOrderProduct.getExtend6());
    dbPurchaseOrderProduct.setExtend7(purchaseOrderProduct.getExtend7());
    dbPurchaseOrderProduct.setExtend8(purchaseOrderProduct.getExtend8());
    dbPurchaseOrderProduct.setExtend9(purchaseOrderProduct.getExtend9());
    dbPurchaseOrderProduct.setExtend10(purchaseOrderProduct.getExtend10());
    dbPurchaseOrderProduct.setExtend11(purchaseOrderProduct.getExtend11());
    dbPurchaseOrderProduct.setTenantCode(purchaseOrderProduct.getTenantCode());
    dbPurchaseOrderProduct.setProductCode(purchaseOrderProduct.getProductCode());
    dbPurchaseOrderProduct.setProductName(purchaseOrderProduct.getProductName());
    dbPurchaseOrderProduct.setMainImagePath(purchaseOrderProduct.getMainImagePath());
    dbPurchaseOrderProduct.setBrandCode(purchaseOrderProduct.getBrandCode());
    dbPurchaseOrderProduct.setBrandName(purchaseOrderProduct.getBrandName());
    dbPurchaseOrderProduct.setProductSpecificationCode(purchaseOrderProduct.getProductSpecificationCode());
    dbPurchaseOrderProduct.setProductSpecificationName(purchaseOrderProduct.getProductSpecificationName());
    dbPurchaseOrderProduct.setUnitCode(purchaseOrderProduct.getUnitCode());
    dbPurchaseOrderProduct.setCategoryCode(purchaseOrderProduct.getCategoryCode());
    dbPurchaseOrderProduct.setCategoryName(purchaseOrderProduct.getCategoryName());
    dbPurchaseOrderProduct.setCategoryFlatCode(purchaseOrderProduct.getCategoryFlatCode());
    dbPurchaseOrderProduct.setBarCode(purchaseOrderProduct.getBarCode());
    dbPurchaseOrderProduct.setQuantity(purchaseOrderProduct.getQuantity());
    dbPurchaseOrderProduct.setPrice(purchaseOrderProduct.getPrice());
    dbPurchaseOrderProduct.setAmount(purchaseOrderProduct.getAmount());
    dbPurchaseOrderProduct.setRemark(purchaseOrderProduct.getRemark());
    dbPurchaseOrderProduct.setPurchaseOrder(purchaseOrderProduct.getPurchaseOrder());
  }

  private void saveValidation(Set<PurchaseOrderProduct> purchaseOrderProducts, PurchaseOrder purchaseOrder) {
    Validate.notNull(purchaseOrder, "传入的采购订单不能为空");
    Validate.notEmpty(purchaseOrderProducts, "采购订单商品不能为空");
    for (PurchaseOrderProduct purchaseOrderProduct : purchaseOrderProducts) {
      this.bascicsValidation(purchaseOrderProduct);
    }

  }

  private void bascicsValidation(PurchaseOrderProduct purchaseOrderProduct) {
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(purchaseOrderProduct.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(purchaseOrderProduct.getProductCode(), "修改信息时，商品编码不能为空！");
    Validate.notBlank(purchaseOrderProduct.getProductName(), "修改信息时，商品名称不能为空！");
    Validate.notBlank(purchaseOrderProduct.getMainImagePath(), "修改信息时，主图路径不能为空！");
    Validate.notBlank(purchaseOrderProduct.getProductSpecificationCode(), "修改信息时，商品（规格）编码不能为空！");
    Validate.notBlank(purchaseOrderProduct.getProductSpecificationName(), "修改信息时，商品（规格）名称不能为空！");
    Validate.notBlank(purchaseOrderProduct.getUnitCode(), "修改信息时，单位不能为空！");
    Validate.notNull(purchaseOrderProduct.getQuantity(), "修改信息时，采购数量不能为空！");
    Validate.notNull(purchaseOrderProduct.getPrice(), "修改信息时，进货单价不能为空！");
    Validate.notNull(purchaseOrderProduct.getAmount(), "修改信息时，小计不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(purchaseOrderProduct.getExtend1() == null || purchaseOrderProduct.getExtend1().length() < 255, "扩展字段1,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend2() == null || purchaseOrderProduct.getExtend2().length() < 255, "扩展字段2,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend3() == null || purchaseOrderProduct.getExtend3().length() < 255, "扩展字段3,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend4() == null || purchaseOrderProduct.getExtend4().length() < 255, "扩展字段4,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend5() == null || purchaseOrderProduct.getExtend5().length() < 255, "扩展字段5,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend6() == null || purchaseOrderProduct.getExtend6().length() < 255, "扩展字段6,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getExtend7() == null || purchaseOrderProduct.getExtend7().length() < 255, "扩展字段7,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getTenantCode() == null || purchaseOrderProduct.getTenantCode().length() < 255, "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getProductCode() == null || purchaseOrderProduct.getProductCode().length() < 64, "商品编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getProductName() == null || purchaseOrderProduct.getProductName().length() < 128, "商品名称,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getMainImagePath() == null || purchaseOrderProduct.getMainImagePath().length() < 255, "主图路径,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getBrandCode() == null || purchaseOrderProduct.getBrandCode().length() < 64, "品牌编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getBrandName() == null || purchaseOrderProduct.getBrandName().length() < 1024, "品牌名称,在进行修改时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getProductSpecificationCode() == null || purchaseOrderProduct.getProductSpecificationCode().length() < 128, "商品（规格）编码,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getProductSpecificationName() == null || purchaseOrderProduct.getProductSpecificationName().length() < 128, "商品（规格）名称,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getUnitCode() == null || purchaseOrderProduct.getUnitCode().length() < 255, "单位,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getCategoryCode() == null || purchaseOrderProduct.getCategoryCode().length() < 255, "分类编码,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getCategoryName() == null || purchaseOrderProduct.getCategoryName().length() < 255, "分类名称,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getCategoryFlatCode() == null || purchaseOrderProduct.getCategoryFlatCode().length() < 255, "快速编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getBarCode() == null || purchaseOrderProduct.getBarCode().length() < 64, "条形码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrderProduct.getRemark() == null || purchaseOrderProduct.getRemark().length() < 255, "备注信息,在进行修改时填入值超过了限定长度(255)，请检查!");

  }


  @Override
  public List<PurchaseOrderProductVo> findInventoryByOrderId(String orderId) {
    if (StringUtils.isBlank(orderId)) {
      return Lists.newArrayList();
    }
    List<PurchaseOrderProduct> list = purchaseOrderProductRepository.findByOrderId(orderId);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newArrayList();
    }
    List<PurchaseOrderProductVo> productVos = new ArrayList<>(list.size());
    PurchaseOrder purchaseOrder = purchaseOrderService.findById(orderId);
    if (purchaseOrder == null) {
      return Lists.newArrayList();
    }
    //如果完成入库 那么直接返回空列表
    if (PurchaseOrderStatus.COMPLETED.getCode().equals(purchaseOrder.getPurchaseOrderStatus())) {
      return Lists.newArrayList();
    }

    Map<String, BigDecimal> map = warehouseProductsEnterService.findWarehouseProductEnterQuantity(purchaseOrder.getPurchaseOrderCode());
    if (map == null) {
      map = new HashMap<>();
    }
    //过滤掉已经完成出库的商品信息
    Map<String, BigDecimal> finalMap = map;
    list = list.stream().filter(e -> {
      BigDecimal quantity = finalMap.getOrDefault(e.getProductSpecificationCode(), BigDecimal.ZERO);
      if (e.getQuantity().compareTo(quantity) == 0) {
        return false;
      }
      return true;
    }).collect(Collectors.toList());

    for (PurchaseOrderProduct e : list) {
      PurchaseOrderProductVo purchaseOrderProductVo = nebulaToolkitService.copyObjectByWhiteList(e, PurchaseOrderProductVo.class, LinkedHashSet.class, ArrayList.class);
      //页面待入库数量设置
      purchaseOrderProductVo.setRemainingQuantity(e.getQuantity().subtract(map.getOrDefault(e.getProductSpecificationCode(),BigDecimal.ZERO)));
      purchaseOrderProductVo.setWarehouseCode(e.getPurchaseOrder().getWarehouseCode());
      purchaseOrderProductVo.setWarehouseName(e.getPurchaseOrder().getWarehouseName());
      productVos.add(purchaseOrderProductVo);
    }
    return productVos;
  }
}
