package com.bizunited.empower.business.purchase.service.internal;

import com.bizunited.empower.business.common.util.SecurityUtils;
import com.bizunited.empower.business.payment.entity.SupplierPaymentInfo;
import com.bizunited.empower.business.payment.service.SupplierPaymentInfoService;
import com.bizunited.empower.business.purchase.common.enums.PurchaseOrderPayStatus;
import com.bizunited.empower.business.purchase.common.enums.PurchaseOrderStatus;
import com.bizunited.empower.business.purchase.dto.PurchaseOrderConditionDto;
import com.bizunited.empower.business.purchase.entity.PurchaseOrder;
import com.bizunited.empower.business.purchase.entity.PurchaseOrderProduct;
import com.bizunited.empower.business.purchase.repository.PurchaseOrderRepository;
import com.bizunited.empower.business.purchase.repository.internal.PurchaseOrderRepositoryCustom;
import com.bizunited.empower.business.purchase.service.PurchaseOrderFileService;
import com.bizunited.empower.business.purchase.service.PurchaseOrderProductService;
import com.bizunited.empower.business.purchase.service.PurchaseOrderService;
import com.bizunited.empower.business.purchase.vo.PurchaseOrderVo;
import com.bizunited.empower.business.warehouse.entity.WarehouseProductsEnter;
import com.bizunited.empower.business.warehouse.service.WarehouseProductVoService;
import com.bizunited.empower.business.warehouse.service.WarehouseProductsEnterService;
import com.bizunited.empower.business.warehouse.vo.WarehouseProductVo;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * PurchaseOrder业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("PurchaseOrderServiceImpl")
public class PurchaseOrderServiceImpl implements PurchaseOrderService {
  @Autowired
  private PurchaseOrderRepository purchaseOrderRepository;
  @Qualifier("_PurchaseOrderRepositoryImpl")
  @Autowired
  private PurchaseOrderRepositoryCustom purchaseOrderRepositoryCustom;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private PurchaseOrderProductService purchaseOrderProductService;
  @Autowired
  private PurchaseOrderFileService purchaseOrderFileService;

  @Autowired
  private SupplierPaymentInfoService supplierPaymentInfoService;

  @Autowired
  private WarehouseProductsEnterService warehouseProductsEnterService;
  @Autowired
  private WarehouseProductVoService warehouseProductVoService;

  private static final String SUPPLIER_ORDER_CODE_PREFIX = "CG";

  /**
   * 生成供应商编码 redis锁前缀
   */
  private static final String SUPPLIER_ORDER_REDIS_LOCK_CODE = "CG_SUPPLIER_";

  @Transactional
  @Override
  public PurchaseOrder create(PurchaseOrder purchaseOrder) {
    PurchaseOrder current = this.createForm(purchaseOrder);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public PurchaseOrder createForm(PurchaseOrder purchaseOrder) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    Date now = new Date();
    purchaseOrder.setCreateAccount(SecurityUtils.getUserAccount());
    purchaseOrder.setCreateTime(now);
    purchaseOrder.setModifyAccount(SecurityUtils.getUserAccount());
    purchaseOrder.setModifyTime(now);
    String tenantCode = TenantUtils.getTenantCode();
    purchaseOrder.setTenantCode(tenantCode);
    //生成订单编码
    String orderCode = genOrderCode(tenantCode);
    //验证订单编码是否重复
    PurchaseOrder purchaseOrderUQ = purchaseOrderRepository.findByPurchaseOrderCodeAndTenantCode(orderCode, tenantCode);
    Validate.isTrue(Objects.isNull(purchaseOrderUQ), "采购单编码重复生成，请重新点击提交，无需更改页面");

    purchaseOrder.setPurchaseOrderCode(orderCode);
    //初始化状态
    purchaseOrder.setPurchaseOrderStatus(PurchaseOrderStatus.TO_AUDIT.getCode());
    purchaseOrder.setPayStatus(PurchaseOrderPayStatus.OBLIGATION.getCode());

    this.createValidation(purchaseOrder);

    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================

    this.purchaseOrderRepository.save(purchaseOrder);

    //保存采购单商品信息
    purchaseOrderProductService.save(purchaseOrder.getProductSet(), purchaseOrder);
    //保存采购单文件信息
    purchaseOrderFileService.save(purchaseOrder.getPurchaseOrderFiles(), purchaseOrder);

    //采购付款单生成
    this.supplierPaymentInfoService.createByAssociatedCode(purchaseOrder.getSupplierCode(), purchaseOrder.getSupplierName(), purchaseOrder.getPurchaseOrderCode(), purchaseOrder.getTotalAmount());

    // 返回最终处理的结果，里面带有详细的关联信息
    return purchaseOrder;
  }

  private String genOrderCode(String tenantCode) {
    String lockCode = StringUtils.join(SUPPLIER_ORDER_REDIS_LOCK_CODE, tenantCode);
    String atomicNumber = this.redisMutexService.getAndIncrement(lockCode, 1, 6);
    Date now = new Date();
    SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd");
    String dateCode = dateFormat.format(now);
    return StringUtils.join(SUPPLIER_ORDER_CODE_PREFIX, dateCode, atomicNumber);


  }

  /**
   * 在创建一个新的PurchaseOrder模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(PurchaseOrder purchaseOrder) {
    Validate.notNull(purchaseOrder, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(purchaseOrder.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    purchaseOrder.setId(null);
    Validate.notBlank(purchaseOrder.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(purchaseOrder.getPurchaseOrderCode(), "添加信息时，采购单编码不能为空！");
    Validate.notNull(purchaseOrder.getReceiptDate(), "添加信息时，收货日期不能为空！");
    Validate.notBlank(purchaseOrder.getSupplierCode(), "添加信息时，供应商编码不能为空！");
    Validate.notBlank(purchaseOrder.getSupplierName(), "添加信息时，供应商名称不能为空！");
    Validate.notBlank(purchaseOrder.getSalesmanCode(), "添加信息时，业务员编号不能为空！");
    Validate.notBlank(purchaseOrder.getSalesmanName(), "添加信息时，业务员名称不能为空！");
    Validate.notNull(purchaseOrder.getPurchaseOrderStatus(), "添加信息时，采购单状态不能为空！");
    Validate.notNull(purchaseOrder.getPayStatus(), "添加信息时，付款状态不能为空！");
    Validate.notBlank(purchaseOrder.getWarehouseCode(), "添加信息时，仓库编码不能为空！");
    Validate.notBlank(purchaseOrder.getWarehouseName(), "添加信息时，仓库名称不能为空！");
    Validate.notNull(purchaseOrder.getTotalAmount(), "添加信息时，商品总额不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(purchaseOrder.getExtend1() == null || purchaseOrder.getExtend1().length() < 255, "扩展字段1,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrder.getExtend2() == null || purchaseOrder.getExtend2().length() < 255, "扩展字段2,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrder.getExtend3() == null || purchaseOrder.getExtend3().length() < 255, "扩展字段3,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrder.getExtend4() == null || purchaseOrder.getExtend4().length() < 255, "扩展字段4,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrder.getExtend5() == null || purchaseOrder.getExtend5().length() < 255, "扩展字段5,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrder.getExtend6() == null || purchaseOrder.getExtend6().length() < 255, "扩展字段6,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrder.getExtend7() == null || purchaseOrder.getExtend7().length() < 255, "扩展字段7,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrder.getTenantCode() == null || purchaseOrder.getTenantCode().length() < 255, "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrder.getPurchaseOrderCode() == null || purchaseOrder.getPurchaseOrderCode().length() < 64, "采购单编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrder.getSupplierCode() == null || purchaseOrder.getSupplierCode().length() < 64, "供应商编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrder.getSupplierName() == null || purchaseOrder.getSupplierName().length() < 128, "供应商名称,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(purchaseOrder.getSalesmanCode() == null || purchaseOrder.getSalesmanCode().length() < 64, "业务员编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrder.getSalesmanName() == null || purchaseOrder.getSalesmanName().length() < 64, "业务员名称,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrder.getWarehouseCode() == null || purchaseOrder.getWarehouseCode().length() < 64, "仓库编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrder.getWarehouseName() == null || purchaseOrder.getWarehouseName().length() < 128, "仓库名称,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(purchaseOrder.getRemark() == null || purchaseOrder.getRemark().length() < 255, "备注信息,在进行添加时填入值超过了限定长度(255)，请检查!");
  }

  @Transactional
  @Override
  public PurchaseOrder update(PurchaseOrder purchaseOrder) {
    PurchaseOrder current = this.updateForm(purchaseOrder);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public PurchaseOrder updateForm(PurchaseOrder purchaseOrder) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(purchaseOrder);
    // ===================基本信息
    String currentId = purchaseOrder.getId();
    Optional<PurchaseOrder> op_currentPurchaseOrder = this.purchaseOrderRepository.findById(currentId);
    PurchaseOrder currentPurchaseOrder = op_currentPurchaseOrder.orElse(null);
    currentPurchaseOrder = Validate.notNull(currentPurchaseOrder, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentPurchaseOrder.setModifyAccount(SecurityUtils.getUserAccount());
    currentPurchaseOrder.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentPurchaseOrder.setExtend1(purchaseOrder.getExtend1());
    currentPurchaseOrder.setExtend2(purchaseOrder.getExtend2());
    currentPurchaseOrder.setExtend3(purchaseOrder.getExtend3());
    currentPurchaseOrder.setExtend4(purchaseOrder.getExtend4());
    currentPurchaseOrder.setExtend5(purchaseOrder.getExtend5());
    currentPurchaseOrder.setExtend6(purchaseOrder.getExtend6());
    currentPurchaseOrder.setExtend7(purchaseOrder.getExtend7());
    currentPurchaseOrder.setExtend8(purchaseOrder.getExtend8());
    currentPurchaseOrder.setExtend9(purchaseOrder.getExtend9());
    currentPurchaseOrder.setExtend10(purchaseOrder.getExtend10());
    currentPurchaseOrder.setExtend11(purchaseOrder.getExtend11());
    currentPurchaseOrder.setTenantCode(purchaseOrder.getTenantCode());
    currentPurchaseOrder.setPurchaseOrderCode(purchaseOrder.getPurchaseOrderCode());
    currentPurchaseOrder.setReceiptDate(purchaseOrder.getReceiptDate());
    currentPurchaseOrder.setSupplierCode(purchaseOrder.getSupplierCode());
    currentPurchaseOrder.setSupplierName(purchaseOrder.getSupplierName());
    currentPurchaseOrder.setSalesmanCode(purchaseOrder.getSalesmanCode());
    currentPurchaseOrder.setSalesmanName(purchaseOrder.getSalesmanName());
    currentPurchaseOrder.setPurchaseOrderStatus(purchaseOrder.getPurchaseOrderStatus());
    currentPurchaseOrder.setPayStatus(purchaseOrder.getPayStatus());
    currentPurchaseOrder.setWarehouseCode(purchaseOrder.getWarehouseCode());
    currentPurchaseOrder.setWarehouseName(purchaseOrder.getWarehouseName());
    currentPurchaseOrder.setTotalAmount(purchaseOrder.getTotalAmount());
    currentPurchaseOrder.setRemark(purchaseOrder.getRemark());

    this.purchaseOrderRepository.saveAndFlush(currentPurchaseOrder);
    return currentPurchaseOrder;
  }

  /**
   * 在更新一个已有的PurchaseOrder模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(PurchaseOrder purchaseOrder) {
    Validate.isTrue(!StringUtils.isBlank(purchaseOrder.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");

    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(purchaseOrder.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(purchaseOrder.getPurchaseOrderCode(), "修改信息时，采购单编码不能为空！");
    Validate.notNull(purchaseOrder.getReceiptDate(), "修改信息时，收货日期不能为空！");
    Validate.notBlank(purchaseOrder.getSupplierCode(), "修改信息时，供应商编码不能为空！");
    Validate.notBlank(purchaseOrder.getSupplierName(), "修改信息时，供应商名称不能为空！");
    Validate.notBlank(purchaseOrder.getSalesmanCode(), "修改信息时，业务员编号不能为空！");
    Validate.notBlank(purchaseOrder.getSalesmanName(), "修改信息时，业务员名称不能为空！");
    Validate.notNull(purchaseOrder.getPurchaseOrderStatus(), "修改信息时，采购单状态不能为空！");
    Validate.notNull(purchaseOrder.getPayStatus(), "修改信息时，付款状态不能为空！");
    Validate.notBlank(purchaseOrder.getWarehouseCode(), "修改信息时，仓库编码不能为空！");
    Validate.notBlank(purchaseOrder.getWarehouseName(), "修改信息时，仓库名称不能为空！");
    Validate.notNull(purchaseOrder.getTotalAmount(), "修改信息时，商品总额不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(purchaseOrder.getExtend1() == null || purchaseOrder.getExtend1().length() < 255, "扩展字段1,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrder.getExtend2() == null || purchaseOrder.getExtend2().length() < 255, "扩展字段2,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrder.getExtend3() == null || purchaseOrder.getExtend3().length() < 255, "扩展字段3,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrder.getExtend4() == null || purchaseOrder.getExtend4().length() < 255, "扩展字段4,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrder.getExtend5() == null || purchaseOrder.getExtend5().length() < 255, "扩展字段5,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrder.getExtend6() == null || purchaseOrder.getExtend6().length() < 255, "扩展字段6,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrder.getExtend7() == null || purchaseOrder.getExtend7().length() < 255, "扩展字段7,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrder.getTenantCode() == null || purchaseOrder.getTenantCode().length() < 255, "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseOrder.getPurchaseOrderCode() == null || purchaseOrder.getPurchaseOrderCode().length() < 64, "采购单编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrder.getSupplierCode() == null || purchaseOrder.getSupplierCode().length() < 64, "供应商编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrder.getSupplierName() == null || purchaseOrder.getSupplierCode().length() < 128, "供应商名称,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(purchaseOrder.getSalesmanCode() == null || purchaseOrder.getSalesmanCode().length() < 64, "业务员编号,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrder.getSalesmanName() == null || purchaseOrder.getSalesmanName().length() < 64, "业务员名称,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrder.getWarehouseCode() == null || purchaseOrder.getWarehouseCode().length() < 64, "仓库编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseOrder.getWarehouseName() == null || purchaseOrder.getWarehouseName().length() < 128, "仓库名称,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(purchaseOrder.getRemark() == null || purchaseOrder.getRemark().length() < 255, "备注信息,在进行修改时填入值超过了限定长度(255)，请检查!");
  }

  @Override
  public PurchaseOrder findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    PurchaseOrder detailsById = this.purchaseOrderRepository.findDetailsById(id);
    //丰富库存数量

    if (!CollectionUtils.isEmpty(detailsById.getProductSet())) {
      List<String> ids = detailsById.getProductSet().stream().map(PurchaseOrderProduct::getProductSpecificationCode).distinct().collect(Collectors.toList());
      List<WarehouseProductVo> warehouseProducts = warehouseProductVoService.findByProductSpecificationCodeList(ids);
      Map<String, WarehouseProductVo> warehouseProductHashMap = new HashMap<>();
      if (!CollectionUtils.isEmpty(warehouseProducts)) {
        warehouseProductHashMap = warehouseProducts.stream().collect(Collectors.toMap(WarehouseProductVo::getProductSpecificationCode, Function.identity()));
      }
      for (PurchaseOrderProduct p : detailsById.getProductSet()) {
        p.setUsableInventory(warehouseProductHashMap.containsKey(p.getProductSpecificationCode()) ? warehouseProductHashMap.get(p.getProductSpecificationCode()).getUsableInventory() : new BigDecimal(0));
      }
    }
    return detailsById;
  }

  @Override
  public PurchaseOrder findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<PurchaseOrder> op = purchaseOrderRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    PurchaseOrder current = this.findById(id);
    if (current != null) {
      this.purchaseOrderRepository.delete(current);
    }
  }

  @Override
  public Page<PurchaseOrderVo> findByConditions(Pageable pageable, PurchaseOrderConditionDto purchaseOrderConditionDto) {
    if (purchaseOrderConditionDto == null) {
      purchaseOrderConditionDto = new PurchaseOrderConditionDto();
    }
    purchaseOrderConditionDto.setTenantCode(TenantUtils.getTenantCode());
    Page<PurchaseOrder> purchaseOrders = purchaseOrderRepositoryCustom.queryPage(pageable, purchaseOrderConditionDto);
    if (purchaseOrders.isEmpty()) {
      return Page.empty(pageable);
    }
    //转换
    List<PurchaseOrder> content = purchaseOrders.getContent();

    Collection<PurchaseOrderVo> purchaseOrderVos = nebulaToolkitService.copyCollectionByWhiteList(content, PurchaseOrder.class, PurchaseOrderVo.class, HashSet.class, ArrayList.class);
    ArrayList<PurchaseOrderVo> list = Lists.newArrayList(purchaseOrderVos);
    return new PageImpl<>(list, pageable, purchaseOrders.getTotalElements());
  }

  @Override
  @Transactional
  public void approve(String id) {
    Validate.notBlank(id, "ID不能为空");
    PurchaseOrder purchaseOrder = this.findById(id);
    Validate.notNull(purchaseOrder, "采购单信息不存在");
    Validate.isTrue(PurchaseOrderStatus.TO_AUDIT.getCode().equals(purchaseOrder.getPurchaseOrderStatus()), "当前状态不能审批");
    purchaseOrder.setPurchaseOrderStatus(PurchaseOrderStatus.STOCK_PENDING.getCode());
    this.purchaseOrderRepository.saveAndFlush(purchaseOrder);

  }

  @Override
  @Transactional
  public void revocation(String id) {
    /**
     * 撤回审批：
     * 1.验证状态： 必须为审批通过且为 未付款状态的采购单能够撤回审批
     * 2.验证出库与付款：必须未创建出库单和付款记录 才能撤回
     * 3.撤回状态修改 ，并且取消创建的付款单
     */

    Validate.notBlank(id, "ID不能为空");
    PurchaseOrder purchaseOrder = this.findById(id);
    Validate.notNull(purchaseOrder, "采购单信息不存在");
    Validate.isTrue(PurchaseOrderStatus.STOCK_PENDING.getCode().equals(purchaseOrder.getPurchaseOrderStatus()) && PurchaseOrderPayStatus.OBLIGATION.getCode().equals(purchaseOrder.getPayStatus()), "当前状态不能撤回");
    //根据当前当好获取入库单列表
    List<WarehouseProductsEnter> warehouseProductsEnters = warehouseProductsEnterService.findDetailsByRelevanceCode(purchaseOrder.getPurchaseOrderCode());
    Validate.isTrue(CollectionUtils.isEmpty(warehouseProductsEnters), "采购单已出库不能撤回");

    //获取付款单列表
    SupplierPaymentInfo supplierPaymentInfo = supplierPaymentInfoService.findByAssociatedCode(purchaseOrder.getPurchaseOrderCode());

    Validate.isTrue(supplierPaymentInfo.getPaymentStatus() == 1, "采购单已付款不能撤回");

    purchaseOrder.setPurchaseOrderStatus(PurchaseOrderStatus.TO_AUDIT.getCode());
    this.purchaseOrderRepository.saveAndFlush(purchaseOrder);


  }

  @Override
  public PurchaseOrder findByCode(String code) {
    if (StringUtils.isBlank(code)) {
      return null;
    }
    return purchaseOrderRepository.findByPurchaseOrderCode(code);
  }


  @Override
  @Transactional
  public void cancel(String id) {
    /**
     * 取消采购订单
     * 1.验证 采购单状态是否为 未审核、如果已经审核则不能取消
     * 3.修改当前订单状态 为已取消
     */
    Validate.notBlank(id, "采购单信息不能为空");
    PurchaseOrder purchaseOrder = this.findById(id);
    Validate.notNull(purchaseOrder, "采购单信息不能为空");
    Validate.isTrue(purchaseOrder.getPurchaseOrderStatus().equals(PurchaseOrderStatus.TO_AUDIT.getCode()), "当前状态不能取消");
    purchaseOrder.setPurchaseOrderStatus(PurchaseOrderStatus.CANCELED.getCode());
    this.purchaseOrderRepository.save(purchaseOrder);
    //取消付款单
    supplierPaymentInfoService.cancelByAssociatedCode(purchaseOrder.getPurchaseOrderCode());
  }
}
