package com.bizunited.empower.business.purchase.service.internal;

import com.bizunited.empower.business.common.util.SecurityUtils;
import com.bizunited.empower.business.payment.entity.SupplierReceivableInfo;
import com.bizunited.empower.business.payment.service.SupplierReceivableInfoService;
import com.bizunited.empower.business.purchase.common.enums.PurchaseOrderStatus;
import com.bizunited.empower.business.purchase.common.enums.PurchaseReturnOrderRefundStatus;
import com.bizunited.empower.business.purchase.common.enums.PurchaseReturnOrderStatus;
import com.bizunited.empower.business.purchase.dto.PurchaseReturnOrderConditionDto;
import com.bizunited.empower.business.purchase.entity.PurchaseReturnOrder;
import com.bizunited.empower.business.purchase.entity.PurchaseReturnOrderProduct;
import com.bizunited.empower.business.purchase.entity.SupplierProduct;
import com.bizunited.empower.business.purchase.repository.PurchaseReturnOrderRepository;
import com.bizunited.empower.business.purchase.repository.SupplierProductRepository;
import com.bizunited.empower.business.purchase.repository.internal.PurchaseReturnOrderRepositoryCustom;
import com.bizunited.empower.business.purchase.service.PurchaseReturnOrderFileService;
import com.bizunited.empower.business.purchase.service.PurchaseReturnOrderProductService;
import com.bizunited.empower.business.purchase.service.PurchaseReturnOrderService;
import com.bizunited.empower.business.purchase.vo.PurchaseReturnOrderVo;
import com.bizunited.empower.business.warehouse.entity.WarehouseProductsExpense;
import com.bizunited.empower.business.warehouse.service.WarehouseProductVoService;
import com.bizunited.empower.business.warehouse.service.WarehouseProductsExpenseService;
import com.bizunited.empower.business.warehouse.service.WarehouseProductsExpenseVoService;
import com.bizunited.empower.business.warehouse.vo.WarehouseProductVo;
import com.bizunited.empower.business.warehouse.vo.WarehouseProductsExpenseProductVo;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * PurchaseReturnOrder业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("PurchaseReturnOrderServiceImpl")
public class PurchaseReturnOrderServiceImpl implements PurchaseReturnOrderService {
  @Autowired
  private PurchaseReturnOrderRepository purchaseReturnOrderRepository;
  @Autowired
  @Qualifier("_PurchaseReturnOrderRepositoryImpl")
  private PurchaseReturnOrderRepositoryCustom purchaseReturnOrderRepositoryCustom;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private PurchaseReturnOrderProductService purchaseReturnOrderProductService;
  @Autowired
  private PurchaseReturnOrderFileService purchaseReturnOrderFileService;
  @Autowired
  private SupplierReceivableInfoService supplierReceivableInfoService;
  @Autowired
  private WarehouseProductsExpenseVoService warehouseProductsExpenseVoService;
  @Autowired
  private WarehouseProductVoService warehouseProductVoService;
  @Autowired
  private SupplierProductRepository supplierProductRepository;

  @Autowired
  private WarehouseProductsExpenseService warehouseProductsExpenseService;

  private static final String SUPPLIER_RETURN_ORDER_CODE_PREFIX = "CT";

  /**
   * 生成供应商编码 redis锁前缀
   */
  private static final String SUPPLIER_RETURN_ORDER_REDIS_LOCK_CODE = "CT_SUPPLIER_RETURN_";

  @Transactional
  @Override
  public PurchaseReturnOrder create(PurchaseReturnOrder purchaseReturnOrder) {
    PurchaseReturnOrder current = this.createForm(purchaseReturnOrder);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public PurchaseReturnOrder createForm(PurchaseReturnOrder purchaseReturnOrder) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    Date now = new Date();
    purchaseReturnOrder.setCreateAccount(SecurityUtils.getUserAccount());
    purchaseReturnOrder.setCreateTime(now);
    purchaseReturnOrder.setModifyAccount(SecurityUtils.getUserAccount());
    purchaseReturnOrder.setModifyTime(now);
    String tenantCode = TenantUtils.getTenantCode();
    purchaseReturnOrder.setTenantCode(tenantCode);
    //生成采购退单编码
    String returnOrderCode = genRefundCode(tenantCode);
    purchaseReturnOrder.setPurchaseReturnOrderCode(tenantCode);
    //编码验重
    PurchaseReturnOrder uq = purchaseReturnOrderRepository.findByPurchaseReturnOrderCodeAndTenantCode(returnOrderCode, tenantCode);
    Validate.isTrue(Objects.isNull(uq), "采购退单编码重复，请重新提交");
    purchaseReturnOrder.setPurchaseReturnOrderCode(returnOrderCode);

    purchaseReturnOrder.setPurchaseReturnOrderStatus(PurchaseReturnOrderStatus.TO_AUDIT.getCode());
    purchaseReturnOrder.setRefundStatus(PurchaseReturnOrderRefundStatus.TO_BE_REFUND.getCode());
    this.createValidation(purchaseReturnOrder);
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================

    this.purchaseReturnOrderRepository.save(purchaseReturnOrder);

    //保存商品信息
    purchaseReturnOrderProductService.save(purchaseReturnOrder.getProductSet(), purchaseReturnOrder);
    //保存文件信息
    purchaseReturnOrderFileService.save(purchaseReturnOrder.getPurchaseOrderFiles(), purchaseReturnOrder);
    //创建 生成 收款单

    this.supplierReceivableInfoService.createByAssociatedCode(purchaseReturnOrder.getSupplierCode(), purchaseReturnOrder.getSupplierName(), purchaseReturnOrder.getPurchaseReturnOrderCode(), purchaseReturnOrder.getTotalAmount());

    return purchaseReturnOrder;
  }

  /**
   * 在创建一个新的PurchaseReturnOrder模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(PurchaseReturnOrder purchaseReturnOrder) {
    Validate.notNull(purchaseReturnOrder, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(purchaseReturnOrder.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    purchaseReturnOrder.setId(null);
    Validate.notBlank(purchaseReturnOrder.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(purchaseReturnOrder.getPurchaseReturnOrderCode(), "添加信息时，退货单编码不能为空！");
    Validate.notBlank(purchaseReturnOrder.getSupplierCode(), "添加信息时，供应商编码不能为空！");
    Validate.notBlank(purchaseReturnOrder.getSupplierName(), "添加信息时，供应商名称不能为空！");
    Validate.notNull(purchaseReturnOrder.getPurchaseReturnOrderStatus(), "添加信息时，退货单状态不能为空！");
    Validate.notNull(purchaseReturnOrder.getTotalAmount(), "添加信息时，退货总额不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(purchaseReturnOrder.getExtend1() == null || purchaseReturnOrder.getExtend1().length() < 255, "扩展字段1,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getExtend2() == null || purchaseReturnOrder.getExtend2().length() < 255, "扩展字段2,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getExtend3() == null || purchaseReturnOrder.getExtend3().length() < 255, "扩展字段3,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getExtend4() == null || purchaseReturnOrder.getExtend4().length() < 255, "扩展字段4,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getExtend5() == null || purchaseReturnOrder.getExtend5().length() < 255, "扩展字段5,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getExtend6() == null || purchaseReturnOrder.getExtend6().length() < 255, "扩展字段6,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getExtend7() == null || purchaseReturnOrder.getExtend7().length() < 255, "扩展字段7,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getTenantCode() == null || purchaseReturnOrder.getTenantCode().length() < 255, "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getPurchaseReturnOrderCode() == null || purchaseReturnOrder.getPurchaseReturnOrderCode().length() < 64, "退货单编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getSupplierCode() == null || purchaseReturnOrder.getSupplierCode().length() < 64, "供应商编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getSupplierName() == null || purchaseReturnOrder.getSupplierName().length() < 128, "供应商名称,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getRemark() == null || purchaseReturnOrder.getRemark().length() < 255, "备注信息,在进行添加时填入值超过了限定长度(255)，请检查!");
  }

  private String genRefundCode(String tenantCode) {
    String lockCode = StringUtils.join(SUPPLIER_RETURN_ORDER_REDIS_LOCK_CODE, tenantCode);
    String atomicNumber = this.redisMutexService.getAndIncrement(lockCode, 1, 6);
    Date now = new Date();
    SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd");
    String dateCode = dateFormat.format(now);
    return StringUtils.join(SUPPLIER_RETURN_ORDER_CODE_PREFIX, dateCode, atomicNumber);
  }


  @Transactional
  @Override
  public PurchaseReturnOrder update(PurchaseReturnOrder purchaseReturnOrder) {
    PurchaseReturnOrder current = this.updateForm(purchaseReturnOrder);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public PurchaseReturnOrder updateForm(PurchaseReturnOrder purchaseReturnOrder) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(purchaseReturnOrder);
    // ===================基本信息
    String currentId = purchaseReturnOrder.getId();
    Optional<PurchaseReturnOrder> op_currentPurchaseReturnOrder = this.purchaseReturnOrderRepository.findById(currentId);
    PurchaseReturnOrder currentPurchaseReturnOrder = op_currentPurchaseReturnOrder.orElse(null);
    currentPurchaseReturnOrder = Validate.notNull(currentPurchaseReturnOrder, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentPurchaseReturnOrder.setModifyAccount(SecurityUtils.getUserAccount());
    currentPurchaseReturnOrder.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentPurchaseReturnOrder.setExtend1(purchaseReturnOrder.getExtend1());
    currentPurchaseReturnOrder.setExtend2(purchaseReturnOrder.getExtend2());
    currentPurchaseReturnOrder.setExtend3(purchaseReturnOrder.getExtend3());
    currentPurchaseReturnOrder.setExtend4(purchaseReturnOrder.getExtend4());
    currentPurchaseReturnOrder.setExtend5(purchaseReturnOrder.getExtend5());
    currentPurchaseReturnOrder.setExtend6(purchaseReturnOrder.getExtend6());
    currentPurchaseReturnOrder.setExtend7(purchaseReturnOrder.getExtend7());
    currentPurchaseReturnOrder.setExtend8(purchaseReturnOrder.getExtend8());
    currentPurchaseReturnOrder.setExtend9(purchaseReturnOrder.getExtend9());
    currentPurchaseReturnOrder.setExtend10(purchaseReturnOrder.getExtend10());
    currentPurchaseReturnOrder.setExtend11(purchaseReturnOrder.getExtend11());
    currentPurchaseReturnOrder.setTenantCode(purchaseReturnOrder.getTenantCode());
    currentPurchaseReturnOrder.setPurchaseReturnOrderCode(purchaseReturnOrder.getPurchaseReturnOrderCode());
    currentPurchaseReturnOrder.setSupplierCode(purchaseReturnOrder.getSupplierCode());
    currentPurchaseReturnOrder.setSupplierName(purchaseReturnOrder.getSupplierName());
    currentPurchaseReturnOrder.setPurchaseReturnOrderStatus(purchaseReturnOrder.getPurchaseReturnOrderStatus());
    currentPurchaseReturnOrder.setTotalAmount(purchaseReturnOrder.getTotalAmount());
    currentPurchaseReturnOrder.setRemark(purchaseReturnOrder.getRemark());

    this.purchaseReturnOrderRepository.saveAndFlush(currentPurchaseReturnOrder);
    return currentPurchaseReturnOrder;
  }

  /**
   * 在更新一个已有的PurchaseReturnOrder模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(PurchaseReturnOrder purchaseReturnOrder) {
    Validate.isTrue(!StringUtils.isBlank(purchaseReturnOrder.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");

    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(purchaseReturnOrder.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(purchaseReturnOrder.getPurchaseReturnOrderCode(), "修改信息时，退货单编码不能为空！");
    Validate.notBlank(purchaseReturnOrder.getSupplierCode(), "修改信息时，供应商编码不能为空！");
    Validate.notBlank(purchaseReturnOrder.getSupplierName(), "修改信息时，供应商名称不能为空！");
    Validate.notNull(purchaseReturnOrder.getPurchaseReturnOrderStatus(), "修改信息时，退货单状态不能为空！");
    Validate.notNull(purchaseReturnOrder.getTotalAmount(), "修改信息时，退货总额不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(purchaseReturnOrder.getExtend1() == null || purchaseReturnOrder.getExtend1().length() < 255, "扩展字段1,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getExtend2() == null || purchaseReturnOrder.getExtend2().length() < 255, "扩展字段2,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getExtend3() == null || purchaseReturnOrder.getExtend3().length() < 255, "扩展字段3,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getExtend4() == null || purchaseReturnOrder.getExtend4().length() < 255, "扩展字段4,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getExtend5() == null || purchaseReturnOrder.getExtend5().length() < 255, "扩展字段5,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getExtend6() == null || purchaseReturnOrder.getExtend6().length() < 255, "扩展字段6,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getExtend7() == null || purchaseReturnOrder.getExtend7().length() < 255, "扩展字段7,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getTenantCode() == null || purchaseReturnOrder.getTenantCode().length() < 255, "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getPurchaseReturnOrderCode() == null || purchaseReturnOrder.getPurchaseReturnOrderCode().length() < 64, "退货单编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getSupplierCode() == null || purchaseReturnOrder.getSupplierCode().length() < 64, "供应商编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getSupplierName() == null || purchaseReturnOrder.getSupplierName().length() < 128, "供应商名称,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(purchaseReturnOrder.getRemark() == null || purchaseReturnOrder.getRemark().length() < 255, "备注信息,在进行修改时填入值超过了限定长度(255)，请检查!");
  }

  @Override
  public PurchaseReturnOrder findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    PurchaseReturnOrder detailsById = this.purchaseReturnOrderRepository.findDetailsById(id);
    if (!CollectionUtils.isEmpty(detailsById.getProductSet())) {
      List<String> ids = detailsById.getProductSet().stream().map(PurchaseReturnOrderProduct::getProductSpecificationCode).distinct().collect(Collectors.toList());
      List<WarehouseProductVo> productSpecificationVos = warehouseProductVoService.findByProductSpecificationCodeList(ids);
      Map<String, WarehouseProductVo> warehouseProductHashMap = new HashMap<>();
      if (!CollectionUtils.isEmpty(productSpecificationVos)) {
        warehouseProductHashMap = productSpecificationVos.stream().collect(Collectors.toMap(WarehouseProductVo::getProductSpecificationCode, Function.identity()));
      }
      Map<String, SupplierProduct> supplierProductMap = new HashMap<>();
      Set<SupplierProduct> supplierProducts = supplierProductRepository.findBySupplierCodeAndTenantCode(detailsById.getSupplierCode(), TenantUtils.getTenantCode());
      if (!CollectionUtils.isEmpty(supplierProducts)) {
        supplierProductMap = supplierProducts.stream().collect(Collectors.toMap(SupplierProduct::getProductSpecificationCode, Function.identity()));
      }
      for (PurchaseReturnOrderProduct p : detailsById.getProductSet()) {
        p.setUsableInventory(warehouseProductHashMap.containsKey(p.getProductSpecificationCode()) ? warehouseProductHashMap.get(p.getProductSpecificationCode()).getUsableInventory() : BigDecimal.ZERO);
        p.setReferencePrice(supplierProductMap.containsKey(p.getProductSpecificationCode()) ? supplierProductMap.get(p.getProductSpecificationCode()).getReferencePrice() : BigDecimal.ZERO);
      }
    }
    return detailsById;
  }

  @Override
  public PurchaseReturnOrder findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<PurchaseReturnOrder> op = purchaseReturnOrderRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    PurchaseReturnOrder current = this.findById(id);
    if (current != null) {
      this.purchaseReturnOrderRepository.delete(current);
    }
  }

  @Override
  public Page<PurchaseReturnOrderVo> findByConditions(Pageable pageable, PurchaseReturnOrderConditionDto purchaseReturnOrderConditionDto) {
    purchaseReturnOrderConditionDto.setTenantCode(TenantUtils.getTenantCode());
    Page<PurchaseReturnOrder> purchaseReturnOrders = purchaseReturnOrderRepositoryCustom.queryPage(pageable, purchaseReturnOrderConditionDto);
    if (purchaseReturnOrders.isEmpty()) {
      return Page.empty(pageable);
    }
    List<PurchaseReturnOrder> content = purchaseReturnOrders.getContent();
    List<PurchaseReturnOrderVo> list = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(content, PurchaseReturnOrder.class, PurchaseReturnOrderVo.class, HashSet.class, ArrayList.class));
    //丰富页面出库仓库字段
    if (!CollectionUtils.isEmpty(list)) {
      List<String> codes = list.stream().map(PurchaseReturnOrderVo::getPurchaseReturnOrderCode).collect(Collectors.toList());
      //获取到订单列表出库单据集合
      List<WarehouseProductsExpense> warehouseProductsExpenses = warehouseProductsExpenseService.findByExpenseCodeList(codes);
      if (!CollectionUtils.isEmpty(warehouseProductsExpenses)) {
        //出库单据集合组装每条出库单据的仓库信息
        Map<String, String> warehouseMap = new HashMap<>();
        for (WarehouseProductsExpense warehouseProductsExpens : warehouseProductsExpenses) {
          if (warehouseMap.containsKey(warehouseProductsExpens.getRelevanceCode())) {
            //验证是否已经包含了仓库名称
            String warehouseName = warehouseMap.get(warehouseProductsExpens.getRelevanceCode());
            //如果已经包含了当前仓库名称，那么直接跳过
            if (warehouseName.contains(warehouseProductsExpens.getWarehouseInfo().getWarehouseName())) {
              continue;
            }
            warehouseMap.put(warehouseProductsExpens.getRelevanceCode(), warehouseName + "," + warehouseProductsExpens.getWarehouseInfo().getWarehouseName());
          } else {
            warehouseMap.put(warehouseProductsExpens.getRelevanceCode(), warehouseProductsExpens.getWarehouseInfo().getWarehouseName());
          }
        }
        //丰富页面出库仓库字段
        for (PurchaseReturnOrderVo purchaseReturnOrderVo : list) {
          if (warehouseMap.containsKey(purchaseReturnOrderVo.getPurchaseReturnOrderCode())) {
            purchaseReturnOrderVo.setWarehouseName(warehouseMap.get(purchaseReturnOrderVo.getPurchaseReturnOrderCode()));
          }
        }
      }
    }
    return new PageImpl<>(list, pageable, purchaseReturnOrders.getTotalElements());
  }

  @Override
  @Transactional
  public void approve(String id) {
    Validate.notBlank(id, "ID不能为空");
    PurchaseReturnOrder purchaseReturnOrder = this.findById(id);
    Validate.notNull(purchaseReturnOrder, "采购退单信息不存在");
    Validate.isTrue(PurchaseOrderStatus.TO_AUDIT.getCode().equals(purchaseReturnOrder.getPurchaseReturnOrderStatus()), "当前状态不能审批");
    purchaseReturnOrder.setPurchaseReturnOrderStatus(PurchaseOrderStatus.STOCK_PENDING.getCode());
    this.purchaseReturnOrderRepository.saveAndFlush(purchaseReturnOrder);
  }

  @Override
  @Transactional
  public void revocation(String id) {
    /**
     * 撤回采购退单
     * 1.状态必须为已审批 且未收款 才能撤回
     * 2.入库单信息必须为空 且收款单信息必须为空才能撤回
     * 3.取消收款单信息
     */
    Validate.notBlank(id, "ID不能为空");
    PurchaseReturnOrder purchaseReturnOrder = this.findById(id);
    Validate.notNull(purchaseReturnOrder, "采购退单信息不存在");
    Validate.isTrue(PurchaseOrderStatus.STOCK_PENDING.getCode().equals(purchaseReturnOrder.getPurchaseReturnOrderStatus()) && PurchaseReturnOrderRefundStatus.TO_BE_REFUND.getCode().equals(purchaseReturnOrder.getRefundStatus()), "当前状态不能撤回");
    //获取出库单列表
    List<WarehouseProductsExpenseProductVo> preemptQuantityByRelevanceCode = warehouseProductsExpenseVoService.findPreemptQuantityByRelevanceCode(purchaseReturnOrder.getPurchaseReturnOrderCode());
    Validate.isTrue(CollectionUtils.isEmpty(preemptQuantityByRelevanceCode), "采购退单已经出库不能撤回");
    //获取收款单信息
    SupplierReceivableInfo supplierReceivableInfo = supplierReceivableInfoService.findByAssociatedCode(purchaseReturnOrder.getPurchaseReturnOrderCode());
    Validate.isTrue(supplierReceivableInfo.getReceivableStatus() == 1, "采购退单已收款不能撤回");

    purchaseReturnOrder.setPurchaseReturnOrderStatus(PurchaseOrderStatus.TO_AUDIT.getCode());
    this.purchaseReturnOrderRepository.saveAndFlush(purchaseReturnOrder);

  }

  @Override
  @Transactional
  public void cancel(String id) {
    Validate.notBlank(id, "ID不能为空");
    PurchaseReturnOrder purchaseReturnOrder = this.findById(id);
    Validate.notNull(purchaseReturnOrder, "采购退单信息不存在");
    Validate.isTrue(PurchaseOrderStatus.TO_AUDIT.getCode().equals(purchaseReturnOrder.getPurchaseReturnOrderStatus()), "当前状态不能取消");
    purchaseReturnOrder.setPurchaseReturnOrderStatus(PurchaseOrderStatus.CANCELED.getCode());
    this.purchaseReturnOrderRepository.saveAndFlush(purchaseReturnOrder);
    //取消成功 取消收款单据
    supplierReceivableInfoService.cancelByAssociatedCode(purchaseReturnOrder.getPurchaseReturnOrderCode());

  }
}
