package com.bizunited.empower.business.purchase.service.internal;

import com.bizunited.empower.business.common.util.SecurityUtils;
import com.bizunited.empower.business.product.service.ProductSpecificationVoService;
import com.bizunited.empower.business.product.vo.ProductSpecificationVo;
import com.bizunited.empower.business.purchase.entity.Supplier;
import com.bizunited.empower.business.purchase.entity.SupplierProduct;
import com.bizunited.empower.business.purchase.repository.SupplierProductRepository;
import com.bizunited.empower.business.purchase.service.SupplierProductService;
import com.bizunited.empower.business.purchase.vo.SupplierPurchaseProductVo;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.platform.script.context.InvokeParams;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * SupplierProduct业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("SupplierProductServiceImpl")
public class SupplierProductServiceImpl implements SupplierProductService {
  @Autowired
  private SupplierProductRepository supplierProductRepository;
  @Autowired
  private ProductSpecificationVoService productSpecificationVoService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Transactional
  @Override
  public SupplierProduct create(SupplierProduct supplierProduct) {
    SupplierProduct current = this.createForm(supplierProduct);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public SupplierProduct createForm(SupplierProduct supplierProduct) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    Date now = new Date();
    supplierProduct.setCreateAccount(SecurityUtils.getUserAccount());
    supplierProduct.setCreateTime(now);
    supplierProduct.setModifyAccount(SecurityUtils.getUserAccount());
    supplierProduct.setModifyTime(now);

    this.createValidation(supplierProduct);

    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================

    this.supplierProductRepository.save(supplierProduct);
    // 返回最终处理的结果，里面带有详细的关联信息
    return supplierProduct;
  }

  /**
   * 在创建一个新的SupplierProduct模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(SupplierProduct supplierProduct) {
    Validate.notNull(supplierProduct, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(supplierProduct.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    supplierProduct.setId(null);
    Validate.notBlank(supplierProduct.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(supplierProduct.getProductCode(), "添加信息时，商品编码不能为空！");
    Validate.notBlank(supplierProduct.getProductName(), "添加信息时，商品名称不能为空！");
    Validate.notNull(supplierProduct.getShelfStatus(), "添加信息时，上架状态不能为空！");
    Validate.notBlank(supplierProduct.getMainImagePath(), "添加信息时，主图路径不能为空！");
    Validate.notBlank(supplierProduct.getProductSpecificationCode(), "添加信息时，商品（规格）编码不能为空！");
    Validate.notBlank(supplierProduct.getProductSpecificationName(), "添加信息时，商品（规格）名称不能为空！");
    Validate.notBlank(supplierProduct.getUnitCode(), "添加信息时，单位不能为空！");
    Validate.notNull(supplierProduct.getReferencePrice(), "添加信息时，订货参考价不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(supplierProduct.getExtend1() == null || supplierProduct.getExtend1().length() < 255, "扩展字段1,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getExtend2() == null || supplierProduct.getExtend2().length() < 255, "扩展字段2,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getExtend3() == null || supplierProduct.getExtend3().length() < 255, "扩展字段3,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getExtend4() == null || supplierProduct.getExtend4().length() < 255, "扩展字段4,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getExtend5() == null || supplierProduct.getExtend5().length() < 255, "扩展字段5,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getExtend6() == null || supplierProduct.getExtend6().length() < 255, "扩展字段6,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getExtend7() == null || supplierProduct.getExtend7().length() < 255, "扩展字段7,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getTenantCode() == null || supplierProduct.getTenantCode().length() < 255, "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getProductCode() == null || supplierProduct.getProductCode().length() < 64, "商品编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(supplierProduct.getProductName() == null || supplierProduct.getProductName().length() < 128, "商品名称,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplierProduct.getMainImagePath() == null || supplierProduct.getMainImagePath().length() < 255, "主图路径,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getBrandCode() == null || supplierProduct.getBrandCode().length() < 64, "品牌编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(supplierProduct.getBrandName() == null || supplierProduct.getBrandName().length() < 1024, "品牌名称,在进行添加时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(supplierProduct.getProductSpecificationCode() == null || supplierProduct.getProductSpecificationCode().length() < 128, "商品（规格）编码,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplierProduct.getProductSpecificationName() == null || supplierProduct.getProductSpecificationName().length() < 128, "商品（规格）名称,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplierProduct.getUnitCode() == null || supplierProduct.getUnitCode().length() < 255, "单位,在进行添加时填入值超过了限定长度(255)，请检查!");
  }

  @Transactional
  @Override
  public SupplierProduct update(SupplierProduct supplierProduct) {
    SupplierProduct current = this.updateForm(supplierProduct);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public SupplierProduct updateForm(SupplierProduct supplierProduct) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(supplierProduct);
    // ===================基本信息
    String currentId = supplierProduct.getId();
    Optional<SupplierProduct> op_currentSupplierProduct = this.supplierProductRepository.findById(currentId);
    SupplierProduct currentSupplierProduct = op_currentSupplierProduct.orElse(null);
    currentSupplierProduct = Validate.notNull(currentSupplierProduct, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentSupplierProduct.setModifyAccount(SecurityUtils.getUserAccount());
    currentSupplierProduct.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentSupplierProduct.setExtend1(supplierProduct.getExtend1());
    currentSupplierProduct.setExtend2(supplierProduct.getExtend2());
    currentSupplierProduct.setExtend3(supplierProduct.getExtend3());
    currentSupplierProduct.setExtend4(supplierProduct.getExtend4());
    currentSupplierProduct.setExtend5(supplierProduct.getExtend5());
    currentSupplierProduct.setExtend6(supplierProduct.getExtend6());
    currentSupplierProduct.setExtend7(supplierProduct.getExtend7());
    currentSupplierProduct.setExtend8(supplierProduct.getExtend8());
    currentSupplierProduct.setExtend9(supplierProduct.getExtend9());
    currentSupplierProduct.setExtend10(supplierProduct.getExtend10());
    currentSupplierProduct.setExtend11(supplierProduct.getExtend11());
    currentSupplierProduct.setTenantCode(supplierProduct.getTenantCode());
    currentSupplierProduct.setProductCode(supplierProduct.getProductCode());
    currentSupplierProduct.setProductName(supplierProduct.getProductName());
    currentSupplierProduct.setShelfStatus(supplierProduct.getShelfStatus());
    currentSupplierProduct.setMainImagePath(supplierProduct.getMainImagePath());
    currentSupplierProduct.setBrandCode(supplierProduct.getBrandCode());
    currentSupplierProduct.setBrandName(supplierProduct.getBrandName());
    currentSupplierProduct.setProductSpecificationCode(supplierProduct.getProductSpecificationCode());
    currentSupplierProduct.setProductSpecificationName(supplierProduct.getProductSpecificationName());
    currentSupplierProduct.setUnitCode(supplierProduct.getUnitCode());
    currentSupplierProduct.setReferencePrice(supplierProduct.getReferencePrice());

    this.supplierProductRepository.saveAndFlush(currentSupplierProduct);
    return currentSupplierProduct;
  }

  /**
   * 在更新一个已有的SupplierProduct模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(SupplierProduct supplierProduct) {

    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(supplierProduct.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(supplierProduct.getProductCode(), "修改信息时，商品编码不能为空！");
    Validate.notBlank(supplierProduct.getProductName(), "修改信息时，商品名称不能为空！");
    Validate.notNull(supplierProduct.getShelfStatus(), "修改信息时，上架状态不能为空！");
    Validate.notBlank(supplierProduct.getMainImagePath(), "修改信息时，主图路径不能为空！");
    Validate.notBlank(supplierProduct.getProductSpecificationCode(), "修改信息时，商品（规格）编码不能为空！");
    Validate.notBlank(supplierProduct.getProductSpecificationName(), "修改信息时，商品（规格）名称不能为空！");
    Validate.notBlank(supplierProduct.getUnitCode(), "修改信息时，单位不能为空！");
    Validate.notNull(supplierProduct.getReferencePrice(), "修改信息时，订货参考价不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(supplierProduct.getExtend1() == null || supplierProduct.getExtend1().length() < 255, "扩展字段1,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getExtend2() == null || supplierProduct.getExtend2().length() < 255, "扩展字段2,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getExtend3() == null || supplierProduct.getExtend3().length() < 255, "扩展字段3,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getExtend4() == null || supplierProduct.getExtend4().length() < 255, "扩展字段4,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getExtend5() == null || supplierProduct.getExtend5().length() < 255, "扩展字段5,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getExtend6() == null || supplierProduct.getExtend6().length() < 255, "扩展字段6,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getExtend7() == null || supplierProduct.getExtend7().length() < 255, "扩展字段7,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getTenantCode() == null || supplierProduct.getTenantCode().length() < 255, "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getProductCode() == null || supplierProduct.getProductCode().length() < 64, "商品编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(supplierProduct.getProductName() == null || supplierProduct.getProductName().length() < 128, "商品名称,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplierProduct.getMainImagePath() == null || supplierProduct.getMainImagePath().length() < 255, "主图路径,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getBrandCode() == null || supplierProduct.getBrandCode().length() < 64, "品牌编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(supplierProduct.getBrandName() == null || supplierProduct.getBrandName().length() < 1024, "品牌名称,在进行修改时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(supplierProduct.getProductSpecificationCode() == null || supplierProduct.getProductSpecificationCode().length() < 128, "商品（规格）编码,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplierProduct.getProductSpecificationName() == null || supplierProduct.getProductSpecificationName().length() < 128, "商品（规格）名称,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplierProduct.getUnitCode() == null || supplierProduct.getUnitCode().length() < 255, "单位,在进行修改时填入值超过了限定长度(255)，请检查!");
  }

  @Override
  public SupplierProduct findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.supplierProductRepository.findDetailsById(id);
  }

  @Override
  public SupplierProduct findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<SupplierProduct> op = supplierProductRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    SupplierProduct current = this.findById(id);
    if (current != null) {
      this.supplierProductRepository.delete(current);
    }
  }

  @Override
  @Transactional
  public void save(Set<SupplierProduct> supplierProducts, Supplier supplier) {
    /**
     * 1.保存数据边界校验
     * 2.工具类拆分数据
     * 3.写入数据库
     */
    this.saveValidation(supplierProducts,supplier);
    supplierProducts = ObjectUtils.defaultIfNull(supplierProducts, Sets.newHashSet());
    Set<SupplierProduct> deletes = Sets.newHashSet();
    Set<SupplierProduct> updates = Sets.newHashSet();
    Set<SupplierProduct> creates = Sets.newHashSet();
    String tenantCode = TenantUtils.getTenantCode();
    Map<String,SupplierProduct> supplierProductMap = supplierProducts.stream().collect(Collectors.toMap(SupplierProduct::getId,v->v,(v1,v2)->v2));
    Set<SupplierProduct> dbSupplierProducts = supplierProductRepository.findBySupplierCodeAndTenantCode(supplier.getSupplierCode(),tenantCode);
    nebulaToolkitService.collectionDiscrepancy(supplierProducts,dbSupplierProducts,SupplierProduct::getId,deletes,updates,creates);
    //删除数据
    //解决遗留问题:当在同一事物中 先使用delete 方法删除 在使用save方法保存时，
    //jpa 会先执行insert 在执行delete，该方法会产生唯一索引异常，
    // 解决方法为：使用deleteInBatch方法或者delete之后手动调用flush方法
    if(!CollectionUtils.isEmpty(deletes)){
      supplierProductRepository.deleteInBatch(deletes);
    }
    //更新数据
    for (SupplierProduct update : updates) {
      SupplierProduct supplierProduct = supplierProductMap.get(update.getId());
      this.basicsUpdate(update,supplierProduct);
      supplierProductRepository.saveAndFlush(update);
    }
    //新增数据
    Date now = new Date();
    String userAccount = SecurityUtils.getUserAccount();
    for (SupplierProduct create : creates) {
      create.setCreateAccount(userAccount);
      create.setCreateTime(now);
      create.setModifyAccount(userAccount);
      create.setModifyTime(now);
      create.setSupplier(supplier);
      create.setId(null);
      supplierProductRepository.saveAndFlush(create);
    }
  }

  /**
   * 更新基础数据
   * @param dbSupplierProduct
   * @param supplierProduct
   */
  private void basicsUpdate(SupplierProduct dbSupplierProduct,SupplierProduct supplierProduct){
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    dbSupplierProduct.setModifyAccount(SecurityUtils.getUserAccount());
    dbSupplierProduct.setModifyTime(now);
    // 开始重新赋值——一般属性
    dbSupplierProduct.setTenantCode(supplierProduct.getTenantCode());
    dbSupplierProduct.setProductCode(supplierProduct.getProductCode());
    dbSupplierProduct.setProductName(supplierProduct.getProductName());
    dbSupplierProduct.setShelfStatus(supplierProduct.getShelfStatus());
    dbSupplierProduct.setMainImagePath(supplierProduct.getMainImagePath());
    dbSupplierProduct.setBrandCode(supplierProduct.getBrandCode());
    dbSupplierProduct.setBrandName(supplierProduct.getBrandName());
    dbSupplierProduct.setProductSpecificationCode(supplierProduct.getProductSpecificationCode());
    dbSupplierProduct.setProductSpecificationName(supplierProduct.getProductSpecificationName());
    dbSupplierProduct.setUnitCode(supplierProduct.getUnitCode());
    dbSupplierProduct.setReferencePrice(supplierProduct.getReferencePrice());

  }

  /**
   * 保存供货商商品信息前数据校验
   * @param supplierProducts
   * @param supplier
   */
  private void saveValidation(Set<SupplierProduct> supplierProducts,Supplier supplier){
    Validate.notNull(supplier,"传入的供应商信息不能为空");
    if(CollectionUtils.isEmpty(supplierProducts)){
      return;
    }
    /**
     * 逐条校验信息
     */
    for (SupplierProduct supplierProduct : supplierProducts) {
      this.bascicsValidation(supplierProduct);
    }
  }

  /**
   * 基础属性校验
   * @param supplierProduct
   */
  private void bascicsValidation(SupplierProduct supplierProduct){
    Validate.notNull(supplierProduct, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.notBlank(supplierProduct.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(supplierProduct.getProductCode(), "添加信息时，商品编码不能为空！");
    Validate.notBlank(supplierProduct.getProductName(), "添加信息时，商品名称不能为空！");
    Validate.notNull(supplierProduct.getShelfStatus(), "添加信息时，上架状态不能为空！");
    Validate.notBlank(supplierProduct.getMainImagePath(), "添加信息时，主图路径不能为空！");
    Validate.notBlank(supplierProduct.getProductSpecificationCode(), "添加信息时，商品（规格）编码不能为空！");
    Validate.notBlank(supplierProduct.getProductSpecificationName(), "添加信息时，商品（规格）名称不能为空！");
    Validate.notBlank(supplierProduct.getUnitCode(), "添加信息时，单位不能为空！");
    Validate.notNull(supplierProduct.getReferencePrice(), "添加信息时，订货参考价不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(supplierProduct.getTenantCode() == null || supplierProduct.getTenantCode().length() < 255, "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getProductCode() == null || supplierProduct.getProductCode().length() < 64, "商品编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(supplierProduct.getProductName() == null || supplierProduct.getProductName().length() < 128, "商品名称,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplierProduct.getMainImagePath() == null || supplierProduct.getMainImagePath().length() < 255, "主图路径,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplierProduct.getBrandCode() == null || supplierProduct.getBrandCode().length() < 64, "品牌编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(supplierProduct.getBrandName() == null || supplierProduct.getBrandName().length() < 1024, "品牌名称,在进行添加时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(supplierProduct.getProductSpecificationCode() == null || supplierProduct.getProductSpecificationCode().length() < 128, "商品（规格）编码,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplierProduct.getProductSpecificationName() == null || supplierProduct.getProductSpecificationName().length() < 128, "商品（规格）名称,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplierProduct.getUnitCode() == null || supplierProduct.getUnitCode().length() < 255, "单位,在进行添加时填入值超过了限定长度(255)，请检查!");
  }

  @Override
  public Page<SupplierPurchaseProductVo> findByConditions(Pageable pageable, InvokeParams invokeParams) {
    /**
     * 获取商品列表
     * 根据获取到的商品规格编码 获取 库存信息
     * 验证经销商编码是否为空，如果不为空，根据获取到的规格编码 丰富经销商定价信息
     */
    Page<ProductSpecificationVo> page = productSpecificationVoService.findByConditions(pageable, invokeParams);
    if (!page.isEmpty()) {
      List<SupplierPurchaseProductVo> list = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(page.getContent(), ProductSpecificationVo.class, SupplierPurchaseProductVo.class, HashSet.class, ArrayList.class,
          "product",
          "product.productCategory",
          "product.productBrand",
          "product.productFiles",
          "product.productSpecifications",
          "product.productUnitAndPrices",
          "product.productUnitAndPrices.productUnit",
          "product.productUnitSpecificationAndPrices",
          "product.productUnitSpecificationAndPrices.productUnit",
          "product.productBarCodeInfos",
          "productBarCodeInfos"));
      String supplierCode = (String) invokeParams.getInvokeParam("supplierCode");
      Map<String, SupplierProduct> supplierProductMap = new HashMap<>();
      if (StringUtils.isNotBlank(supplierCode)) {
        Set<SupplierProduct> supplierProducts = supplierProductRepository.findBySupplierCodeAndTenantCode(supplierCode, TenantUtils.getTenantCode());
        if (!CollectionUtils.isEmpty(supplierProducts)) {
          supplierProductMap = supplierProducts.stream().collect(Collectors.toMap(SupplierProduct::getProductSpecificationCode, Function.identity()));
        }
      }
      for (SupplierPurchaseProductVo p : list) {
        if (supplierProductMap.containsKey(p.getProductSpecificationCode())) {
          BigDecimal referencePrice = supplierProductMap.get(p.getProductSpecificationCode()).getReferencePrice();
          p.setReferencePrice(referencePrice);
        }
      }
      return new PageImpl<>(list, pageable, page.getTotalElements());
    }
    return Page.empty(pageable);
  }
}
