package com.bizunited.empower.business.purchase.service.internal;

import com.bizunited.empower.business.purchase.entity.Supplier;
import com.bizunited.empower.business.purchase.repository.SupplierRepository;
import com.bizunited.empower.business.purchase.service.SupplierProductService;
import com.bizunited.empower.business.purchase.service.SupplierService;
import com.bizunited.empower.business.purchase.vo.SupplierVo;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;

import java.lang.Override;
import java.lang.String;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import javax.transaction.Transactional;

import com.google.common.collect.Maps;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import static com.bizunited.platform.common.constant.Constants.DEFAULT_PAGEABLE;

/**
 * Supplier业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("SupplierServiceImpl")
public class SupplierServiceImpl implements SupplierService {

  private static final String SUPPLIER_CODE_PREFIX = "GY";

  /**
   * 生成供应商编码 redis锁前缀
   */
  private static final String SUPPLIER_REDIS_LOCK_CODE = "GY_SUPPLIER_";


  @Autowired
  private SupplierRepository supplierRepository;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private RedisMutexService redisMutexService;

  @Autowired
  private SupplierProductService supplierProductService;

  @Transactional
  @Override
  public Supplier create(Supplier supplier) {
    Supplier current = this.createForm(supplier);
    //====================================================
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public Supplier createForm(Supplier supplier) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    Date now = new Date();
    supplier.setCreateAccount(SecurityUtils.getUserAccount());
    supplier.setCreateTime(now);
    supplier.setModifyAccount(SecurityUtils.getUserAccount());
    supplier.setModifyTime(now);
    //生成供应商信息
    String tenantCode = TenantUtils.getTenantCode();
    supplier.setTenantCode(tenantCode);
    String supplierCode = generateCode(tenantCode);
    //验证供应商编码是否重复
    Supplier supplierUQ = supplierRepository.findBySupplierCodeAndTenantCode(supplierCode, tenantCode);
    Validate.isTrue(Objects.isNull(supplierUQ), "供应商编号重复生成，请重新点击提交，无需更改页面");
    supplier.setSupplierCode(supplierCode);
    this.createValidation(supplier);

    // ===============================
    //  和业务有关的验证填写在这个区域
    // ===============================

    this.supplierRepository.save(supplier);
    //保存供应商商品信息
    supplierProductService.save(supplier.getProducts(), supplier);


    // 返回最终处理的结果，里面带有详细的关联信息
    return supplier;
  }

  /**
   * 在创建一个新的Supplier模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(Supplier supplier) {
    Validate.notNull(supplier, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(supplier.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    supplier.setId(null);
    Validate.notBlank(supplier.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(supplier.getSupplierCode(), "添加信息时，供应商编码不能为空！");
    Validate.notBlank(supplier.getSupplierName(), "添加信息时，供应商名称不能为空！");
    Validate.notBlank(supplier.getContanct(), "添加信息时，联系人不能为空！");
    Validate.notBlank(supplier.getMobile(), "添加信息时，手机号不能为空！");
    Validate.notNull(supplier.getTstatus(), "添加信息时，状态不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(supplier.getExtend1() == null || supplier.getExtend1().length() < 255, "扩展字段1,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplier.getExtend2() == null || supplier.getExtend2().length() < 255, "扩展字段2,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplier.getExtend3() == null || supplier.getExtend3().length() < 255, "扩展字段3,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplier.getExtend4() == null || supplier.getExtend4().length() < 255, "扩展字段4,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplier.getExtend5() == null || supplier.getExtend5().length() < 255, "扩展字段5,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplier.getExtend6() == null || supplier.getExtend6().length() < 255, "扩展字段6,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplier.getExtend7() == null || supplier.getExtend7().length() < 255, "扩展字段7,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplier.getTenantCode() == null || supplier.getTenantCode().length() < 255, "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplier.getSupplierCode() == null || supplier.getSupplierCode().length() < 64, "供应商编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(supplier.getSupplierName() == null || supplier.getSupplierName().length() < 128, "供应商名称,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getContanct() == null || supplier.getContanct().length() < 128, "联系人,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getMobile() == null || supplier.getMobile().length() < 128, "手机号,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getSupplierAddress() == null || supplier.getSupplierAddress().length() < 128, "供应商地址,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getAccountName() == null || supplier.getAccountName().length() < 128, "开户名称,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getBankName() == null || supplier.getBankName().length() < 128, "开户银行,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getBankCode() == null || supplier.getBankCode().length() < 128, "银行账号,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getInvoiceTitle() == null || supplier.getInvoiceTitle().length() < 128, "开票抬头,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getTaxpayerNumber() == null || supplier.getTaxpayerNumber().length() < 128, "纳税人识别号,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getRemark() == null || supplier.getRemark().length() < 255, "备注信息,在进行添加时填入值超过了限定长度(255)，请检查!");
  }

  @Transactional
  @Override
  public Supplier update(Supplier supplier) {
    Supplier current = this.updateForm(supplier);
    //====================================================
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public Supplier updateForm(Supplier supplier) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     *
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */

    this.updateValidation(supplier);
    // ===================基本信息
    String currentId = supplier.getId();
    Optional<Supplier> op_currentSupplier = this.supplierRepository.findById(currentId);
    Supplier currentSupplier = op_currentSupplier.orElse(null);
    currentSupplier = Validate.notNull(currentSupplier, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentSupplier.setModifyAccount(SecurityUtils.getUserAccount());
    currentSupplier.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentSupplier.setExtend1(supplier.getExtend1());
    currentSupplier.setExtend2(supplier.getExtend2());
    currentSupplier.setExtend3(supplier.getExtend3());
    currentSupplier.setExtend4(supplier.getExtend4());
    currentSupplier.setExtend5(supplier.getExtend5());
    currentSupplier.setExtend6(supplier.getExtend6());
    currentSupplier.setExtend7(supplier.getExtend7());
    currentSupplier.setExtend8(supplier.getExtend8());
    currentSupplier.setExtend9(supplier.getExtend9());
    currentSupplier.setExtend10(supplier.getExtend10());
    currentSupplier.setExtend11(supplier.getExtend11());
    currentSupplier.setSupplierName(supplier.getSupplierName());
    currentSupplier.setContanct(supplier.getContanct());
    currentSupplier.setMobile(supplier.getMobile());
    currentSupplier.setSupplierAddress(supplier.getSupplierAddress());
    currentSupplier.setAccountName(supplier.getAccountName());
    currentSupplier.setBankName(supplier.getBankName());
    currentSupplier.setBankCode(supplier.getBankCode());
    currentSupplier.setInvoiceTitle(supplier.getInvoiceTitle());
    currentSupplier.setTaxpayerNumber(supplier.getTaxpayerNumber());
    currentSupplier.setRemark(supplier.getRemark());

    this.supplierRepository.saveAndFlush(currentSupplier);
    //修改商品信息
    supplierProductService.save(supplier.getProducts(), currentSupplier);

    return currentSupplier;
  }

  /**
   * 在更新一个已有的Supplier模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(Supplier supplier) {
    Validate.isTrue(!StringUtils.isBlank(supplier.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");

    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(supplier.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(supplier.getSupplierCode(), "修改信息时，供应商编码不能为空！");
    Validate.notBlank(supplier.getSupplierName(), "修改信息时，供应商名称不能为空！");
    Validate.notBlank(supplier.getContanct(), "修改信息时，联系人不能为空！");
    Validate.notBlank(supplier.getMobile(), "修改信息时，手机号不能为空！");
    Validate.notNull(supplier.getTstatus(), "修改信息时，状态不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(supplier.getExtend1() == null || supplier.getExtend1().length() < 255, "扩展字段1,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplier.getExtend2() == null || supplier.getExtend2().length() < 255, "扩展字段2,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplier.getExtend3() == null || supplier.getExtend3().length() < 255, "扩展字段3,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplier.getExtend4() == null || supplier.getExtend4().length() < 255, "扩展字段4,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplier.getExtend5() == null || supplier.getExtend5().length() < 255, "扩展字段5,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplier.getExtend6() == null || supplier.getExtend6().length() < 255, "扩展字段6,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplier.getExtend7() == null || supplier.getExtend7().length() < 255, "扩展字段7,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplier.getTenantCode() == null || supplier.getTenantCode().length() < 255, "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(supplier.getSupplierCode() == null || supplier.getSupplierCode().length() < 64, "供应商编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(supplier.getSupplierName() == null || supplier.getSupplierName().length() < 128, "供应商名称,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getContanct() == null || supplier.getContanct().length() < 128, "联系人,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getMobile() == null || supplier.getMobile().length() < 128, "手机号,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getSupplierAddress() == null || supplier.getSupplierAddress().length() < 128, "供应商地址,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getAccountName() == null || supplier.getAccountName().length() < 128, "开户名称,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getBankName() == null || supplier.getBankName().length() < 128, "开户银行,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getBankCode() == null || supplier.getBankCode().length() < 128, "银行账号,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getInvoiceTitle() == null || supplier.getInvoiceTitle().length() < 128, "开票抬头,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getTaxpayerNumber() == null || supplier.getTaxpayerNumber().length() < 128, "纳税人识别号,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(supplier.getRemark() == null || supplier.getRemark().length() < 255, "备注信息,在进行修改时填入值超过了限定长度(255)，请检查!");
  }

  @Override
  public Supplier findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.supplierRepository.findDetailsById(id);
  }

  @Override
  public Supplier findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    Optional<Supplier> op = supplierRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  public Supplier findByCode(String code) {
    if(StringUtils.isBlank(code)){
      return null;
    }
    return supplierRepository.findBySupplierCodeAndTenantCode(code,TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    Supplier current = this.findById(id);
    if (current != null) {
      this.supplierRepository.delete(current);
    }
  }

  @Override
  public Page<SupplierVo> findByConditions(Pageable pageable, Map<String, Object> conditions) {
    pageable = ObjectUtils.defaultIfNull(pageable, DEFAULT_PAGEABLE);
    conditions = ObjectUtils.defaultIfNull(conditions, Maps.newHashMap());
    String tenantCode = TenantUtils.getTenantCode();
    conditions.put("tenantCode", tenantCode);
    Page<Supplier> page = supplierRepository.findByConditions(pageable, conditions);
    List<Supplier> supplierList = page.getContent();
    List<SupplierVo> supplierVos = new ArrayList<>(supplierList.size());
    supplierList.forEach(e -> {
      SupplierVo supplierVo = nebulaToolkitService.copyObjectByWhiteList(e, SupplierVo.class, LinkedHashSet.class, ArrayList.class, "products");
      supplierVos.add(supplierVo);
    });
    return new PageImpl<>(supplierVos, pageable, page.getTotalElements());
  }

  /**
   * 根据租户编码生成供应商编码
   *
   * @param tenantCode
   * @return
   */
  private String generateCode(String tenantCode) {
    String lockCode = StringUtils.join(SUPPLIER_REDIS_LOCK_CODE, tenantCode);
    String atomicNumber = this.redisMutexService.getAndIncrement(lockCode, 1, 6);
    return StringUtils.join(SUPPLIER_CODE_PREFIX, atomicNumber);
  }

  @Override
  public void updateByStatus(String id, Integer status) {
    //为空验证
    Validate.notBlank(id, "ID不能为空");
    Validate.notNull(status, "状态不能为空");
    //验证数据是否存在
    Supplier supplier = this.findById(id);
    Validate.notNull(supplier, "修改信息不能为空");
    supplier.setTstatus(status);
    this.supplierRepository.saveAndFlush(supplier);
  }

  @Override
  public List<Supplier> findByTstatus(Integer tstatus) {
    if(tstatus==null){
      return null;
    }
    return supplierRepository.findByTstatusAndTenantCode(tstatus,TenantUtils.getTenantCode());
  }
}
