package com.bizunited.empower.business.purchase.service.notifier;

import com.bizunited.empower.business.purchase.common.enums.PurchaseOrderStatus;
import com.bizunited.empower.business.purchase.entity.PurchaseOrder;
import com.bizunited.empower.business.purchase.entity.PurchaseOrderProduct;
import com.bizunited.empower.business.purchase.repository.PurchaseOrderRepository;
import com.bizunited.empower.business.purchase.service.PurchaseOrderService;
import com.bizunited.empower.business.warehouse.entity.WarehouseProductsEnter;
import com.bizunited.empower.business.warehouse.entity.WarehouseProductsEnterProduct;
import com.bizunited.empower.business.warehouse.service.WarehouseProductsEnterService;
import com.bizunited.empower.business.warehouse.service.notifier.WarehouseEnterEventListener;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

/**
 * 采购单入库事物监听
 * @author lihuan
 */
@Component("WarehouseEnterEventForPurchaseOrderListenerImpl")
public class WarehouseEnterEventForPurchaseOrderListenerImpl implements WarehouseEnterEventListener {

  @Autowired
  private WarehouseProductsEnterService warehouseProductsEnterService;

  @Autowired
  private PurchaseOrderService purchaseOrderService;
  @Autowired
  private PurchaseOrderRepository purchaseOrderRepository;

  @Override
  @Transactional
  public void onCompleted(WarehouseProductsEnter warehouseProductsEnter) {
    /*
     * 一旦一张入库单完成入库过程，采购单模块需要完成以下工作：
     * 1、如果当前入库单单的主类型不是“采购入库”，则不用理会，否则就要继续以下处理
     * 2、验证当前入库单商品入库数量，如果全部入库，那么修改为已入库，如果部分入库，修改为部分入库
     * */
    // 1、========
    if(warehouseProductsEnter.getType() != 1) {
      return;
    }
    //获取当前采购单 采购总数
    PurchaseOrder purchaseOrder = this.purchaseOrderService.findByCode(warehouseProductsEnter.getRelevanceCode());
    BigDecimal reduce = purchaseOrder.getProductSet().stream().map(PurchaseOrderProduct::getQuantity).reduce(BigDecimal.ZERO, BigDecimal::add);

    //获取当前已入库的数量
    List<WarehouseProductsEnter> list = warehouseProductsEnterService.findDetailsByRelevanceCode(warehouseProductsEnter.getRelevanceCode());
    BigDecimal newCount = new BigDecimal(0);
    for (WarehouseProductsEnter enter : list) {
      for (WarehouseProductsEnterProduct product : enter.getProducts()) {
        newCount = newCount.add(product.getQuantity());
      }
    }
    //修改状态
    if(reduce.compareTo(newCount)==0){
      purchaseOrder.setPurchaseOrderStatus(PurchaseOrderStatus.COMPLETED.getCode());

    }else{
      purchaseOrder.setPurchaseOrderStatus(PurchaseOrderStatus.PART_WAREHOUSE.getCode());
    }
    purchaseOrderRepository.saveAndFlush(purchaseOrder);

  }

  @Override
  @Transactional
  public void onCancelled(WarehouseProductsEnter warehouseProductsEnter) {
    //入库单取消 采购模块不需要任何操作
  }

  @Override
  @Transactional
  public void onCreated(WarehouseProductsEnter warehouseProductsEnter) {
    /*
     * 当一个入库单被创建后，采购单功能会实现一个接口方法，并监控这个事件，
     * 主要完成逻辑判定。
     *
     * 1、当前入库单是采购入库，且关联了采购单信息
     * 如果以上前提不存在，则不再向后处理
     * 2.根据当前采购单，找到所有对应的入库单（当然这些入库单状态也是正确的，非"作废"状态）
     * 3、基于商品规格信息，累加这些采购单上的所有明细，
     * 如果发现某一条商品规格，已经超过了采购单上的订货数量，则不允许本次创建
     * */

    // 1、=======
    String purchaseOrderCode = warehouseProductsEnter.getRelevanceCode();
    Integer warehouseEnterType = warehouseProductsEnter.getType();
    if(warehouseEnterType != 1 || StringUtils.isBlank(purchaseOrderCode)) {
      return;
    }
    PurchaseOrder purchaseOrder = this.purchaseOrderService.findByCode(purchaseOrderCode);
    Validate.notNull(purchaseOrder , "未发现编号为【%s】的采购单信息，请检查!!" , purchaseOrderCode);
    int status = purchaseOrder.getPurchaseOrderStatus();
    //采购单未审核 不能入库
    Validate.isTrue(PurchaseOrderStatus.STOCK_PENDING.getCode()==status|| PurchaseOrderStatus.PART_WAREHOUSE.getCode()==status,"当前状态不能入库");

    // 2、=======
    Set<PurchaseOrderProduct> productSet = purchaseOrder.getProductSet();
    // 查询采购单所有状态正确的入库单
    List<WarehouseProductsEnter> warehouseProductsEnters = this.warehouseProductsEnterService.findDetailsByRelevanceCode(purchaseOrderCode);
    // 如果还没有任何入库单（当然，这并不可能出现），也不用再校验超出的情况
    if(CollectionUtils.isEmpty(warehouseProductsEnters)) {
      return;
    }

    // 以下是本次超出校验的对比源——基于订单明细统计的订单明细数量
    Map<String, BigDecimal> sourceTotalMapping = Maps.newHashMap();
    for (PurchaseOrderProduct orderProductItem : productSet) {
      String orderProductId = orderProductItem.getId();
      BigDecimal orderQuantity = orderProductItem.getQuantity();
      // 不太可能出现，除非是测试系统中的脏数据
      if(orderQuantity == null) {
        orderQuantity = BigDecimal.ZERO;
      }
      if(sourceTotalMapping.get(orderProductId) == null) {
        sourceTotalMapping.put(orderProductId, orderQuantity);
      } else {
        sourceTotalMapping.put(orderProductId, orderQuantity.add(sourceTotalMapping.get(orderProductId)));
      }
    }
    // 以下是本次超出校验的对比目标——基于订单明细的，目前和订单关联的所有状态正常的退货入库单中的明细累加数量
    Map<String, BigDecimal> targetTotalMapping = Maps.newHashMap();
    for (WarehouseProductsEnter enterItem : warehouseProductsEnters) {
      Set<WarehouseProductsEnterProduct> enterProducts = enterItem.getProducts();
      if(enterProducts != null) {
        for (WarehouseProductsEnterProduct enterProductItem : enterProducts) {
          String orderRelevanceDetailCode = enterProductItem.getRelevanceDetailCode();
          BigDecimal enterQuantity = enterProductItem.getQuantity();
          // 不太可能出现，除非是测试系统中的脏数据
          if(enterQuantity == null) {
            enterQuantity = BigDecimal.ZERO;
          }
          if(targetTotalMapping.get(orderRelevanceDetailCode) == null) {
            targetTotalMapping.put(orderRelevanceDetailCode, enterQuantity);
          } else {
            targetTotalMapping.put(orderRelevanceDetailCode, enterQuantity.add(targetTotalMapping.get(orderRelevanceDetailCode)));
          }
        }
      }
    }

    // 3、=======
    for (Entry<String, BigDecimal> targetItem : targetTotalMapping.entrySet()) {
      String productSpecificationCode = targetItem.getKey();
      BigDecimal targetQuantity = targetItem.getValue();
      if(targetQuantity == null) {
        continue;
      }
      // 开始比较
      BigDecimal sourceQuantity = sourceTotalMapping.get(productSpecificationCode);
      if(sourceQuantity == null) {
        continue;
      }
      Validate.isTrue(targetQuantity.floatValue() <= sourceQuantity.floatValue() , "错误：在本次入库单创建后，编号[%s]的商品规格入库总数，将超过采购订单[%s]中该商品规格的总采购数量，请检查!!" , productSpecificationCode , purchaseOrderCode);
    }
  }
}
