package com.bizunited.empower.business.purchase.service.notifier;

import com.bizunited.empower.business.purchase.common.enums.PurchaseReturnOrderStatus;
import com.bizunited.empower.business.purchase.entity.PurchaseReturnOrder;
import com.bizunited.empower.business.purchase.entity.PurchaseReturnOrderProduct;
import com.bizunited.empower.business.purchase.repository.PurchaseReturnOrderRepository;
import com.bizunited.empower.business.warehouse.dto.AuthorizeDto;
import com.bizunited.empower.business.warehouse.entity.WarehouseProductsExpense;
import com.bizunited.empower.business.warehouse.service.WarehouseProductsExpenseVoService;
import com.bizunited.empower.business.warehouse.service.notifier.WarehouseExpenseEventListener;
import com.bizunited.empower.business.warehouse.vo.WarehouseProductsExpenseProductVo;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 当仓储模块的出库单正式完成后，采购退单模块会得到通知，并针对采购出库类型的出库单做特定处理</br>
 * 当仓库模块的出库单完成取消操作后，采购退单模块也会得到通知，并干预出库单是否能够取消得判定逻辑
 *
 * @author lihuan
 */
@Component("WarehouseExpenseEventForPurchaseReturnOrderListenerImpl")
public class WarehouseExpenseEventForPurchaseReturnOrderListenerImpl implements WarehouseExpenseEventListener {

  @Autowired
  private WarehouseProductsExpenseVoService warehouseProductsExpenseVoService;

  @Autowired
  private PurchaseReturnOrderRepository purchaseReturnOrderRepository;

  @Override
  @Transactional
  public void onTransport(WarehouseProductsExpense warehouseProductsExpense) {

  }

  @Override
  @Transactional
  public void onCompleted(WarehouseProductsExpense warehouseProductsExpense) {
    /*
     * 一旦某张出库单成功完成出库，订单模块需要完成以下工作
     * 1、如果当前出库单的主类型不是“采购退单”，则不用理会，否则就要继续以下处理
     * 2、获取当前退单出库单列表
     * 3、对比在第2步取得的物资明细信息计数，做一下处理：
     *  a、如果当前订单上的有任何一个物资还未完成出库，则修改状态为“部分出库”
     *  c、如果当前订单的所有物资均已出库，则修改配送状态为“已完成”
     * */

    // 1、========
    if (warehouseProductsExpense.getType() != 2) {
      return;
    }
    // 2、========== 统计当前出库单需出库总数 和已出库总数
    List<WarehouseProductsExpenseProductVo> completedQuantityByRelevanceCode = warehouseProductsExpenseVoService.findCompletedQuantityByRelevanceCode(warehouseProductsExpense.getRelevanceCode());
    BigDecimal newQuantity = completedQuantityByRelevanceCode.stream().map(WarehouseProductsExpenseProductVo::getQuantity).reduce(BigDecimal.ZERO, BigDecimal::add);

    PurchaseReturnOrder purchaseReturnOrder = purchaseReturnOrderRepository.findByPurchaseReturnOrderCodeAndTenantCode(warehouseProductsExpense.getRelevanceCode(), TenantUtils.getTenantCode());
    BigDecimal quantity = purchaseReturnOrder.getProductSet().stream().map(PurchaseReturnOrderProduct::getQuantity).reduce(BigDecimal.ZERO, BigDecimal::add);

    // 3、======= 当所有物资都已经出库 修改出库状态为已完成
    if (quantity.compareTo(newQuantity) == 0) {
      purchaseReturnOrder.setPurchaseReturnOrderStatus(PurchaseReturnOrderStatus.COMPLETED.getCode());
      this.purchaseReturnOrderRepository.save(purchaseReturnOrder);
    } else {
      purchaseReturnOrder.setPurchaseReturnOrderStatus(PurchaseReturnOrderStatus.PART_OUTBOUND.getCode());
      this.purchaseReturnOrderRepository.save(purchaseReturnOrder);
    }
  }


  @Override
  @Transactional
  public void onCancelled(WarehouseProductsExpense warehouseProductsExpense) {
    // 退货单取消时，目前的业务场景下，订单无需做任何处理动作
  }

  @Override
  public void onCreated(WarehouseProductsExpense warehouseProductsExpense) {
    /*
     * 一旦某张出库单完成创建后，采购退单模块需要完成以下工作
     * 1、如果当前出库单的主类型不是“采购退单”，则不用理会，否则就要继续以下处理
     * 2、通过调用库存模块的接口，取得和这张订单有关的所有以正确完成的出库单信息
     * 主要是取得这些出库单合并后的出库商品信息
     * 3、一个订单关联的出库单被新建后，是否会出现超卖（超出）的情况，需要做两个维度的考虑
     *   a、按照关联的订单明细进行分组，订单明细下商品规格的当次出库数量 + 已出库数量，不超过关联订单明细的总订购数量
     *   b、按照商品规格对当前出库单的明细进行分组，每个商品规格的出库数量不超过当前仓库中，这个商品规格的可用库存数量 （该步骤在出库单模块进行，监听器不再负责）
     * */

    // 1、========
    if (warehouseProductsExpense.getType() != 2) {
      return;
    }
    // 2、========
    List<WarehouseProductsExpenseProductVo> completedQuantityByRelevanceCode = warehouseProductsExpenseVoService.findPreemptQuantityByRelevanceCode(warehouseProductsExpense.getRelevanceCode());
    Map<String, WarehouseProductsExpenseProductVo> collect = completedQuantityByRelevanceCode.stream().collect(Collectors.toMap(WarehouseProductsExpenseProductVo::getProductSpecificationCode, Function.identity()));
    PurchaseReturnOrder byPurchaseReturnOrderCodeAndTenantCode = purchaseReturnOrderRepository.findByPurchaseReturnOrderCodeAndTenantCode(warehouseProductsExpense.getRelevanceCode(), TenantUtils.getTenantCode());
    Set<PurchaseReturnOrderProduct> orderProducts = byPurchaseReturnOrderCodeAndTenantCode.getProductSet();

    // 3、=======
    // a、从productOrderMapping找到已完成出库的数量，并进行判定 
    for (PurchaseReturnOrderProduct orderProductItem : orderProducts) {
      String productSpecificationCode = orderProductItem.getProductSpecificationCode();
      BigDecimal orderQuantity = orderProductItem.getQuantity();
      BigDecimal expenseQuantity = collect.get(productSpecificationCode) == null ? BigDecimal.valueOf(0) : collect.get(productSpecificationCode).getQuantity();
      if (orderQuantity.floatValue() < expenseQuantity.floatValue()) {
        throw new IllegalArgumentException(String.format("编号【%s】的商品规格，在本次正式出库后，订单明细将会超出（超卖），请检查!!", productSpecificationCode));
      }
    }
  }

  @Transactional
  @Override
  public void onAuthorize(AuthorizeDto authorizeDto) {

  }

  @Override
  public String getOrderWarehouseCode(String orderCode) {
    return null;
  }
}