package com.bizunited.empower.business.tenant.service;

import com.bizunited.empower.business.tenant.dto.TenantInfoDto;
import com.bizunited.empower.business.tenant.entity.TenantInfo;
import com.bizunited.empower.business.tenant.vo.TenantInfoVo;
import com.bizunited.platform.script.context.InvokeParams;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import java.util.List;

/**
 * TenantInfo业务模型的服务层接口定义
 * @author saturn
 */
public interface TenantInfoService {
  /**
   * 创建一个新的TenantInfo模型对象（包括了可能的第三方系统调用、复杂逻辑处理等）
   */
  TenantInfo create(TenantInfoDto tenantInfoDto);

  /**
   * 更新一个已有的TenantInfo模型对象，其主键属性必须有值(1.1.4-release版本调整)。
   * 这个方法实际上一共分为三个步骤（默认）：</br>
   * 1、调用updateValidation方法完成表单数据更新前的验证</br>
   * 2、调用updateForm方法完成表单数据的更新</br>
   * 3、完成开发人员自行在本update方法中书写的，进行第三方系统调用（或特殊处理过程）的执行。</br>
   * 这样做的目的，实际上是为了保证updateForm方法中纯粹是处理表单数据的，在数据恢复表单引擎默认调用updateForm方法时，不会影响任何第三方业务数据
   * （当然，如果系统有特别要求，可由开发人员自行完成代码逻辑调整）
   */
  TenantInfo update(TenantInfoDto tenantInfoDto);

  /**
   * 按照主键进行详情查询（包括关联信息）
   * @param id 主键
   */
  TenantInfo findDetailsById(String id);
  /**
   * 按照TenantInfo的主键编号，查询指定的数据信息（不包括任何关联信息）
   * @param id 主键
   * */
  TenantInfo findById(String id);

  /**
   * 按照租户编码进行查询
   * @param code
   * @return
   */
  TenantInfo findByCode(String code);

  /**
   * 多条件分页查询
   * @param pageable
   * @param conditions
   * @return
   */
  Page<TenantInfo> findByConditions(Pageable pageable, InvokeParams conditions);

  /**
   * 获取厂商下的所有经销商
   * @param appCode
   * @return
   */
  List<TenantInfo> findByAppCode(String appCode);

  /**
   * 根据登录用户手机号获取登录账号关联的经销商
   * @param phone
   * @return
   */
  List<TenantInfo> findByUserPhone(String phone);

  /**
   * 根据登录用户账号获取登录账号关联的经销商
   * @param account
   * @return
   */
  List<TenantInfo> findByUserAccount(String account);

  /**
   * 切换当前登录账号的经销商信息
   * @param tenantCode
   */
  TenantInfoVo switchTenant(String tenantCode);

  /**
   * 获取品牌登录页logo(使用缓存保存,注意此方案仅防止同一个key的重复提交，若需要过滤无效key可使用BloomFilter设置有效key的白名单实现)，流程为：
   * 1、查询缓存，有值直接返回
   * 2、查询DB（没有值：将空对象存入缓存，有值：将对象存入缓存），返回对象
   * @return
   */
  TenantInfoVo findLoginLogoByAppCode();

  /**
   * 获取经销商logo（系统首页位置）
   * @return TenantInfoVo 获取经销商logo地址及名称
   */
  TenantInfoVo findTenantLogoByTenantCode();

  /**
   * 获取品牌商下的所有租户状态为：正常 的经销商
   * @return  品牌商下的所有经销商
   */
  List<TenantInfo> findTenantInfos();
}