package com.bizunited.empower.business.tenant.service.internal;

import com.bizunited.empower.business.tenant.entity.TenantProcessingFlowSetting;
import com.bizunited.empower.business.tenant.repository.TenantProcessingFlowSettingRepository;
import com.bizunited.empower.business.tenant.service.TenantProcessingFlowSettingService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.List;
import java.util.Optional;

/**
 * TenantProcessingFlowSettingServiceImpl
 *
 * @Author: hefan
 * @Date: 2021/6/24 15:35
 */
@Service("TenantProcessingFlowSettingServiceImpl")
public class TenantProcessingFlowSettingServiceImpl implements TenantProcessingFlowSettingService {

  @Autowired
  private TenantProcessingFlowSettingRepository tenantProcessingFlowSettingRepository;

  @Transactional
  @Override
  public TenantProcessingFlowSetting create(TenantProcessingFlowSetting tenantProcessingFlowSetting) {
    TenantProcessingFlowSetting current = this.createForm(tenantProcessingFlowSetting);
    //====================================================
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public TenantProcessingFlowSetting createForm(TenantProcessingFlowSetting tenantProcessingFlowSetting) {
    tenantProcessingFlowSetting.setTenantCode(TenantUtils.getTenantCode());
    this.createValidation(tenantProcessingFlowSetting);

    // ===============================
    //  和业务有关的验证填写在这个区域
    // ===============================

    this.tenantProcessingFlowSettingRepository.saveAndFlush(tenantProcessingFlowSetting);

    // 返回最终处理的结果，里面带有详细的关联信息
    return tenantProcessingFlowSetting;
  }

  /**
   * 在创建一个新的 TenantProcessingFlowSetting 模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(TenantProcessingFlowSetting tenantProcessingFlowSetting) {
    Validate.notNull(tenantProcessingFlowSetting, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(tenantProcessingFlowSetting.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    tenantProcessingFlowSetting.setId(null);
    Validate.notNull(tenantProcessingFlowSetting.getProcessingFlowType(), "添加信息时，处理流程类型 不能为空！");
    Validate.notNull(tenantProcessingFlowSetting.getProcessingFlowName(), "添加信息时，处理流程名称 不能为空！");
    Validate.notNull(tenantProcessingFlowSetting.getNodeNumber(), "添加信息时，处理流程节点编号 不能为空！");
    Validate.notNull(tenantProcessingFlowSetting.getNodeName(), "添加信息时，处理流程节点名称 不能为空！");
    Validate.notNull(tenantProcessingFlowSetting.getNodeStatus(), "添加信息时，节点状态 不能为空！");
    Validate.notNull(tenantProcessingFlowSetting.getTenantCode(), "添加信息时，租户信息 不能为空！");
    // 唯一判定
    long count = tenantProcessingFlowSettingRepository.countByTenantCodeAndNodeNumberAndProcessingFlowType(tenantProcessingFlowSetting.getTenantCode(),tenantProcessingFlowSetting.getNodeNumber(),tenantProcessingFlowSetting.getProcessingFlowType());
    Validate.isTrue(count == 0, "添加信息时，一个流程的一个节点只能有一条信息！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(tenantProcessingFlowSetting.getTenantCode() == null || tenantProcessingFlowSetting.getTenantCode().length() <= 64, "租户编号长度不能超过64");
    Validate.isTrue(tenantProcessingFlowSetting.getNodeName() == null || tenantProcessingFlowSetting.getNodeName().length() <= 64, "租户编号长度不能超过64");
    Validate.isTrue(tenantProcessingFlowSetting.getProcessingFlowName() == null || tenantProcessingFlowSetting.getProcessingFlowName().length() <= 64, "租户编号长度不能超过64");

  }

  @Transactional
  @Override
  public TenantProcessingFlowSetting update(TenantProcessingFlowSetting tenantProcessingFlowSetting) {
    TenantProcessingFlowSetting current = this.updateForm(tenantProcessingFlowSetting);
    //====================================================
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public TenantProcessingFlowSetting updateForm(TenantProcessingFlowSetting tenantProcessingFlowSetting) {
    this.updateValidation(tenantProcessingFlowSetting);
    // ===================基本信息
    String currentId = tenantProcessingFlowSetting.getId();
    Optional<TenantProcessingFlowSetting> op_currentTenantSetting = this.tenantProcessingFlowSettingRepository.findById(currentId);
    TenantProcessingFlowSetting currentTenantSetting = op_currentTenantSetting.orElse(null);
    currentTenantSetting = Validate.notNull(currentTenantSetting, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    // 开始重新赋值——一般属性
    currentTenantSetting.setProcessingFlowType(tenantProcessingFlowSetting.getProcessingFlowType());
    currentTenantSetting.setProcessingFlowName(tenantProcessingFlowSetting.getProcessingFlowName());
    currentTenantSetting.setNodeNumber(tenantProcessingFlowSetting.getNodeNumber());
    currentTenantSetting.setNodeName(tenantProcessingFlowSetting.getNodeName());
    currentTenantSetting.setNodeStatus(tenantProcessingFlowSetting.getNodeStatus());
    this.tenantProcessingFlowSettingRepository.saveAndFlush(currentTenantSetting);
    // 通知事件触发
    return currentTenantSetting;
  }

  /**
   * 在更新一个已有的 TenantProcessingFlowSetting 模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(TenantProcessingFlowSetting tenantProcessingFlowSetting) {
    Validate.isTrue(!StringUtils.isBlank(tenantProcessingFlowSetting.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");

    // 基础信息判断，基本属性，需要满足not null
    Validate.notNull(tenantProcessingFlowSetting.getProcessingFlowType(), "修改信息时，处理流程类型 不能为空！");
    Validate.notNull(tenantProcessingFlowSetting.getProcessingFlowName(), "修改信息时，处理流程名称 不能为空！");
    Validate.notNull(tenantProcessingFlowSetting.getNodeNumber(), "修改信息时，处理流程节点编号 不能为空！");
    Validate.notNull(tenantProcessingFlowSetting.getNodeName(), "修改信息时，处理流程节点名称 不能为空！");
    Validate.notNull(tenantProcessingFlowSetting.getNodeStatus(), "修改信息时，节点状态 不能为空！");
    Validate.notNull(tenantProcessingFlowSetting.getTenantCode(), "修改信息时，租户信息 不能为空！");
    // 唯一判定
    List<TenantProcessingFlowSetting> count = tenantProcessingFlowSettingRepository.findByTenantCodeAndNodeNumberAndProcessingFlowType(tenantProcessingFlowSetting.getTenantCode(),tenantProcessingFlowSetting.getNodeNumber(),tenantProcessingFlowSetting.getProcessingFlowType());
    Validate.isTrue(CollectionUtils.isEmpty(count) || count.size() == 1, "修改信息时, 数据库内存在相同流程相同节点的多条信息！");
    if (!CollectionUtils.isEmpty(count)) {
      TenantProcessingFlowSetting db = count.get(0);
      Validate.isTrue(tenantProcessingFlowSetting.getId().equals(db.getId()), "修改信息时, 数据库内已存在相同流程相同节点的一条信息！");
    }
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(tenantProcessingFlowSetting.getTenantCode() == null || tenantProcessingFlowSetting.getTenantCode().length() <= 64, "租户编号长度不能超过64");
    Validate.isTrue(tenantProcessingFlowSetting.getNodeName() == null || tenantProcessingFlowSetting.getNodeName().length() <= 64, "租户编号长度不能超过64");
    Validate.isTrue(tenantProcessingFlowSetting.getProcessingFlowName() == null || tenantProcessingFlowSetting.getProcessingFlowName().length() <= 64, "租户编号长度不能超过64");

  }

  @Override
  public TenantProcessingFlowSetting findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    Optional<TenantProcessingFlowSetting> op = tenantProcessingFlowSettingRepository.findById(id);
    return op.orElse(null);
  }

  @Transactional
  @Override
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    TenantProcessingFlowSetting current = this.findById(id);
    if (current != null) {
      this.tenantProcessingFlowSettingRepository.delete(current);
    }
  }

  @Override
  public List<TenantProcessingFlowSetting> findByTenantCode() {
    return this.tenantProcessingFlowSettingRepository.findByTenantCode(TenantUtils.getTenantCode());
  }
}
