package com.bizunited.empower.business.tenant.service.internal;

import com.bizunited.empower.business.common.util.SecurityUtils;
import com.bizunited.empower.business.tenant.common.enums.BrowseProductRuleEnum;
import com.bizunited.empower.business.tenant.entity.TenantSetting;
import com.bizunited.empower.business.tenant.repository.TenantSettingRepository;
import com.bizunited.empower.business.tenant.service.TenantSettingService;
import com.bizunited.empower.business.tenant.service.notifier.TenantSettingEventListener;
import com.bizunited.empower.business.tenant.service.notifier.TenantWarehouseEventListener;
import com.bizunited.empower.business.tenant.utils.DealerTenantUtils;
import com.bizunited.empower.business.tenant.utils.TenantSettingEnumUtil;
import com.bizunited.empower.business.tenant.vo.TenantSettingVo;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.base.Objects;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;

/**
 * TenantSetting业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("TenantSettingServiceImpl")
public class TenantSettingServiceImpl implements TenantSettingService {
  @Autowired
  private TenantSettingRepository tenantSettingRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private List<TenantWarehouseEventListener> tenantWarehouseEventListeners;
  @Autowired(required = false)
  private List<TenantSettingEventListener> tenantSettingEventListeners;


  @Override
  @Transactional
  public TenantSetting save(TenantSetting tenantSetting) {
    Validate.notNull(tenantSetting, "进行当前操作时，信息对象必须传入!!");
    Date now = new Date();

    TenantSetting oldTenantSetting = null;
    TenantSettingVo oldTenantSettingVo = null;
    if (StringUtils.isBlank(tenantSetting.getId())) {
      tenantSetting.setCreateAccount("System");
      tenantSetting.setCreateTime(now);
      tenantSetting.setModifyAccount("System");
      tenantSetting.setModifyTime(now);
    } else {
      oldTenantSetting = tenantSettingRepository.findById(tenantSetting.getId()).orElse(null);
      oldTenantSettingVo = nebulaToolkitService.copyObjectByBlankList(oldTenantSetting, TenantSettingVo.class, HashSet.class, ArrayList.class);

      this.updateForm(tenantSetting);
      tenantSetting.setCreateAccount("System");
      tenantSetting.setCreateTime(now);
      tenantSetting.setModifyAccount(SecurityUtils.getUserAccount());
      tenantSetting.setModifyTime(now);
    }
    this.validate(tenantSetting);
    Long baseSettingSwitch = TenantSettingEnumUtil.getSettingSwitchByEnums(tenantSetting.getBaseSettingSwitchMap());
    tenantSetting.setBaseSettingSwitch(baseSettingSwitch);

    tenantSetting = tenantSettingRepository.saveAndFlush(tenantSetting);
    TenantSettingVo newTenantSettingVo = nebulaToolkitService.copyObjectByBlankList(tenantSetting, TenantSettingVo.class, HashSet.class, ArrayList.class);

    // 通知租户设置信息变动
    this.notifySettingChangeEvent(oldTenantSettingVo, newTenantSettingVo);
    return tenantSetting;
  }


  /**
   * 在创建一个新的TenantSetting模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void validate(TenantSetting tenantSetting) {
    Validate.notNull(tenantSetting, "进行当前操作时，信息对象必须传入!!");
    Validate.notNull(tenantSetting.getWarehouseStatus(), "仓库状态不能为空！");
    Validate.notNull(tenantSetting.getCashPayStatus(), "现金支付状态不能为空！");
    Validate.notNull(tenantSetting.getOrderPartialPayStatus(), "订单部分付款状态不能为空！");
    Validate.notNull(tenantSetting.getGoodsBeforePayStatus(), "先货后款状态不能为空！");
    Validate.notNull(tenantSetting.getAccountPeriodTypeStatus(), "账期类型状态不能为空！");
    Validate.notNull(tenantSetting.getCreditBillStatus(), "信用支付状态不能为空！");

    Validate.notNull(tenantSetting.getOrderRequiredReceiptAddressStatus(), "订单必填收货地址状态不能为空！");
    Validate.notNull(tenantSetting.getOrderRequiredReceiptAddressValue(), "订单必填收货地址的业务类型不能为空！");
    Validate.notNull(tenantSetting.getOrderRequiredDeliveryDateStatus(), "订单必填交货日期状态不能为空！");
    Validate.notNull(tenantSetting.getOrderRequiredDeliveryDateValue(), "订单必填交货日期的业务类型不能为空！");
    Validate.notNull(tenantSetting.getCustomerCancelOrderStatus(), "客户取消订单状态不能为空！");
    Validate.notNull(tenantSetting.getMinimumOrderAmountStatus(), "最低下单金额状态不能为空！");
    Validate.notNull(tenantSetting.getMinimumOrderAmount(), "最低下单金额不能为空！");
    Validate.notNull(tenantSetting.getOrderTimeStatus(), "下单时间状态不能为空！");
    Validate.notNull(tenantSetting.getAutomaticReceiptStatus(), "客户超时未收货，系统自动收货状态不能为空！");
    Validate.notNull(tenantSetting.getAutomaticReceiptValue(), "客户超时未收货，系统自动收货值不能为空！");
    Validate.notNull(tenantSetting.getAutomaticCancelOrderStatus(), "自动取消订单状态不能为空！");
    Validate.notNull(tenantSetting.getAutomaticCancelOrderValue(), "自动取消订单值不能为空！");
    Validate.notNull(tenantSetting.getBanModifyReceiptAddressStatus(), "禁止客户修改收货地址状态不能为空！");
    Validate.notNull(tenantSetting.getReturnLimitStatus(), "退货限制状态不能为空！");
    Validate.notNull(tenantSetting.getReturnLimitValue(), "退货限制值不能为空！");

    Validate.notNull(tenantSetting.getWarehouseOccupyStatus(), "商品库存占用状态不能为空！");
    Validate.notNull(tenantSetting.getNegativeWarehouseStatus(), "负库存订货状态不能为空！");
    Validate.notNull(tenantSetting.getOverflowWarehouseStatus(), "超库存出库状态不能为空！");
    Validate.notNull(tenantSetting.getDistanceStatus(), "签到、离店距离状态不能为空！");
    Validate.notNull(tenantSetting.getBaseSettingSwitch(), "租户功能设置开关的全局聚合设置不能为空！");

    Validate.isTrue(tenantSetting.getTouristBrowseProduct() == null || BrowseProductRuleEnum.findEnumByRuleCode(tenantSetting.getTouristBrowseProduct()) != null,
            "游客（客户未登录前）访问订货平台浏览商品的设置规则无效");
    Validate.isTrue(tenantSetting.getUnreviewedCustomerBrowseProduct() == null || BrowseProductRuleEnum.findEnumByRuleCode(tenantSetting.getUnreviewedCustomerBrowseProduct()) != null,
            "客户注册订货平台以后-未通过审核前浏览商品的设置规则无效");

    Validate.notNull(tenantSetting.getTenantCode(), "租户编号不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(tenantSetting.getTenantCode() == null || tenantSetting.getTenantCode().length() <= 64, "租户编号长度不能超过64");
    Validate.isTrue(tenantSetting.getWarehouseLeDisplayText() == null || tenantSetting.getWarehouseLeDisplayText().length() <= 12, "客户订货端的库存显示的自定义文本 长度不能超过12");
    Validate.isTrue(tenantSetting.getWarehouseGtDisplayText() == null || tenantSetting.getWarehouseGtDisplayText().length() <= 12, "客户订货端的库存显示的自定义文本 长度不能超过12");
    //如果设置为多仓 仓库区域必填
    //查询租户库存配置 是否是多仓库 (true多仓库  false单仓库)
    Boolean multipleWarehouses = DealerTenantUtils.getMultipleWarehouses();
    multipleWarehouses = BooleanUtils.toBooleanDefaultIfNull(multipleWarehouses, false);
    if (multipleWarehouses){
      //查询租户下所有仓库 判断是否有没有区域的仓库
      if (!CollectionUtils.isEmpty(tenantWarehouseEventListeners)) {
        for (TenantWarehouseEventListener tenantWarehouseEventListener : tenantWarehouseEventListeners) {
          tenantWarehouseEventListener.isSetArea(tenantSetting.getTenantCode());
        }
      }
    }
  }

  private TenantSetting updateForm(TenantSetting tenantSetting) {
    // ===================基本信息
    String currentId = tenantSetting.getId();
    Optional<TenantSetting> op_currentTenantSetting = this.tenantSettingRepository.findById(currentId);
    TenantSetting currentTenantSetting = op_currentTenantSetting.orElse(null);
    Validate.notNull(currentTenantSetting, "未发现指定的原始模型对象信息");
    // 开始赋值——更新时间与更新人
    Date now = new Date();

    currentTenantSetting.setModifyAccount(SecurityUtils.getUserAccount());
    currentTenantSetting.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentTenantSetting.setWarehouseStatus(tenantSetting.getWarehouseStatus());
    currentTenantSetting.setCashPayStatus(tenantSetting.getCashPayStatus());
    currentTenantSetting.setOrderPartialPayStatus(tenantSetting.getOrderPartialPayStatus());
    currentTenantSetting.setGoodsBeforePayStatus(tenantSetting.getGoodsBeforePayStatus());
    currentTenantSetting.setAccountPeriodTypeStatus(tenantSetting.getAccountPeriodTypeStatus());
    currentTenantSetting.setCreditBillStatus(tenantSetting.getCreditBillStatus());

    currentTenantSetting.setOrderRequiredReceiptAddressStatus(tenantSetting.getOrderRequiredReceiptAddressStatus());
    currentTenantSetting.setOrderRequiredReceiptAddressValue(tenantSetting.getOrderRequiredReceiptAddressValue());
    currentTenantSetting.setOrderRequiredDeliveryDateStatus(tenantSetting.getOrderRequiredDeliveryDateStatus());
    currentTenantSetting.setOrderRequiredDeliveryDateValue(tenantSetting.getOrderRequiredDeliveryDateValue());
    currentTenantSetting.setCustomerCancelOrderStatus(tenantSetting.getCustomerCancelOrderStatus());
    currentTenantSetting.setMinimumOrderAmountStatus(tenantSetting.getMinimumOrderAmountStatus());
    currentTenantSetting.setMinimumOrderAmount(tenantSetting.getMinimumOrderAmount());
    currentTenantSetting.setOrderTimeStatus(tenantSetting.getOrderTimeStatus());
    currentTenantSetting.setOrderTime(tenantSetting.getOrderTime());
    currentTenantSetting.setAutomaticReceiptStatus(tenantSetting.getAutomaticReceiptStatus());
    currentTenantSetting.setAutomaticReceiptValue(tenantSetting.getAutomaticReceiptValue());
    currentTenantSetting.setAutomaticCancelOrderStatus(tenantSetting.getAutomaticCancelOrderStatus());
    currentTenantSetting.setAutomaticCancelOrderValue(tenantSetting.getAutomaticCancelOrderValue());
    currentTenantSetting.setBanModifyReceiptAddressStatus(tenantSetting.getBanModifyReceiptAddressStatus());
    currentTenantSetting.setReturnLimitStatus(tenantSetting.getReturnLimitStatus());
    currentTenantSetting.setReturnLimitValue(tenantSetting.getReturnLimitValue());
    currentTenantSetting.setWarehouseOccupyStatus(tenantSetting.getWarehouseOccupyStatus());
    currentTenantSetting.setNegativeWarehouseStatus(tenantSetting.getNegativeWarehouseStatus());
    currentTenantSetting.setWarehouseGtDisplayOption(tenantSetting.getWarehouseGtDisplayOption());
    currentTenantSetting.setWarehouseGtDisplayText(tenantSetting.getWarehouseGtDisplayText());
    currentTenantSetting.setWarehouseGtDisplayThreshold(tenantSetting.getWarehouseGtDisplayThreshold());
    currentTenantSetting.setWarehouseLeDisplayOption(tenantSetting.getWarehouseLeDisplayOption());
    currentTenantSetting.setWarehouseLeDisplayText(tenantSetting.getWarehouseLeDisplayText());
    currentTenantSetting.setWarehouseLeDisplayThreshold(tenantSetting.getWarehouseLeDisplayThreshold());
    currentTenantSetting.setOverflowWarehouseStatus(tenantSetting.getOverflowWarehouseStatus());
    currentTenantSetting.setDistanceStatus(tenantSetting.getDistanceStatus());
    currentTenantSetting.setDistanceThreshold(tenantSetting.getDistanceThreshold());
    currentTenantSetting.setBaseSettingSwitch(tenantSetting.getBaseSettingSwitch());
    currentTenantSetting.setUnreviewedCustomerBrowseProduct(tenantSetting.getUnreviewedCustomerBrowseProduct());
    currentTenantSetting.setTouristBrowseProduct(tenantSetting.getTouristBrowseProduct());

    if (!CollectionUtils.isEmpty(tenantSetting.getBaseSettingSwitchMap())) {
      Long baseSettingSwitch = TenantSettingEnumUtil.setSettingSwitchByEnums(tenantSetting.getBaseSettingSwitch(), tenantSetting.getBaseSettingSwitchMap());
      tenantSetting.setBaseSettingSwitch(baseSettingSwitch);
    }

    // 如果是需要修改仓库状态的场景， 且是由关闭到开启状态则，要需要满足厂库库存必须大于零（由开启到关闭则不需要）
    if (!Objects.equal(currentTenantSetting.getWarehouseStatus(),
            currentTenantSetting.getWarehouseStatus())) {
      this.tenantSettingRepository.saveAndFlush(currentTenantSetting);
      TenantSettingVo tenantSettingVo = nebulaToolkitService.copyObjectByBlankList(currentTenantSetting, TenantSettingVo.class, HashSet.class, ArrayList.class);
      // 通知事件触发
      this.notifyWarehouseEvent(tenantSettingVo);
    }
    return currentTenantSetting;
  }


  @Override
  public TenantSetting findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    Optional<TenantSetting> op = tenantSettingRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    TenantSetting current = this.findById(id);
    if (current != null) {
      this.tenantSettingRepository.delete(current);
    }
  }

  @Override
  public TenantSetting findByTenantCode(String tenantCode) {
    if (StringUtils.isBlank(tenantCode)) {
      return null;
    }
    return this.tenantSettingRepository.findByTenantCode(tenantCode);
  }

  @Override
  @Transactional
  public TenantSetting updateWarehouseStatus(String tenantCode, Boolean status) {
    TenantSetting currentTenantSetting = this.findByTenantCode(tenantCode);
    Validate.notNull(currentTenantSetting, "未发现指定的原始模型对象信息");
    Validate.notNull(status, "指定的状态值不能为空");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentTenantSetting.setModifyAccount(SecurityUtils.getUserAccount());
    currentTenantSetting.setModifyTime(now);
    currentTenantSetting.setWarehouseStatus(status);
    this.tenantSettingRepository.saveAndFlush(currentTenantSetting);

    TenantSettingVo tenantSettingVo = nebulaToolkitService.copyObjectByWhiteList(currentTenantSetting, TenantSettingVo.class, HashSet.class, ArrayList.class);
    this.notifyWarehouseEvent(tenantSettingVo);
    return currentTenantSetting;
  }

  @Override
  public TenantSetting findSettingImgByTenantCode() {
    String tenantCode = TenantUtils.getTenantCode();
    if (StringUtils.isBlank(tenantCode)) {
      return null;
    }
    return tenantSettingRepository.findByTenantCode(tenantCode);
  }

  /**
   * 仓库状态修改事件通知
   */
  private void notifyWarehouseEvent(TenantSettingVo tenantSettingVo) {
    if (!CollectionUtils.isEmpty(tenantWarehouseEventListeners)) {
      for (TenantWarehouseEventListener tenantWarehouseEventListener : tenantWarehouseEventListeners) {
        tenantWarehouseEventListener.onChange(tenantSettingVo);
      }
    }
  }

  /**
   * 租户设置信息变动通知
   *
   * @param oldTenantSetting
   * @param newTenantSetting
   */
  private void notifySettingChangeEvent(TenantSettingVo oldTenantSetting, TenantSettingVo newTenantSetting) {
    if (!CollectionUtils.isEmpty(tenantSettingEventListeners)) {
      for (TenantSettingEventListener tenantSettingEventListener : tenantSettingEventListeners) {
        tenantSettingEventListener.onChange(oldTenantSetting, newTenantSetting);
      }
    }
  }

}
