package com.bizunited.empower.business.tenant.service.internal;

import com.bizunited.empower.business.tenant.common.enums.TenantSmsBusinessType;
import com.bizunited.empower.business.tenant.service.TenantSmsService;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.core.service.sms.SmsService;
import com.bizunited.platform.core.service.sms.SmsTypeEnums;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import static com.bizunited.empower.business.tenant.common.constant.Constants.DEFAULT_SMS_CODE_LENGTH;
import static com.bizunited.empower.business.tenant.common.constant.Constants.DEFAULT_SMS_EXPIRE_TIME;
import static com.bizunited.empower.business.tenant.common.constant.RedisKeys.TENANT_SMS_CODE_MAP_KEY;
import static com.bizunited.empower.business.tenant.common.constant.RedisKeys.TENANT_SMS_CODE_PREFIX;
import static com.bizunited.empower.business.tenant.common.constant.RedisKeys.TENANT_SMS_MOBILE_MAP_KEY;

/**
 * 客户短信服务
 * @Author: Paul Chan
 * @Date: 2021/4/17 上午10:38
 */
@Service
public class TenantSmsServiceImpl implements TenantSmsService {

  @Autowired
  private SmsService smsService;
  @Autowired
  private RedisMutexService redisMutexService;

  @Override
  public void sendValidCode(String tenantCode, String mobile, TenantSmsBusinessType businessType) {
    this.sendValidCode(tenantCode, mobile, businessType, DEFAULT_SMS_CODE_LENGTH, DEFAULT_SMS_EXPIRE_TIME);
  }

  @Override
  public void sendValidCode(String tenantCode, String mobile, TenantSmsBusinessType businessType, int codeLength, long expireTime) {
    Validate.isTrue(codeLength > 0, "验证码长度要大于0");
    String code = RandomStringUtils.randomNumeric(codeLength);
    this.sendValidCode(tenantCode, mobile, businessType, code, expireTime);
  }

  @Override
  public void sendValidCode(String tenantCode, String mobile, TenantSmsBusinessType businessType, String validCode, long expireTime) {
    Validate.notBlank(tenantCode, "经销商编码不能为空");
    Validate.notNull(businessType, "短信业务类型不能为空");
    Validate.notBlank(validCode, "验证码不能为空");
    Validate.notBlank(mobile, "手机号不能为空");
    String redisKey = String.format(TENANT_SMS_CODE_PREFIX, tenantCode, businessType.getType());
    redisMutexService.setMCode(redisKey, TENANT_SMS_CODE_MAP_KEY, validCode, expireTime);
    redisMutexService.setMCode(redisKey, TENANT_SMS_MOBILE_MAP_KEY, mobile, expireTime);
    smsService.sendSms(mobile, validCode, SmsTypeEnums.INFO);
  }

  @Override
  public void verifyValidCode(String tenantCode, TenantSmsBusinessType businessType, String validCode) {
    Validate.notBlank(tenantCode, "经销商编码不能为空");
    Validate.notNull(businessType, "短信业务类型不能为空");
    Validate.notBlank(validCode, "验证码不能为空");
    String redisKey = String.format(TENANT_SMS_CODE_PREFIX, tenantCode, businessType.getType());
    String code = redisMutexService.getMCode(redisKey, TENANT_SMS_CODE_MAP_KEY);
    Validate.isTrue(validCode.equals(code), "验证码不正确");
  }

  @Override
  public void verifyValidCode(String tenantCode, String mobile, TenantSmsBusinessType businessType, String validCode) {
    Validate.notBlank(tenantCode, "经销商编码不能为空");
    Validate.notBlank(mobile, "手机号码不能为空");
    Validate.notNull(businessType, "短信业务类型不能为空");
    Validate.notBlank(validCode, "验证码不能为空");
    String redisKey = String.format(TENANT_SMS_CODE_PREFIX, tenantCode, businessType.getType());
    String code = redisMutexService.getMCode(redisKey, TENANT_SMS_CODE_MAP_KEY);
    String codeMobile = redisMutexService.getMCode(redisKey, TENANT_SMS_MOBILE_MAP_KEY);
    Validate.isTrue(validCode.equals(code), "验证码不正确");
    Validate.isTrue(mobile.equals(codeMobile), "手机号码与接收验证码手机号不匹配");
  }

}
