package com.bizunited.empower.business.tenant.service.internal;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.empower.business.tenant.common.constant.RedisKeys;
import com.bizunited.empower.business.tenant.dto.TenantInfoDto;
import com.bizunited.empower.business.tenant.dto.TenantInfoEventDto;
import com.bizunited.empower.business.tenant.entity.TenantInfo;
import com.bizunited.empower.business.tenant.entity.TenantSetting;
import com.bizunited.empower.business.tenant.repository.TenantInfoRepository;
import com.bizunited.empower.business.tenant.service.TenantInfoService;
import com.bizunited.empower.business.tenant.service.TenantSettingService;
import com.bizunited.empower.business.tenant.vo.TenantInfoVo;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.script.context.InvokeParams;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantContextHolder;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.platform.rbac2.sdk.service.RoleVoCacheService;
import com.bizunited.platform.rbac2.sdk.vo.RoleVo;
import com.bizunited.platform.security.sdk.vo.LoginDetails;
import com.bizunited.platform.user2.sdk.enums.UserStatusEnum;
import com.bizunited.platform.user2.sdk.service.user.UserVoService;
import com.bizunited.platform.user2.sdk.vo.UserVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.redisson.api.RTopic;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.bizunited.empower.business.tenant.common.constant.RedisKeys.TOPIC_TENANT_EVENT;
import static com.bizunited.platform.common.constant.Constants.DEFAULT_PAGEABLE;

/**
 * TenantInfo业务模型的服务层接口实现
 *
 * @author saturn keller
 */
@Service("TenantInfoServiceImpl")
public class TenantInfoServiceImpl implements TenantInfoService {
  @Autowired
  private TenantInfoRepository tenantInfoRepository;
  @Autowired
  private UserVoService userService;
  @Autowired
  private RoleVoCacheService roleVoService;
  @Autowired
  private RedissonClient redissonClient;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private TenantSettingService tenantSettingService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Transactional
  @Override
  public TenantInfo create(TenantInfoDto tenantInfoDto) {
    return this.createForm(tenantInfoDto);
  }

  private TenantInfo createForm(TenantInfoDto tenantInfoDto) {
    Date now = new Date();
    TenantInfo tenantInfo = tenantInfoDto.getTenantInfo();
    tenantInfo.setCreateTime(now);
    tenantInfo.setModifyTime(now);
    if(StringUtils.isBlank(tenantInfo.getAppCode())){
      tenantInfo.setAppCode(TenantUtils.getAppCode());
    }
    this.createValidation(tenantInfo, tenantInfoDto.getUserVo());
    tenantInfo.setCreateAccount(tenantInfoDto.getUserVo().getPhone());
    tenantInfo.setModifyAccount(tenantInfoDto.getUserVo().getPhone());
    this.tenantInfoRepository.saveAndFlush(tenantInfo);

    // 初始化租户配置信息
    TenantSetting tenantSetting = new TenantSetting();
    tenantSetting.setTenantCode(tenantInfo.getTenantCode());
    tenantSetting.setAccountPeriodTypeStatus(false);
    tenantSetting.setAutomaticCancelOrderStatus(false);
    tenantSetting.setAutomaticCancelOrderValue(0);
    tenantSetting.setAutomaticReceiptStatus(false);
    tenantSetting.setAutomaticReceiptValue(0);
    tenantSetting.setBanModifyReceiptAddressStatus(false);
    tenantSetting.setCashPayStatus(false);
    tenantSetting.setCreditBillStatus(false);
    tenantSetting.setCustomerCancelOrderStatus(false);
    tenantSetting.setDistanceStatus(false);
    tenantSetting.setGoodsBeforePayStatus(false);
    tenantSetting.setMinimumOrderAmount(BigDecimal.ZERO);
    tenantSetting.setMinimumOrderAmountStatus(false);
    tenantSetting.setNegativeWarehouseStatus(false);
    tenantSetting.setOrderPartialPayStatus(false);
    tenantSetting.setOrderRequiredDeliveryDateStatus(false);
    tenantSetting.setOrderRequiredDeliveryDateValue(0);
    tenantSetting.setOrderRequiredReceiptAddressStatus(false);
    tenantSetting.setOrderRequiredReceiptAddressValue(0);
    tenantSetting.setOrderTimeStatus(false);
    tenantSetting.setOverflowWarehouseStatus(false);
    tenantSetting.setReturnLimitStatus(false);
    tenantSetting.setReturnLimitValue(0);
    tenantSetting.setWarehouseOccupyStatus(false);
    tenantSetting.setWarehouseStatus(false);
    tenantSetting.setAuditNodeSettings(this.defaultAuditNodeSettings());
    tenantSetting.setReturnAuditNodeSettings(this.defaultReturnAuditNodeSettings());
    tenantSetting.setBaseSettingSwitch(0L);

    this.tenantSettingService.save(tenantSetting);

    //发送redis订阅通知，执行经销商人员、权限等初始化
    this.publishTenantInitEvent(1, tenantInfo, tenantInfoDto.getUserVo());
    return tenantInfo;
  }

  /**
   * 在创建一个新的TenantInfo模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(TenantInfo tenantInfo, UserVo userVo) {
    Validate.notNull(tenantInfo, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(tenantInfo.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    tenantInfo.setId(null);
    Validate.notBlank(tenantInfo.getTenantName(), "添加信息时，租户名称不能为空！");
    Validate.notNull(tenantInfo.getTenantStatus(), "添加信息时，租户状态不能为空！");
    Validate.notBlank(tenantInfo.getTenantType(), "添加信息时，租户类型不能为空！");
    Validate.notNull(tenantInfo.getTenantExpired(), "添加信息时，租户有效期不能为空！");
    Validate.notBlank(tenantInfo.getAppCode(),"添加信息时候，厂商编码不能为空！");
    Validate.notBlank(tenantInfo.getTenantCode(),"添加信息时候，租户编码不能为空！");

    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(tenantInfo.getTenantName() == null || tenantInfo.getTenantName().length() < 255, "租户名称,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(tenantInfo.getAppCode() == null || tenantInfo.getAppCode().length() < 255, "厂商编码,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(tenantInfo.getTenantType() == null || tenantInfo.getTenantType().length() < 32, "租户类型,在进行添加时填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(tenantInfo.getTenantDescription() == null || tenantInfo.getTenantDescription().length() < 1024, "租户介绍,在进行添加时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(tenantInfo.getLogoRelativePath() == null || tenantInfo.getLogoRelativePath().length() < 1024, "logo相对路径,在进行添加时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(tenantInfo.getLogoFileName() == null || tenantInfo.getLogoFileName().length() < 1024, "logo文件名,在进行添加时填入值超过了限定长度(1024)，请检查!");
    Validate.notNull(userVo, "管理员信息不能为空");
    Validate.notBlank(userVo.getPhone(), "管理员手机号不能为空");

    long count = tenantInfoRepository.countByTenantCode(tenantInfo.getTenantCode());
    Validate.isTrue(count == 0, "经销商编码重复");
    count = tenantInfoRepository.countByTenantName(tenantInfo.getTenantName());
    Validate.isTrue(count == 0, "经销商名称重复");
  }


  private String defaultAuditNodeSettings(){
    JSONArray array = new JSONArray();
    JSONObject node1 = new JSONObject();
    node1.put("node","订单审核");
    JSONArray roles = new JSONArray();
    roles.add("ORDERAUDITOR");
    node1.put("roleCodes",roles);
    node1.put("sorts",1);

    JSONObject node2 = new JSONObject();
    node2.put("node","财务审核");
    JSONArray roles2 = new JSONArray();
    roles2.add("FINANCEAUDITOR");
    node2.put("roleCodes",roles2);
    node2.put("sorts",2);

    JSONObject node3 = new JSONObject();
    node2.put("node","发货审核");
    JSONArray roles3 = new JSONArray();
    roles3.add("DELIVERYAUDITOR");
    node3.put("roleCodes",roles3);
    node3.put("sorts",3);

    array.add(node1);
    array.add(node2);
    array.add(node3);
    return array.toJSONString();
  }

  private String defaultReturnAuditNodeSettings(){
    JSONArray array = new JSONArray();
    JSONObject node1 = new JSONObject();
    node1.put("node","订单审核");
    JSONArray roles = new JSONArray();
    roles.add("ORDERAUDITOR");
    node1.put("roleCodes",roles);
    node1.put("sorts",1);

    array.add(node1);
    return array.toJSONString();
  }

  @Transactional
  @Override
  public TenantInfo update(TenantInfoDto tenantInfoDto) {
    return this.updateForm(tenantInfoDto);
  }

  private TenantInfo updateForm(TenantInfoDto tenantInfoDto) {
    TenantInfo tenantInfo = tenantInfoDto.getTenantInfo();
    this.updateValidation(tenantInfo);
    // ===================基本信息
    String currentId = tenantInfo.getId();
    Optional<TenantInfo> op_currentTenantInfo = this.tenantInfoRepository.findById(currentId);
    TenantInfo currentTenantInfo = op_currentTenantInfo.orElse(null);
    Validate.notNull(currentTenantInfo, "未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentTenantInfo.setModifyAccount("System");
    currentTenantInfo.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentTenantInfo.setTenantStatus(tenantInfo.getTenantStatus());
    currentTenantInfo.setTenantType(tenantInfo.getTenantType());
    currentTenantInfo.setTenantExpired(tenantInfo.getTenantExpired());
    currentTenantInfo.setTenantDescription(tenantInfo.getTenantDescription());
    currentTenantInfo.setLogoRelativePath(tenantInfo.getLogoRelativePath());
    currentTenantInfo.setLogoFileName(tenantInfo.getLogoFileName());

    this.tenantInfoRepository.saveAndFlush(currentTenantInfo);
    return currentTenantInfo;
  }

  /**
   * 在更新一个已有的TenantInfo模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(TenantInfo tenantInfo) {
    Validate.isTrue(!StringUtils.isBlank(tenantInfo.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(tenantInfo.getTenantName(), "修改信息时，租户名称不能为空！");
    Validate.notBlank(tenantInfo.getTenantCode(), "修改信息时，租户编码不能为空！");
    Validate.notNull(tenantInfo.getTenantStatus(), "修改信息时，租户状态不能为空！");
    Validate.notBlank(tenantInfo.getTenantType(), "修改信息时，租户类型不能为空！");
    Validate.notNull(tenantInfo.getTenantExpired(), "修改信息时，租户有效期不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(tenantInfo.getTenantName() == null || tenantInfo.getTenantName().length() < 255, "租户名称,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(tenantInfo.getTenantCode() == null || tenantInfo.getTenantCode().length() < 64, "租户编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(tenantInfo.getTenantType() == null || tenantInfo.getTenantType().length() < 32, "租户类型,在进行修改时填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(tenantInfo.getTenantDescription() == null || tenantInfo.getTenantDescription().length() < 1024, "租户介绍,在进行修改时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(tenantInfo.getLogoRelativePath() == null || tenantInfo.getLogoRelativePath().length() < 1024, "logo相对路径,在进行修改时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(tenantInfo.getLogoFileName() == null || tenantInfo.getLogoFileName().length() < 1024, "logo文件名,在进行修改时填入值超过了限定长度(1024)，请检查!");
  }

  @Override
  public TenantInfo findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.tenantInfoRepository.findDetailsById(id);
  }

  @Override
  public TenantInfo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    Optional<TenantInfo> op = tenantInfoRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  public TenantInfo findByCode(String code) {
    if (StringUtils.isBlank(code)) {
      return null;
    }
    return this.tenantInfoRepository.findByCode(code);
  }

  @Override
  public Page<TenantInfo> findByConditions(Pageable pageable, InvokeParams conditions) {
    pageable = ObjectUtils.defaultIfNull(pageable, DEFAULT_PAGEABLE);
    return tenantInfoRepository.queryPage(pageable, conditions);
  }

  @Override
  public List<TenantInfo> findByAppCode(String appCode) {
    if(StringUtils.isBlank(appCode)) {
      return Lists.newArrayList();
    }
    return tenantInfoRepository.findByAppCode(appCode);
  }

  @Override
  public List<TenantInfo> findByUserPhone(String phone) {
    if(StringUtils.isBlank(phone)) {
      return Lists.newArrayList();
    }
    List<UserVo> users = Lists.newArrayList();
    Set<UserVo> enableUsers = userService.findByPhoneAndStatus(phone, UserStatusEnum.ENABLE.getStatus());
    Set<UserVo> unActiveUsers = userService.findByPhoneAndStatus(phone, UserStatusEnum.NOT_ACTIVATED.getStatus());
    users.addAll(ObjectUtils.defaultIfNull(enableUsers, Lists.newArrayList()));
    users.addAll(ObjectUtils.defaultIfNull(unActiveUsers, Lists.newArrayList()));
    if(CollectionUtils.isEmpty(users)) {
      return Lists.newArrayList();
    }
    List<String> tenantCodes = users.stream().map(UserVo::getTenantCode).collect(Collectors.toList());
    return tenantInfoRepository.findByCodes(tenantCodes);
  }

  @Override
  public List<TenantInfo> findByUserAccount(String account) {
    if(StringUtils.isBlank(account)) {
      return Lists.newArrayList();
    }
    Set<UserVo> users = userService.findByAccountAndStatus(account, UserStatusEnum.ENABLE.getStatus());
    if(CollectionUtils.isEmpty(users)) {
      return Lists.newArrayList();
    }
    List<String> tenantCodes = users.stream().map(UserVo::getTenantCode).collect(Collectors.toList());
    return tenantInfoRepository.findByCodes(tenantCodes);
  }

  @Override
  public TenantInfoVo switchTenant(String targetTenantCode) {
    /*
     * 进行租户切换的前提条件是：
     * 1、当前操作者已经登录（既是存在LoginDetails信息）
     * 2、当前操作者在新的tenantCode下，也存在账户，按照tenantCode和account进行查询
     *
     * 租户的切换动作为：
     * 1、从上下文取出登录信息详情，并对登录信息详情进行修改（主要是改他的tenantCode信息）
     * 包括重新为鉴权上下文生成新的AuthenticationToken
     * 2、基于TenantContextHolder对TenantContext上下文信息进行修改
     * 并构造返回信息
     * */
    Validate.notBlank(targetTenantCode, "新切换的租户编码不能为空");
    SecurityContext securityContext = SecurityContextHolder.getContext();
    Validate.notNull(securityContext , "鉴权上下文未被发现，请检查!!");
    Authentication authentication = securityContext.getAuthentication();
    Validate.notNull(authentication , "鉴权上下文未被发现，请检查!!");
    Object authenticationDetails = authentication.getDetails();
    Validate.isTrue(authenticationDetails instanceof LoginDetails , "登录详情未被发现，请检查!!");
    LoginDetails loginDetails = (LoginDetails)authenticationDetails;
    String account = loginDetails.getAccount();
    // 切换后的用户状态也要正确
    UserVo currentUser = this.userService.findByTenantCodeAndAccount(targetTenantCode, account);
    Validate.notNull(currentUser , "未找到用户信息或用户状态不正确，不能进行切换");
    Validate.isTrue(currentUser.getUseStatus().intValue() == UserStatusEnum.ENABLE.getStatus() , "当前登录用户状态错误，不能进行切换");

    // 1、========
    loginDetails.setTenantCode(targetTenantCode);
    // 重新准备角色信息
    Set<RoleVo> currentRoles = this.roleVoService.findByTenantCodeAndUserAccount(targetTenantCode, account);
    Validate.isTrue(!CollectionUtils.isEmpty(currentRoles) , "切换用户账号后，未发现该用户关联的任何功能角色，不允许进行切换");
    List<SimpleGrantedAuthority> authorities = Lists.newLinkedList();
    for (RoleVo role : currentRoles) {
      SimpleGrantedAuthority authoritie = new SimpleGrantedAuthority(role.getRoleCode());
      authorities.add(authoritie);
    }
    // 写入新的用户信息
    UsernamePasswordAuthenticationToken token = new UsernamePasswordAuthenticationToken(currentUser.getAccount(), currentUser.getPassword(), authorities);
    token.setDetails(loginDetails);
    securityContext.setAuthentication(token);

    // 2、=======
    TenantContextHolder.setTenant(targetTenantCode);
    TenantInfo tenantInfo = tenantInfoRepository.findByCode(targetTenantCode);
    TenantInfoVo tenantInfoVo = nebulaToolkitService.copyObjectByWhiteList(tenantInfo, TenantInfoVo.class, LinkedHashSet.class, ArrayList.class);
    return tenantInfoVo;
  }

  @Override
  public TenantInfoVo findTenantLogoByTenantCode() {
    TenantInfo tenantInfo = tenantInfoRepository.findByCode(TenantUtils.getTenantCode());
    if (Objects.isNull(tenantInfo)){
      return null;
    }
    TenantInfoVo tenantInfoVo = new TenantInfoVo();
    tenantInfoVo.setLogoFileName(tenantInfo.getLogoFileName());
    tenantInfoVo.setLogoRelativePath(tenantInfo.getLogoRelativePath());
    return tenantInfoVo;
  }

  @Override
  public List<TenantInfo> findTenantInfos() {
    return tenantInfoRepository.findByTenantStatusAndAppCode(true, TenantUtils.getAppCode());
  }

  @Override
  public TenantInfoVo findLoginLogoByAppCode() {
    // 获取缓存数据
    String appCode = TenantUtils.getAppCode();
    String redisKey = String.format(RedisKeys.APP_CODE_LOGO_LOGIN_CACHE_PREFIX, appCode);
    String data = redisMutexService.getMCode(RedisKeys.APP_CODE_LOGO_LOGIN_CACHE_PREFIX, appCode);
    if (StringUtils.isNotBlank(data)){
      return JSONObject.parseObject(data, TenantInfoVo.class);
    }

    // DB没有数据就设置一个空数据（其他地方进行维护时同时同步缓存即可替换此处的空值）
    List<TenantInfo> tenantInfos = tenantInfoRepository.findByAppCode(appCode);
    TenantInfo tenantInfo = tenantInfos.stream().filter(item ->
                    !StringUtils.isAnyBlank(item.getIconFileName(), item.getIconRelativePath(),
                            item.getLoginLogoFileName(), item.getLoginLogoRelativePath()))
            .findFirst()
            .orElse(null);
    if (Objects.isNull(tenantInfo)){
      redisMutexService.setMCode(redisKey, appCode,
              JSONObject.toJSONString(new TenantInfoVo()), TimeUnit.MINUTES.toMillis(3));
      return null;
    }

    // DB有就转换数据并且更新至缓存
    TenantInfoVo tenantInfoVo = new TenantInfoVo();
    tenantInfoVo.setAppCode(tenantInfo.getAppCode());
    tenantInfoVo.setIconFileName(tenantInfo.getIconFileName());
    tenantInfoVo.setIconRelativePath(tenantInfo.getIconRelativePath());
    tenantInfoVo.setLoginLogoFileName(tenantInfo.getLoginLogoFileName());
    tenantInfoVo.setLoginLogoRelativePath(tenantInfo.getLoginLogoRelativePath());
    redisMutexService.setMCode(redisKey, appCode, JSONObject.toJSONString(tenantInfoVo), TimeUnit.DAYS.toMillis(1));

    return tenantInfoVo;
  }

  /**
   * 通知租户初始化监听者
   *  @param event  事件类型，1：新增，2：修改，3：删除
   * @param tenantInfo 租户信息
   * @param userVo 用户
   */
  private void publishTenantInitEvent(int event, TenantInfo tenantInfo, UserVo userVo) {
    TenantInfoEventDto eventDto = new TenantInfoEventDto();
    TenantInfoDto tenantInfoDto = new TenantInfoDto();
    tenantInfoDto.setTenantInfo(tenantInfo);
    tenantInfoDto.setUserVo(userVo);
    eventDto.setEvent(event);
    eventDto.setTenantInfo(tenantInfoDto);
    RTopic topic = redissonClient.getTopic(TOPIC_TENANT_EVENT);
    topic.publish(eventDto);
  }
}
