package com.bizunited.empower.business.tenant.utils;

import com.bizunited.empower.business.tenant.common.enums.TenantSettingSwitchEnum;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.util.CollectionUtils;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 租户功能设置开关枚举转换工具类
 */
public class TenantSettingEnumUtil {

  /**
   * 获取所有设置枚举的默认设置
   *
   * @return 租户所有的默认设置聚合值
   */
  public static Long getDefaultSettingSwitch() {
    long num = 0L;
    for (TenantSettingSwitchEnum settingSwitchEnum : TenantSettingSwitchEnum.values()) {
      Integer enable = settingSwitchEnum.getDefaultValue();
      if (enable != 0) {
        Long binaryBit = settingSwitchEnum.getBinaryBit();
        num += (1L << binaryBit);
      }
    }
    return num;
  }

  /**
   * 根据传入的设置项枚举列表转换为聚合聚合值（注意：此方法会基于设置类列表将缺失设置项枚举使用默认值填充，枚举列表为空将使用默认的枚举列表！！！）
   *
   * @param baseSettingSwitchMap 需要转换的配置项枚举列表
   * @return 转为Long类型的租户配置聚合值
   */
  public static Long getSettingSwitchByEnums(Map<TenantSettingSwitchEnum, Integer> baseSettingSwitchMap) {
    if (CollectionUtils.isEmpty(baseSettingSwitchMap)) {
      return getDefaultSettingSwitch();
    }

    Map<Long, Integer> tenantSettingSwitchEnumMap = new HashMap<>(baseSettingSwitchMap.size());
    baseSettingSwitchMap.forEach((k, v) -> {
      TenantSettingSwitchEnum enumByBinaryBit =
              TenantSettingSwitchEnum.findTenantSettingSwitchEnumByBinaryBit(k.getBinaryBit());
      Validate.notNull(enumByBinaryBit, "未找到指定:%s 序号的配置项枚举，请检查！！！", k.getBinaryBit());

      tenantSettingSwitchEnumMap.put(enumByBinaryBit.getBinaryBit(), v);
    });

    // 遍历所有的设置枚举将，指定的枚举设置为指定的值。没有指定的枚举使用默认的值
    long baseSettingSwitch = 0L;
    for (TenantSettingSwitchEnum settingSwitchEnum : TenantSettingSwitchEnum.values()) {
      if (tenantSettingSwitchEnumMap.containsKey(settingSwitchEnum.getBinaryBit())) {
        settingSwitchEnum.setDefaultValue(tenantSettingSwitchEnumMap.get(settingSwitchEnum.getBinaryBit()));
      }
      Integer enable = settingSwitchEnum.getDefaultValue();
      if (enable != 0) {
        Long binaryBit = settingSwitchEnum.getBinaryBit();
        baseSettingSwitch += (1L << binaryBit);
      }
    }
    return baseSettingSwitch;
  }

  /**
   * 根据设置聚合参数转换为设置项枚举列表(注意：设置聚合参数为空则返回默认设置列表值，缺失的设置将使用默认设置值填充！！！)
   *
   * @param baseSettingSwitch 租户配置聚合值
   * @return 租户设置枚举列表
   */
  public static Map<TenantSettingSwitchEnum, Integer> getSettingSwitchEnumsBySetting(Long baseSettingSwitch) {
    // 为空则返回所有的枚举
    if (baseSettingSwitch == null) {
      return Arrays.stream(TenantSettingSwitchEnum.values())
              .collect(Collectors.toMap(Function.identity(), TenantSettingSwitchEnum::getDefaultValue));
    }

    int[] binaryByteArray = longToBinaryByteArray(baseSettingSwitch, null);
    Validate.isTrue(Objects.nonNull(binaryByteArray) && binaryByteArray.length > 0,
            "通过租户的配置值：%s, 获取到的配置信息数组为空", baseSettingSwitch);

    // 将租户设置的二进制数据映射到全局配置枚举中，缺失的bit位使用默认的枚举
    HashMap<TenantSettingSwitchEnum, Integer> baseSettingSwitchMap = new HashMap<>();
    for (TenantSettingSwitchEnum tenantSettingSwitchEnum : TenantSettingSwitchEnum.values()) {
      if (tenantSettingSwitchEnum.getBinaryBit() < binaryByteArray.length) {
        baseSettingSwitchMap.put(tenantSettingSwitchEnum, binaryByteArray[tenantSettingSwitchEnum.getBinaryBit().intValue()]);
      } else {
        baseSettingSwitchMap.put(tenantSettingSwitchEnum, tenantSettingSwitchEnum.getDefaultValue());
      }
    }
    return baseSettingSwitchMap;
  }

  /**
   * 将指定设置枚举列表值聚合到指定的聚合值中（注意：缺失的设置枚举将使用默认设置枚举值填充！！！）
   *
   * @param baseSettingSwitch    需要填充的目标租户设置聚合值
   * @param baseSettingSwitchMap 填充的源设置枚举列表
   * @return 填充后的租户配置聚合值
   */
  public static Long setSettingSwitchByEnums(Long baseSettingSwitch,
                                             Map<TenantSettingSwitchEnum, Integer> baseSettingSwitchMap) {
    int[] totalBinaries = addMissEnums(baseSettingSwitch);
    Validate.isTrue(Objects.nonNull(totalBinaries) && totalBinaries.length > 0, "填充后的配置信息数组为空");

    // 将指定枚举设置到全局聚合二进制数组中
    baseSettingSwitchMap.forEach((tenantSettingSwitchEnum, isEnable) -> {
      TenantSettingSwitchEnum enumByBinaryBit =
              TenantSettingSwitchEnum.findTenantSettingSwitchEnumByBinaryBit(tenantSettingSwitchEnum.getBinaryBit());

      int binaryBit = enumByBinaryBit.getBinaryBit().intValue();
      Validate.isTrue(totalBinaries.length > binaryBit,
              "补充失败, 需要补充的二进制数据长度：%s 大于了 系统设置的最大长度：%s", binaryBit, totalBinaries.length);

      totalBinaries[binaryBit] = isEnable == 0 ? 0 : 1;
    });

    return binaryByteArrayToLong(totalBinaries);
  }

  /**
   * 将指定设置枚举值聚合到指定的聚合值中（注意：缺失的设置枚举将使用默认设置枚举值填充！！！）
   *
   * @param baseSettingSwitch       需要填充的目标租户设置聚合值
   * @param tenantSettingSwitchEnum 填充的源设置枚举
   * @return 填充后的租户配置聚合值
   */
  public static Long setSettingSwitchByEnum(Long baseSettingSwitch, TenantSettingSwitchEnum tenantSettingSwitchEnum) {
    if (Objects.isNull(baseSettingSwitch) || Objects.isNull(tenantSettingSwitchEnum)) {
      return null;
    }

    int[] totalBinaries = addMissEnums(baseSettingSwitch);
    Validate.isTrue(totalBinaries.length > 0, "获取到的配置信息数组为空");

    TenantSettingSwitchEnum enumByBinaryBit =
            TenantSettingSwitchEnum.findTenantSettingSwitchEnumByBinaryBit(tenantSettingSwitchEnum.getBinaryBit());

    int binaryBit = enumByBinaryBit.getBinaryBit().intValue();
    Validate.isTrue(totalBinaries.length > binaryBit,
            "补充失败, 需要补充的二进制数据长度：%s 大于了 系统设置的最大长度：%s", binaryBit, totalBinaries.length);

    totalBinaries[binaryBit] = tenantSettingSwitchEnum.getDefaultValue() == 0 ? 0 : 1;
    return binaryByteArrayToLong(totalBinaries);
  }

  /**
   * 获取指定设置项在系统中的配置值
   *
   * @param baseSettingSwitch       租户配置聚合值
   * @param tenantSettingSwitchEnum 指定的目标配置枚举
   * @return 配置值
   */
  public static Integer getSettingIsEnableByNum(Long baseSettingSwitch,
                                                TenantSettingSwitchEnum tenantSettingSwitchEnum) {
    if (Objects.isNull(tenantSettingSwitchEnum)) {
      return null;
    }

    TenantSettingSwitchEnum enumByBinaryBit =
            TenantSettingSwitchEnum.findTenantSettingSwitchEnumByBinaryBit(tenantSettingSwitchEnum.getBinaryBit());

    if (Objects.isNull(baseSettingSwitch)) {
      return enumByBinaryBit.getDefaultValue();
    }

    int[] binaryByteArray = longToBinaryByteArray(baseSettingSwitch, TenantSettingSwitchEnum.values().length);

    int binaryBit = enumByBinaryBit.getBinaryBit().intValue();

    return binaryByteArray[binaryBit];
  }

  /**
   * 根据设置枚举聚合值填充为缺失的的设置枚举
   *
   * @param baseSettingSwitch 指定的租户配置值
   * @return 填充后指定的租户配置数组
   */
  private static int[] addMissEnums(Long baseSettingSwitch) {
    if (Objects.isNull(baseSettingSwitch)) {
      Long defaultSettingSwitch = getDefaultSettingSwitch();
      return longToBinaryByteArray(defaultSettingSwitch, TenantSettingSwitchEnum.values().length);
    }

    int[] binaryByteArray = longToBinaryByteArray(baseSettingSwitch, null);
    Validate.isTrue(Objects.nonNull(binaryByteArray) && binaryByteArray.length > 0,
            "通过租户的配置值：%s, 获取到的配置信息数组为空", baseSettingSwitch);

    TenantSettingSwitchEnum[] values = TenantSettingSwitchEnum.values();
    Validate.validIndex(values, binaryByteArray.length - 1,
            "补充失败, 需要补充的二进制数据长度：%s 大于了 系统设置的最大长度：%s", binaryByteArray.length, values.length);

    if (binaryByteArray.length == values.length) {
      return binaryByteArray;
    }

    Long defaultSettingSwitch = getDefaultSettingSwitch();
    int[] totalBinaries = longToBinaryByteArray(defaultSettingSwitch, TenantSettingSwitchEnum.values().length);
    Validate.isTrue(Objects.nonNull(totalBinaries) && totalBinaries.length > 0,
            "获取租户的默认配置信息数组为空");

    for (int i = 0; i < totalBinaries.length; i++) {
      if (binaryByteArray.length <= i) {
        break;
      }
      totalBinaries[i] = binaryByteArray[i];
    }
    return totalBinaries;
  }

  /**
   * long类型转为二进制int数组，超出的索引位为默认值
   *
   * @param baseSettingSwitch 需要转换的long值
   * @param length            指定的数组的长度，不指定则为转换后的默认长度
   * @return 转换后的二进制数组
   */
  private static int[] longToBinaryByteArray(Long baseSettingSwitch, Integer length) {
    if (Objects.isNull(baseSettingSwitch)) {
      return null;
    }

    char[] chars = StringUtils.reverse(Long.toBinaryString(baseSettingSwitch)).toCharArray();
    int[] var = new int[Objects.isNull(length) ? chars.length : length];
    for (int i = 0; i < var.length; i++) {
      if (chars.length <= i) {
        break;
      }
      var[i] = Character.digit(chars[i], 10);
    }
    return var;
  }

  /**
   * 二进制数组转long类型
   *
   * @param binaryArray 需要转换的二进制数组
   * @return 转换后的long值
   */
  private static Long binaryByteArrayToLong(int[] binaryArray) {
    if (Objects.isNull(binaryArray) || binaryArray.length == 0) {
      return null;
    }

    StringBuilder sb = new StringBuilder();
    for (int i = binaryArray.length - 1; i >= 0; i--) {
      sb.append(binaryArray[i]);
    }
    return Long.parseLong(sb.toString(), 2);
  }
}
