package com.bizunited.empower.business.vehicle.repository;

import com.bizunited.empower.business.vehicle.entity.Vehicle;
import com.bizunited.empower.business.vehicle.repository.internal.VehicleRepositoryCustom;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.List;

/**
 * Vehicle业务模型的数据库方法支持
 * @author saturn
 */
@Repository("_VehicleRepository")
public interface VehicleRepository
    extends
      JpaRepository<Vehicle, String>
      ,JpaSpecificationExecutor<Vehicle>
      ,VehicleRepositoryCustom 
  {

  /**
   * 按照主键进行详情查询（包括关联信息）
   * @param id 主键
   * */
  @Query("select distinct vehicle from Vehicle vehicle "
      + " where vehicle.id=:id and vehicle.delete = 0 ")
  Vehicle findDetailsById(@Param("id") String id);

  /**
   * 根据车辆编号和租户编号查询
   * @param vehicleCode
   * @param tenantCode
   * @return
   */
  @Query("from Vehicle v where v.vehicleCode=:vehicleCode and v.tenantCode = :tenantCode and v.delete = 0")
  Vehicle findByVehicleCodeAndTenantCode(@Param("vehicleCode") String vehicleCode, @Param("tenantCode") String tenantCode);

  /**
   * 根据车牌号和租户编号查询
   * @param carNumber
   * @param tenantCode
   * @return
   */
  @Query("from Vehicle v where v.carNumber=:carNumber and v.tenantCode = :tenantCode and v.delete = 0")
  Vehicle findByCarNumberAndTenantCode(@Param("carNumber") String carNumber, @Param("tenantCode") String tenantCode);

  /**
   * 查询租户下所有车辆
   * @param tenantCode
   * @return
   */
  @Query("from Vehicle v where v.tenantCode = :tenantCode and v.vehicleStatus =:vehicleStatus and v.delete = 0 ")
  List<Vehicle> findAllByTenantCodeAndStatus(@Param("tenantCode") String tenantCode, @Param("vehicleStatus") Integer vehicleStatus);

  /**
   * 根据编号更改车辆状态
   * @param vehicleCode
   * @param vehicleStatus
   */
  @Modifying
  @Query(value = "update vehicle  set vehicle_status = :vehicleStatus where vehicle_code = :vehicleCode and tenant_code = :tenantCode", nativeQuery = true)
  void updateVehicleStatusByVehicleCodeAndTenantCode(@Param("vehicleCode") String vehicleCode, @Param("tenantCode") String tenantCode,
                                                     @Param("vehicleStatus") Integer vehicleStatus);

    /**
     * 根据车辆编号更改车辆使用状态
     * @param vehicleCode
     * @param tenantCode
     * @param vehicleTransportStatus
     */
  @Modifying
  @Query(value = "update vehicle  set vehicle_transport_status = :vehicleTransportStatus where vehicle_code = :vehicleCode and tenant_code = :tenantCode", nativeQuery = true)
  void updateVehicleTransportStatus(@Param("vehicleCode") String vehicleCode, @Param("tenantCode") String tenantCode, @Param("vehicleTransportStatus") Integer vehicleTransportStatus);

  /**
   * 根据用户账号查询关联车辆
   * @param userAccount
   * @param tenantCode
   * @return
   */
  @Query(value = "select * from vehicle v" +
          "    left join vehicle_user_mapping vum on v.vehicle_code = vum.vehicle_code" +
          "    where v.tenant_code = :tenantCode" +
          "    and vum.user_account = :userAccount and v.is_delete = 0" ,nativeQuery = true)
  List<Vehicle> findByUserAccount(@Param("userAccount") String userAccount, @Param("tenantCode") String tenantCode);

  @Query("from Vehicle v where v.vehicleCode in(:vehicleCodeList) and v.tenantCode = :tenantCode and v.delete = 0")
  List<Vehicle> findByVehicleCodesAndTenantCode(@Param("vehicleCodeList") List<String> vehicleCodeList, @Param("tenantCode") String tenantCode);
  }