package com.bizunited.empower.business.vehicle.repository;

import com.bizunited.empower.business.vehicle.entity.VehicleUnload;
import com.bizunited.empower.business.vehicle.repository.internal.VehicleUnloadRepositoryCustom;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.List;

/**
 * VehicleUnload业务模型的数据库方法支持
 * @author saturn
 */
@Repository("_VehicleUnloadRepository")
public interface VehicleUnloadRepository
    extends
      JpaRepository<VehicleUnload, String>
      ,JpaSpecificationExecutor<VehicleUnload>
      , VehicleUnloadRepositoryCustom {

  /**
   * 按照主键进行详情查询（包括关联信息）
   * @param id 主键
   * */
  @Query("select distinct vehicleUnload from VehicleUnload vehicleUnload "
      + " where vehicleUnload.id=:id ")
  VehicleUnload findDetailsById(@Param("id") String id);

  @Query("from VehicleUnload e where e.relevanceCode=:relevanceCode and e.vehicleProductType = :vehicleProductType and e.tenantCode=:tenantCode ")
  VehicleUnload findByRelevanceCodeAndVehicleProductTypeAndTenantCode(@Param("relevanceCode") String relevanceCode, @Param("vehicleProductType") Integer vehicleProductType, @Param("tenantCode") String tenantCode);

  /**
   * 根据装货单编号和租户编号查询
   * @param vehicleUnloadCode
   * @param tenantCode
   * @return
   */
  @Query("from VehicleUnload v where v.vehicleUnloadCode=:vehicleUnloadCode and v.tenantCode = :tenantCode")
  VehicleUnload findByVehicleUnloadCodeAndTenantCode(@Param("vehicleUnloadCode") String vehicleUnloadCode, @Param("tenantCode") String tenantCode);

  @Modifying
  @Query(value = "update vehicle_unload  set vehicle_unload_status = :vehicleUnloadStatus where vehicle_unload_code = :vehicleUnloadCode and tenant_code = :tenantCode", nativeQuery = true)
  void updateVehicleUnloadStatus(@Param("vehicleUnloadCode") String vehicleUnloadCode,
                                 @Param("vehicleUnloadStatus") Integer vehicleUnloadStatus,
                                 @Param("tenantCode") String tenantCode);

  /**
   * 根据出车任务编号和卸货单状态查询
   * @param vehicleTaskCode
   * @param vehicleUnloadStatus
   * @param tenantCode
   * @return
   */
  @Query("from VehicleUnload v where v.vehicleTaskCode=:vehicleTaskCode and v.vehicleUnloadStatus = :vehicleUnloadStatus and v.tenantCode = :tenantCode")
  List<VehicleUnload> findByVehicleTaskCodeAndUnloadStatusAndTenantCode(@Param("vehicleTaskCode") String vehicleTaskCode,
                                                                        @Param("vehicleUnloadStatus") Integer vehicleUnloadStatus,
                                                                        @Param("tenantCode") String tenantCode);

  @Query("from VehicleUnload v where v.vehicleTaskCode=:vehicleTaskCode and v.tenantCode = :tenantCode order by v.createTime desc , v.id ")
  List<VehicleUnload> findByVehicleTaskCodeAndTenantCode(@Param("vehicleTaskCode") String vehicleTaskCode, @Param("tenantCode") String tenantCode);

  /**
   * 根据任务编号和车辆编号，查询卸货记录
   * 注：只查询运输途中出货，且已交货状态的出库信息
   * 包含车销出库和配送出库，不包含收车出库
   * @param vehicleTaskCode 出车任务编码
   * @param vehicleCode 车辆编码
   */
  @Query("from VehicleUnload v where v.vehicleTaskCode=:vehicleTaskCode and v.vehicleCode=:vehicleCode and v.tenantCode = :tenantCode and v.vehicleUnloadType in (1,2) and v.vehicleUnloadStatus = 2")
  List<VehicleUnload> findByVehicleTaskCodeAndVehicleCodeAndTenantCode(@Param("vehicleTaskCode") String vehicleTaskCode,
                                                                       @Param("vehicleCode") String vehicleCode,
                                                                       @Param("tenantCode") String tenantCode);

  /**
   * 根据出车任务编号和商品类型和卸货单状态查询
   * @param vehicleTaskCode
   * @param vehicleUnloadStatus
   * @param tenantCode
   * @return
   */
  @Query("from VehicleUnload v where v.vehicleTaskCode=:vehicleTaskCode and v.vehicleProductType = :vehicleProductType and v.vehicleUnloadStatus = :vehicleUnloadStatus and v.tenantCode = :tenantCode")
  List<VehicleUnload> findByVehicleTaskCodeAndProductTypeAndUnloadStatusAndTenantCode(@Param("vehicleTaskCode") String vehicleTaskCode,
                                                                                      @Param("vehicleProductType") Integer vehicleProductType,
                                                                                      @Param("vehicleUnloadStatus") Integer vehicleUnloadStatus,
                                                                                      @Param("tenantCode") String tenantCode);

  /**
   * 根据出车任务编号和商品类型查询
   * @param vehicleTaskCode
   * @param tenantCode
   * @return
   */
  @Query("from VehicleUnload v where v.vehicleTaskCode=:vehicleTaskCode and v.vehicleProductType = :vehicleProductType and v.tenantCode = :tenantCode")
  List<VehicleUnload> findByVehicleTaskCodeAndProductTypeAndTenantCode(@Param("vehicleTaskCode") String vehicleTaskCode,
                                                                       @Param("vehicleProductType") Integer vehicleProductType,
                                                                       @Param("tenantCode") String tenantCode);

  /**
   * 根据出车任务编号和卸货单类型查询
   * @param vehicleTaskCode
   * @param vehicleUnloadType
   * @param tenantCode
   * @return
   */
  @Query("from VehicleUnload v where v.vehicleTaskCode=:vehicleTaskCode and v.vehicleUnloadType = :vehicleUnloadType and v.tenantCode = :tenantCode")
  List<VehicleUnload> findByVehicleTaskCodeAndUnloadTypeAndTenantCode(@Param("vehicleTaskCode")String vehicleTaskCode,
                                                                      @Param("vehicleUnloadType")Integer vehicleUnloadType,
                                                                      @Param("tenantCode")String tenantCode);
}