package com.bizunited.empower.business.vehicle.service.internal;

import com.bizunited.empower.business.common.util.SecurityUtils;
import com.bizunited.empower.business.vehicle.entity.VehicleLoad;
import com.bizunited.empower.business.vehicle.enums.VehicleLoadStatusEnum;
import com.bizunited.empower.business.vehicle.repository.VehicleLoadRepository;
import com.bizunited.empower.business.vehicle.service.VehicleLoadProductService;
import com.bizunited.empower.business.vehicle.service.VehicleLoadService;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Optional;

import static com.bizunited.empower.business.vehicle.constant.VehicleConstants.VEHICLE_LOAD_PREFIX;
import static com.bizunited.empower.business.vehicle.constant.VehicleRedisKey.VEHICLE_LOAD_CODE_AUTO_INC_KEY;

/**
 * VehicleLoad业务模型的服务层接口实现
 * @author saturn
 */
@Service("VehicleLoadServiceImpl")
public class VehicleLoadServiceImpl implements VehicleLoadService {
  @Autowired
  private VehicleLoadRepository vehicleLoadRepository;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private VehicleLoadProductService vehicleLoadProductService;
  @Transactional
  @Override
  public VehicleLoad create(VehicleLoad vehicleLoad) {
    VehicleLoad current = this.createForm(vehicleLoad);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public VehicleLoad createForm(VehicleLoad vehicleLoad) {
   /* 
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */
    Date now = new Date();
    vehicleLoad.setCreateAccount(SecurityUtils.getUserAccount());
    vehicleLoad.setCreateTime(now);
    vehicleLoad.setModifyAccount(SecurityUtils.getUserAccount());
    vehicleLoad.setModifyTime(now);
    vehicleLoad.setTenantCode(TenantUtils.getTenantCode());
    vehicleLoad.setVehicleLoadCode(this.generateCode(TenantUtils.getTenantCode()));
    //初始状态为待确认
    vehicleLoad.setVehicleLoadStatus(VehicleLoadStatusEnum.WAIT_CONFIRM.getType());

    this.createValidation(vehicleLoad);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.vehicleLoadRepository.save(vehicleLoad);
    
    // 返回最终处理的结果，里面带有详细的关联信息
    return vehicleLoad;
  }
  /**
   * 在创建一个新的VehicleLoad模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(VehicleLoad vehicleLoad) {
    Validate.notNull(vehicleLoad, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(vehicleLoad.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    vehicleLoad.setId(null);
    Validate.notBlank(vehicleLoad.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(vehicleLoad.getVehicleLoadCode(), "添加信息时，装货单编号不能为空！");
    Validate.notBlank(vehicleLoad.getVehicleTaskCode(), "添加信息时，出车任务编号不能为空！");
    Validate.notNull(vehicleLoad.getVehicleProductType(), "添加信息时，车载商品类型不能为空！");
    Validate.notBlank(vehicleLoad.getVehicleCode(), "添加信息时，车辆编码不能为空！");
    Validate.notBlank(vehicleLoad.getVehicleName(), "添加信息时，车辆名称不能为空！");
    Validate.notBlank(vehicleLoad.getSaleManAccount(), "添加信息时，业务员编号(账号)不能为空！");
    Validate.notBlank(vehicleLoad.getSaleManName(), "添加信息时，业务员名称不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(vehicleLoad.getExtend1() == null || vehicleLoad.getExtend1().length() < 255 , "扩展字段1,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getExtend2() == null || vehicleLoad.getExtend2().length() < 255 , "扩展字段2,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getExtend3() == null || vehicleLoad.getExtend3().length() < 255 , "扩展字段3,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getExtend4() == null || vehicleLoad.getExtend4().length() < 255 , "扩展字段4,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getExtend5() == null || vehicleLoad.getExtend5().length() < 255 , "扩展字段5,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getExtend6() == null || vehicleLoad.getExtend6().length() < 255 , "扩展字段6,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getExtend7() == null || vehicleLoad.getExtend7().length() < 255 , "扩展字段7,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getTenantCode() == null || vehicleLoad.getTenantCode().length() < 255 , "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getVehicleLoadCode() == null || vehicleLoad.getVehicleLoadCode().length() < 64 , "装货单编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleLoad.getVehicleTaskCode() == null || vehicleLoad.getVehicleTaskCode().length() < 64 , "出车任务编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleLoad.getVehicleCode() == null || vehicleLoad.getVehicleCode().length() < 64 , "车辆编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleLoad.getVehicleName() == null || vehicleLoad.getVehicleName().length() < 64 , "车辆名称,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleLoad.getSaleManAccount() == null || vehicleLoad.getSaleManAccount().length() < 255 , "业务员编号(账号),在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getSaleManName() == null || vehicleLoad.getSaleManName().length() < 255 , "业务员名称,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getCustomerCode() == null || vehicleLoad.getCustomerCode().length() < 64 , "客户编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleLoad.getCustomerName() == null || vehicleLoad.getCustomerName().length() < 64 , "客户名称,在进行添加时填入值超过了限定长度(64)，请检查!");
    VehicleLoad byVehicleLoadCode = this.findByVehicleLoadCode(vehicleLoad.getVehicleLoadCode());
    Validate.isTrue(byVehicleLoadCode == null, "装货单编号已存在,请检查");
  }
  @Transactional
  @Override
  public VehicleLoad update(VehicleLoad vehicleLoad) {
    VehicleLoad current = this.updateForm(vehicleLoad);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public VehicleLoad updateForm(VehicleLoad vehicleLoad) {
    /* 
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     * 
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    
    this.updateValidation(vehicleLoad);
    // ===================基本信息
    String currentId = vehicleLoad.getId();
    Optional<VehicleLoad> op_currentVehicleLoad = this.vehicleLoadRepository.findById(currentId);
    VehicleLoad currentVehicleLoad = op_currentVehicleLoad.orElse(null);
    currentVehicleLoad = Validate.notNull(currentVehicleLoad,"未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentVehicleLoad.setModifyAccount(SecurityUtils.getUserAccount());
    currentVehicleLoad.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentVehicleLoad.setExtend1(vehicleLoad.getExtend1());
    currentVehicleLoad.setExtend2(vehicleLoad.getExtend2());
    currentVehicleLoad.setExtend3(vehicleLoad.getExtend3());
    currentVehicleLoad.setExtend4(vehicleLoad.getExtend4());
    currentVehicleLoad.setExtend5(vehicleLoad.getExtend5());
    currentVehicleLoad.setExtend6(vehicleLoad.getExtend6());
    currentVehicleLoad.setExtend7(vehicleLoad.getExtend7());
    currentVehicleLoad.setExtend8(vehicleLoad.getExtend8());
    currentVehicleLoad.setExtend9(vehicleLoad.getExtend9());
    currentVehicleLoad.setExtend10(vehicleLoad.getExtend10());
    currentVehicleLoad.setExtend11(vehicleLoad.getExtend11());
    currentVehicleLoad.setTenantCode(vehicleLoad.getTenantCode());
    currentVehicleLoad.setVehicleLoadCode(vehicleLoad.getVehicleLoadCode());
    currentVehicleLoad.setVehicleTaskCode(vehicleLoad.getVehicleTaskCode());
    currentVehicleLoad.setVehicleProductType(vehicleLoad.getVehicleProductType());
    currentVehicleLoad.setVehicleCode(vehicleLoad.getVehicleCode());
    currentVehicleLoad.setVehicleName(vehicleLoad.getVehicleName());
    currentVehicleLoad.setSaleManAccount(vehicleLoad.getSaleManAccount());
    currentVehicleLoad.setSaleManName(vehicleLoad.getSaleManName());
    currentVehicleLoad.setCustomerCode(vehicleLoad.getCustomerCode());
    currentVehicleLoad.setCustomerName(vehicleLoad.getCustomerName());
    currentVehicleLoad.setVehicleLoadStatus(vehicleLoad.getVehicleLoadStatus());
    
    this.vehicleLoadRepository.saveAndFlush(currentVehicleLoad);
    return currentVehicleLoad;
  }
  /**
   * 在更新一个已有的VehicleLoad模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(VehicleLoad vehicleLoad) {
    Validate.isTrue(!StringUtils.isBlank(vehicleLoad.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(vehicleLoad.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(vehicleLoad.getVehicleLoadCode(), "修改信息时，装货单编号不能为空！");
    Validate.notBlank(vehicleLoad.getVehicleTaskCode(), "修改信息时，出车任务编号不能为空！");
    Validate.notNull(vehicleLoad.getVehicleProductType(), "修改信息时，车载商品类型不能为空！");
    Validate.notBlank(vehicleLoad.getVehicleCode(), "修改信息时，车辆编码不能为空！");
    Validate.notBlank(vehicleLoad.getVehicleName(), "修改信息时，车辆名称不能为空！");
    Validate.notBlank(vehicleLoad.getSaleManAccount(), "修改信息时，业务员编号(账号)不能为空！");
    Validate.notBlank(vehicleLoad.getSaleManName(), "修改信息时，业务员名称不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(vehicleLoad.getExtend1() == null || vehicleLoad.getExtend1().length() < 255 , "扩展字段1,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getExtend2() == null || vehicleLoad.getExtend2().length() < 255 , "扩展字段2,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getExtend3() == null || vehicleLoad.getExtend3().length() < 255 , "扩展字段3,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getExtend4() == null || vehicleLoad.getExtend4().length() < 255 , "扩展字段4,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getExtend5() == null || vehicleLoad.getExtend5().length() < 255 , "扩展字段5,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getExtend6() == null || vehicleLoad.getExtend6().length() < 255 , "扩展字段6,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getExtend7() == null || vehicleLoad.getExtend7().length() < 255 , "扩展字段7,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getTenantCode() == null || vehicleLoad.getTenantCode().length() < 255 , "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getVehicleLoadCode() == null || vehicleLoad.getVehicleLoadCode().length() < 64 , "装货单编号,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleLoad.getVehicleTaskCode() == null || vehicleLoad.getVehicleTaskCode().length() < 64 , "出车任务编号,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleLoad.getVehicleCode() == null || vehicleLoad.getVehicleCode().length() < 64 , "车辆编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleLoad.getVehicleName() == null || vehicleLoad.getVehicleName().length() < 64 , "车辆名称,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleLoad.getSaleManAccount() == null || vehicleLoad.getSaleManAccount().length() < 255 , "业务员编号(账号),在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getSaleManName() == null || vehicleLoad.getSaleManName().length() < 255 , "业务员名称,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleLoad.getCustomerCode() == null || vehicleLoad.getCustomerCode().length() < 64 , "客户编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleLoad.getCustomerName() == null || vehicleLoad.getCustomerName().length() < 64 , "客户名称,在进行修改时填入值超过了限定长度(64)，请检查!");
  }

  /**
   * 生成装货单编码
   */
  private String generateCode(String tenantCode) {
    String redisKey = String.format(VEHICLE_LOAD_CODE_AUTO_INC_KEY, tenantCode);
    String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
    SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
    return StringUtils.join(VEHICLE_LOAD_PREFIX,format.format(new Date()), index);
  }

  public VehicleLoad findByVehicleLoadCode(String vehicleLoadCode) {
    if(StringUtils.isBlank(vehicleLoadCode) || StringUtils.isBlank(TenantUtils.getTenantCode())) {
      return null;
    }
    return vehicleLoadRepository.findByVehicleLoadCodeAndTenantCode(vehicleLoadCode,TenantUtils.getTenantCode());
  }

  @Override
  public VehicleLoad findDetailsById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.vehicleLoadRepository.findDetailsById(id);
  }
  @Override
  public VehicleLoad findById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    Optional<VehicleLoad> op = vehicleLoadRepository.findById(id);
    return op.orElse(null); 
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    VehicleLoad current = this.findById(id);
    if(current != null) { 
      this.vehicleLoadRepository.delete(current);
    }
  }

  @Override
  public VehicleLoad saveVehicleLoad(VehicleLoad vehicleLoad) {
    vehicleLoad = create(vehicleLoad);
    vehicleLoadProductService.batchSave(vehicleLoad.getProducts());
    return vehicleLoad;
  }

  @Override
  public List<VehicleLoad> findByVehicleTaskCode(String vehicleTaskCode) {
    if (StringUtils.isEmpty(vehicleTaskCode)){
      return Lists.newArrayList();
    }
    return vehicleLoadRepository.findByVehicleTaskCodeAndTenantCode(vehicleTaskCode,TenantUtils.getTenantCode());
  }

  /**
   * 根据出车任务编码和车辆编码，查询退货入库信息
   * 注：只查询退货入库信息，且已确认状态的入库信息
   * @param vehicleTaskCode 任务编码
   * @param vehicleCode 车辆编码
   */
  @Override
  public List<VehicleLoad> findByVehicleTaskCodeAndVehicleCode(String vehicleTaskCode, String vehicleCode) {
    if (StringUtils.isAnyEmpty(vehicleTaskCode,vehicleCode,TenantUtils.getTenantCode())){
      return Lists.newArrayList();
    }
    return vehicleLoadRepository.findByVehicleTaskCodeAndVehicleCodeAndTenantCode(vehicleTaskCode,vehicleCode,TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public void updateStatusByVehicleLoadCode(String vehicleLoadCode,VehicleLoadStatusEnum vehicleLoadStatus) {
    Validate.notBlank(vehicleLoadCode , "根据装货单编码更改状态，装货单编码不能为空!!");
    VehicleLoad vehicleLoad = vehicleLoadRepository.findByVehicleLoadCodeAndTenantCode(vehicleLoadCode, TenantUtils.getTenantCode());
    vehicleLoad.setVehicleLoadStatus(vehicleLoadStatus.getType());
    vehicleLoad.setVehicleLoadTime(new Date());
    vehicleLoadRepository.save(vehicleLoad);
  }

  @Override
  public List<VehicleLoad> findByVehicleTaskCodeAndVehicleUnloadType(String vehicleTaskCode, Integer vehicleLoadType) {
    if (StringUtils.isBlank(vehicleTaskCode) || null == vehicleLoadType){
      return new ArrayList<>();
    }
    return vehicleLoadRepository.findByVehicleTaskCodeAndVehicleLoadTypeAndTenantCode(vehicleTaskCode,vehicleLoadType,TenantUtils.getTenantCode());
  }
} 
