package com.bizunited.empower.business.vehicle.service.internal;

import com.bizunited.empower.business.product.entity.ProductSpecification;
import com.bizunited.empower.business.product.entity.ProductUnitSpecificationAndPrice;
import com.bizunited.empower.business.product.service.ProductSpecificationService;
import com.bizunited.empower.business.vehicle.entity.Vehicle;
import com.bizunited.empower.business.vehicle.entity.VehicleProductStock;
import com.bizunited.empower.business.vehicle.repository.VehicleProductStockRepository;
import com.bizunited.empower.business.vehicle.service.VehicleProductStockService;
import com.bizunited.empower.business.vehicle.service.VehicleService;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import static com.bizunited.empower.business.vehicle.constant.VehicleRedisKey.LOCK_VEHICLE_STOCK_CODE_PREFIX;

/**
 * VehicleProductStock业务模型的服务层接口实现
 * @author saturn
 */
@Service("VehicleProductStockServiceImpl")
public class VehicleProductStockServiceImpl implements VehicleProductStockService {
  @Autowired
  private VehicleService vehicleService;
  @Autowired
  private VehicleProductStockRepository vehicleProductStockRepository;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private ProductSpecificationService productSpecificationService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Transactional
  @Override
  public VehicleProductStock create(VehicleProductStock vehicleProductStock) {
    VehicleProductStock current = this.createForm(vehicleProductStock);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public VehicleProductStock createForm(VehicleProductStock vehicleProductStock) {
   /* 
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */
    vehicleProductStock.setTenantCode(TenantUtils.getTenantCode());

    this.createValidation(vehicleProductStock);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.vehicleProductStockRepository.save(vehicleProductStock);
    
    // 返回最终处理的结果，里面带有详细的关联信息
    return vehicleProductStock;
  }
  /**
   * 在创建一个新的VehicleProductStock模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(VehicleProductStock vehicleProductStock) {
    Validate.notNull(vehicleProductStock , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(vehicleProductStock.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    vehicleProductStock.setId(null);
    Validate.notBlank(vehicleProductStock.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notNull(vehicleProductStock.getVehicleProductType(), "添加信息时，车载商品类型不能为空！");
    Validate.notBlank(vehicleProductStock.getProductCode(), "添加信息时，商品编码不能为空！");
    Validate.notBlank(vehicleProductStock.getProductName(), "添加信息时，商品名称不能为空！");
    Validate.notBlank(vehicleProductStock.getProductSpecificationCode(), "添加信息时，规格编码不能为空！");
    Validate.notBlank(vehicleProductStock.getProductSpecificationName(), "添加信息时，规格名称不能为空！");
    Validate.notBlank(vehicleProductStock.getRelativePath(), "添加信息时，相对路径不能为空！");
    Validate.notBlank(vehicleProductStock.getFileName(), "添加信息时，文件名不能为空！");
    Validate.notNull(vehicleProductStock.getInventory(), "添加信息时，库存量（后三位是小数的转换）不能为空！");
    Validate.notNull(vehicleProductStock.getPreemptInventory(), "添加信息时，预占库存量（后三位是小数的转换）不能为空！");
    Validate.notNull(vehicleProductStock.getUsableInventory(), "添加信息时，可用库存量（后三位是小数的转换）不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(vehicleProductStock.getTenantCode() == null || vehicleProductStock.getTenantCode().length() < 255 , "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleProductStock.getUnitCode() == null || vehicleProductStock.getUnitCode().length() < 64 , "单位编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleProductStock.getUnitName() == null || vehicleProductStock.getUnitName().length() < 128 , "单位名称,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(vehicleProductStock.getProductCode() == null || vehicleProductStock.getProductCode().length() < 64 , "商品编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleProductStock.getProductName() == null || vehicleProductStock.getProductName().length() < 64 , "商品名称,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleProductStock.getBrandCode() == null || vehicleProductStock.getBrandCode().length() < 64 , "品牌编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleProductStock.getBrandName() == null || vehicleProductStock.getBrandName().length() < 1024 , "品牌名称,在进行添加时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(vehicleProductStock.getCategoryName() == null || vehicleProductStock.getCategoryName().length() < 255 , "分类名称,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleProductStock.getCategoryCode() == null || vehicleProductStock.getCategoryCode().length() < 64 , "分类编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleProductStock.getProductSpecificationCode() == null || vehicleProductStock.getProductSpecificationCode().length() < 64 , "规格编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleProductStock.getProductSpecificationName() == null || vehicleProductStock.getProductSpecificationName().length() < 128 , "规格名称,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(vehicleProductStock.getRelativePath() == null || vehicleProductStock.getRelativePath().length() < 255 , "相对路径,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleProductStock.getFileName() == null || vehicleProductStock.getFileName().length() < 255 , "文件名,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleProductStock.getBarCode() == null || vehicleProductStock.getBarCode().length() < 255 , "条形码,在进行添加时填入值超过了限定长度(255)，请检查!");
  }
  @Transactional
  @Override
  public VehicleProductStock update(VehicleProductStock vehicleProductStock) {
    VehicleProductStock current = this.updateForm(vehicleProductStock);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public VehicleProductStock updateForm(VehicleProductStock vehicleProductStock) {
    /* 
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     * 
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    
    this.updateValidation(vehicleProductStock);
    // ===================基本信息
    String currentId = vehicleProductStock.getId();
    Optional<VehicleProductStock> op_currentVehicleProductStock = this.vehicleProductStockRepository.findById(currentId);
    VehicleProductStock currentVehicleProductStock = op_currentVehicleProductStock.orElse(null);
    currentVehicleProductStock = Validate.notNull(currentVehicleProductStock ,"未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    // 开始重新赋值——一般属性
    currentVehicleProductStock.setTenantCode(vehicleProductStock.getTenantCode());
    currentVehicleProductStock.setVehicleProductType(vehicleProductStock.getVehicleProductType());
    currentVehicleProductStock.setUnitCode(vehicleProductStock.getUnitCode());
    currentVehicleProductStock.setUnitName(vehicleProductStock.getUnitName());
    currentVehicleProductStock.setProductCode(vehicleProductStock.getProductCode());
    currentVehicleProductStock.setProductName(vehicleProductStock.getProductName());
    currentVehicleProductStock.setBrandCode(vehicleProductStock.getBrandCode());
    currentVehicleProductStock.setBrandName(vehicleProductStock.getBrandName());
    currentVehicleProductStock.setCategoryName(vehicleProductStock.getCategoryName());
    currentVehicleProductStock.setCategoryCode(vehicleProductStock.getCategoryCode());
    currentVehicleProductStock.setProductSpecificationCode(vehicleProductStock.getProductSpecificationCode());
    currentVehicleProductStock.setProductSpecificationName(vehicleProductStock.getProductSpecificationName());
    currentVehicleProductStock.setRelativePath(vehicleProductStock.getRelativePath());
    currentVehicleProductStock.setFileName(vehicleProductStock.getFileName());
    currentVehicleProductStock.setBarCode(vehicleProductStock.getBarCode());
    currentVehicleProductStock.setInventory(vehicleProductStock.getInventory());
    currentVehicleProductStock.setPreemptInventory(vehicleProductStock.getPreemptInventory());
    currentVehicleProductStock.setUsableInventory(vehicleProductStock.getUsableInventory());
    currentVehicleProductStock.setVehicle(vehicleProductStock.getVehicle());
    
    this.vehicleProductStockRepository.saveAndFlush(currentVehicleProductStock);
    return currentVehicleProductStock;
  }
  /**
   * 在更新一个已有的VehicleProductStock模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(VehicleProductStock vehicleProductStock) {
    Validate.isTrue(!StringUtils.isBlank(vehicleProductStock.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(vehicleProductStock.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notNull(vehicleProductStock.getVehicleProductType(), "修改信息时，车载商品类型不能为空！");
    Validate.notBlank(vehicleProductStock.getProductCode(), "修改信息时，商品编码不能为空！");
    Validate.notBlank(vehicleProductStock.getProductName(), "修改信息时，商品名称不能为空！");
    Validate.notBlank(vehicleProductStock.getProductSpecificationCode(), "修改信息时，规格编码不能为空！");
    Validate.notBlank(vehicleProductStock.getProductSpecificationName(), "修改信息时，规格名称不能为空！");
    Validate.notBlank(vehicleProductStock.getRelativePath(), "修改信息时，相对路径不能为空！");
    Validate.notBlank(vehicleProductStock.getFileName(), "修改信息时，文件名不能为空！");
    Validate.notNull(vehicleProductStock.getInventory(), "修改信息时，库存量（后三位是小数的转换）不能为空！");
    Validate.notNull(vehicleProductStock.getPreemptInventory(), "修改信息时，预占库存量（后三位是小数的转换）不能为空！");
    Validate.notNull(vehicleProductStock.getUsableInventory(), "修改信息时，可用库存量（后三位是小数的转换）不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(vehicleProductStock.getTenantCode() == null || vehicleProductStock.getTenantCode().length() < 255 , "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleProductStock.getUnitCode() == null || vehicleProductStock.getUnitCode().length() < 64 , "单位编号,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleProductStock.getUnitName() == null || vehicleProductStock.getUnitName().length() < 128 , "单位名称,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(vehicleProductStock.getProductCode() == null || vehicleProductStock.getProductCode().length() < 64 , "商品编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleProductStock.getProductName() == null || vehicleProductStock.getProductName().length() < 64 , "商品名称,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleProductStock.getBrandCode() == null || vehicleProductStock.getBrandCode().length() < 64 , "品牌编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleProductStock.getBrandName() == null || vehicleProductStock.getBrandName().length() < 1024 , "品牌名称,在进行修改时填入值超过了限定长度(1024)，请检查!");
    Validate.isTrue(vehicleProductStock.getCategoryName() == null || vehicleProductStock.getCategoryName().length() < 255 , "分类名称,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleProductStock.getCategoryCode() == null || vehicleProductStock.getCategoryCode().length() < 64 , "分类编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleProductStock.getProductSpecificationCode() == null || vehicleProductStock.getProductSpecificationCode().length() < 64 , "规格编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleProductStock.getProductSpecificationName() == null || vehicleProductStock.getProductSpecificationName().length() < 128 , "规格名称,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(vehicleProductStock.getRelativePath() == null || vehicleProductStock.getRelativePath().length() < 255 , "相对路径,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleProductStock.getFileName() == null || vehicleProductStock.getFileName().length() < 255 , "文件名,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleProductStock.getBarCode() == null || vehicleProductStock.getBarCode().length() < 255 , "条形码,在进行修改时填入值超过了限定长度(255)，请检查!");
    
    // 关联性判断，关联属性判断，需要满足ManyToOne或者OneToOne，且not null 且是主模型
  } 
  @Override
  public Set<VehicleProductStock> findDetailsByVehicle(String vehicle) {
    if(StringUtils.isBlank(vehicle)) { 
      return Sets.newHashSet();
    }
    return this.vehicleProductStockRepository.findDetailsByVehicle(vehicle);
  }

  @Override
  public List<VehicleProductStock> findByVehicleCodeAndProductTypeAndProductSpecificationCodes(String vehicleCode, Integer vehicleProductType, List<String> productSpecificationCodes) {
    if(StringUtils.isBlank(vehicleCode) || vehicleProductType == null || CollectionUtils.isEmpty(productSpecificationCodes)) {
      return Lists.newArrayList();
    }
    return this.vehicleProductStockRepository.findByTenantCodeAndVehicleCodeAndProductSpecificationCodes(TenantUtils.getTenantCode(),vehicleCode,productSpecificationCodes,vehicleProductType);
  }

  @Override
  public VehicleProductStock findByVehicleCodeAndProductTypeAndProductSpecificationCode(String vehicleCode, Integer vehicleProductType, String productSpecificationCode) {
    if(StringUtils.isAnyBlank(vehicleCode,productSpecificationCode) || vehicleProductType == null ) {
      return null;
    }
    return this.vehicleProductStockRepository.findByTenantCodeAndVehicleCodeAndProductSpecificationCode(TenantUtils.getTenantCode(),vehicleCode,productSpecificationCode,vehicleProductType);
  }

  @Override
  public VehicleProductStock findDetailsById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.vehicleProductStockRepository.findDetailsById(id);
  }
  @Override
  public VehicleProductStock findById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    Optional<VehicleProductStock> op = vehicleProductStockRepository.findById(id);
    return op.orElse(null); 
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    VehicleProductStock current = this.findById(id);
    if(current != null) { 
      this.vehicleProductStockRepository.delete(current);
    }
  }

  @Override
  @Transactional
  public VehicleProductStock preemption(String productSpecificationCode, String vehicleCode, BigDecimal quantity, String unitCode, Integer vehicleProductType) {
    //转换基本单位数量
    quantity = this.conversionUnit(productSpecificationCode, quantity, unitCode);
    String redisLockKey = String.format(LOCK_VEHICLE_STOCK_CODE_PREFIX, TenantUtils.getTenantCode(), vehicleCode);
    boolean isLocked = redisMutexService.tryLock(redisLockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "服务繁忙，请稍后再试！");
    try {
      //提货装车时会初始化商品
      VehicleProductStock vehicleProductStock = vehicleProductStockRepository
              .findByTenantCodeAndVehicleCodeAndProductSpecificationCode(TenantUtils.getTenantCode(), vehicleCode, productSpecificationCode,vehicleProductType);
      Vehicle vehicle = vehicleService.findByVehicleCode(vehicleCode);
      Validate.notNull(vehicle, vehicleCode+"车辆编码查询不到车辆");
      Validate.notNull(vehicleProductStock, "商品"+vehicleProductStock.getProductName()+"规格("+vehicleProductStock.getProductSpecificationName()+")在车辆"+vehicle.getCarNumber()+"无库存");
      //原有预占库存量
      BigDecimal preemptInventory = vehicleProductStock.getPreemptInventory();
      //可用库存量
      BigDecimal usableInventory = vehicleProductStock.getUsableInventory();
      //如果预占库存大于可用库存
      Validate.isTrue(usableInventory.compareTo(quantity)>=0,"商品("+vehicleProductStock.getProductName()+
              ")规格("+vehicleProductStock.getProductSpecificationName()+")可用库存不足,可用库存数量为"+usableInventory);
      vehicleProductStock.setPreemptInventory(preemptInventory.add(quantity));
      vehicleProductStock.setUsableInventory(usableInventory.subtract(quantity));
      vehicleProductStockRepository.save(vehicleProductStock);
      return vehicleProductStock;
    }finally {
      if (redisMutexService.islock(redisLockKey)){
        redisMutexService.unlock(redisLockKey);
      }
    }
  }

  @Override
  @Transactional
  public VehicleProductStock cancelPreemption(String productSpecificationCode, String vehicleCode, BigDecimal quantity, String unitCode, Integer vehicleProductType) {
    //转换基本单位数量
    quantity = this.conversionUnit(productSpecificationCode, quantity, unitCode);
    String redisLockKey = String.format(LOCK_VEHICLE_STOCK_CODE_PREFIX, TenantUtils.getTenantCode(), vehicleCode);
    boolean isLocked = redisMutexService.tryLock(redisLockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "服务繁忙，请稍后再试！");
    try {
      VehicleProductStock vehicleProductStock = vehicleProductStockRepository
              .findByTenantCodeAndVehicleCodeAndProductSpecificationCode(TenantUtils.getTenantCode(), vehicleCode, productSpecificationCode,vehicleProductType);
      // 预占库存量
      BigDecimal preemptInventory = vehicleProductStock.getPreemptInventory();
      //可用库存量
      BigDecimal usableInventory = vehicleProductStock.getUsableInventory();
      vehicleProductStock.setPreemptInventory(preemptInventory.subtract(quantity));
      vehicleProductStock.setUsableInventory(usableInventory.add(quantity));
      vehicleProductStockRepository.save(vehicleProductStock);
      return vehicleProductStock;
    }finally {
      if (redisMutexService.islock(redisLockKey)){
        redisMutexService.unlock(redisLockKey);
      }
    }
  }

  @Override
  @Transactional
  public void batchSave(String vehicleCode, List<VehicleProductStock> vehicleProductStockList) {
    Validate.notBlank(vehicleCode , "初始化车辆库存时，车辆编号不能为空!!");
    Validate.notNull(vehicleProductStockList , "初始化车辆库存时，商品集合不能为空!!");
    Vehicle vehicle = vehicleService.findByVehicleCode(vehicleCode);
    Validate.notNull(vehicle , "初始化车辆库存时，未查询到车辆信息!!");
    vehicleProductStockRepository.deleteByVehicleIdAndTenantCode(vehicle.getId(),TenantUtils.getTenantCode());
    for (VehicleProductStock vehicleProductStock : vehicleProductStockList) {
      vehicleProductStock.setVehicle(vehicle);
      vehicleProductStock.setTenantCode(vehicle.getTenantCode());
    }
    vehicleProductStockRepository.saveAll(vehicleProductStockList);
  }

  @Override
  @Transactional
  public void batchAdd(String vehicleCode, List<VehicleProductStock> vehicleProductStockList) {
    Validate.notBlank(vehicleCode , "添加车辆库存时，车辆编号不能为空!!");
    Validate.notEmpty(vehicleProductStockList , "添加车辆库存时，商品集合不能为空!!");
    Vehicle vehicle = vehicleService.findByVehicleCode(vehicleCode);
    Validate.notNull(vehicle , "添加车辆库存时，未查询到车辆信息!!");
    String redisLockKey = String.format(LOCK_VEHICLE_STOCK_CODE_PREFIX, TenantUtils.getTenantCode(), vehicleCode);
    boolean isLocked = redisMutexService.tryLock(redisLockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "服务繁忙，请稍后再试！");
    try {
      List<VehicleProductStock> vehicleProductStockSaveList = new ArrayList<>();
      List<VehicleProductStock> vehicleProductStocks = vehicleProductStockRepository.findByVehicleIdAndTenantCode(vehicle.getId(),TenantUtils.getTenantCode());
      //遍历要添加的商品
      for (VehicleProductStock productStock : vehicleProductStockList) {
        //判断要添加的商品是否在车仓库存中存在
        VehicleProductStock filterProductStock = vehicleProductStocks.stream()
                .filter(product -> product.getProductCode().equals(productStock.getProductCode())
                        && product.getProductSpecificationCode().equals(productStock.getProductSpecificationCode())
                        && product.getUnitCode().equals(productStock.getUnitCode())
                        && product.getVehicleProductType().equals(productStock.getVehicleProductType()))
                .findFirst().orElse(null);
        if (null == filterProductStock){
          //如果不存在直接加入车仓库商品集合中
          productStock.setVehicle(vehicle);
          productStock.setTenantCode(vehicle.getTenantCode());
          vehicleProductStockSaveList.add(productStock);
        }else {
          //累计数量
          filterProductStock.setInventory(filterProductStock.getInventory().add(productStock.getInventory()));
          filterProductStock.setUsableInventory(filterProductStock.getUsableInventory().add(productStock.getUsableInventory()));
          vehicleProductStockSaveList.add(filterProductStock);
        }
      }
      vehicleProductStockRepository.saveAll(vehicleProductStockSaveList);
    } finally {
      if (redisMutexService.islock(redisLockKey)){
        redisMutexService.unlock(redisLockKey);
      }
    }

  }

  @Override
  @Transactional
  public VehicleProductStock preemptionClose(String productSpecificationCode, String vehicleCode, BigDecimal quantity, String unitCode, Integer vehicleProductType) {
    //转换基本单位数量
    quantity = this.conversionUnit(productSpecificationCode, quantity, unitCode);
    String redisLockKey = String.format(LOCK_VEHICLE_STOCK_CODE_PREFIX, TenantUtils.getTenantCode(), vehicleCode);
    boolean isLocked = redisMutexService.tryLock(redisLockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "服务繁忙，请稍后再试！");
    try {
      VehicleProductStock vehicleProductStock = vehicleProductStockRepository
              .findByTenantCodeAndVehicleCodeAndProductSpecificationCode(TenantUtils.getTenantCode(), vehicleCode, productSpecificationCode,vehicleProductType);
      Vehicle vehicle = vehicleService.findByVehicleCode(vehicleCode);
      Validate.notNull(vehicle, "%s车辆编码查询不到车辆", vehicleCode);
      Validate.notNull(vehicleProductStock, "规格编号【%s】在车辆【%s】无库存", productSpecificationCode, vehicle.getCarNumber());
      //库存数量
      BigDecimal inventory = vehicleProductStock.getInventory();
      //预占库存量
      BigDecimal preemptInventory = vehicleProductStock.getPreemptInventory();
      Validate.isTrue(preemptInventory.compareTo(quantity)>=0,"商品 "+vehicleProductStock.getProductName()
              +" 规格 "+productSpecificationCode+" 车辆占用库存不足,占用库存数量为"+preemptInventory);
      vehicleProductStock.setPreemptInventory(preemptInventory.subtract(quantity));
      vehicleProductStock.setInventory(inventory.subtract(quantity));
      vehicleProductStockRepository.save(vehicleProductStock);
      return vehicleProductStock;
    }finally {
      if (redisMutexService.islock(redisLockKey)){
        redisMutexService.unlock(redisLockKey);
      }
    }
  }

  @Override
  public Boolean validateVehicleStock(String productSpecificationCode, String vehicleCode, BigDecimal quantity, String unitCode, Integer vehicleProductType) {
    //转换基本单位数量
    quantity = this.conversionUnit(productSpecificationCode, quantity, unitCode);
    VehicleProductStock vehicleProductStock = vehicleProductStockRepository
            .findByTenantCodeAndVehicleCodeAndProductSpecificationCode(TenantUtils.getTenantCode(), vehicleCode, productSpecificationCode,vehicleProductType);
    Validate.notNull(vehicleProductStock,"该商品未查询到库存");
    //可用库存量
    BigDecimal usableInventory = vehicleProductStock.getUsableInventory();
    //校验数量是否大于可用库存
    //新需求要求报错提示商品剩余数量
    Validate.isTrue(usableInventory.compareTo(quantity)>=0,"商品 "+vehicleProductStock.getProductName()
            +" 规格 "+productSpecificationCode+" 车辆库存不足,可用数量为"+usableInventory);
    return usableInventory.compareTo(quantity)>=0;
  }

  @Override
  @Transactional
  public BigDecimal conversionUnit(String productSpecificationCode, BigDecimal quantity, String unitCode) {
    Validate.notNull(productSpecificationCode, "出库商品规格必填！");
    Validate.notNull(quantity, "出库商品规格数量必填！");
    Validate.notNull(unitCode, "出库商品规格单位必填！");
    // 入参数量转基本单位数量
    ProductSpecification ps = productSpecificationService.findByProductSpecificationCode(productSpecificationCode);
    Validate.notNull(ps, "出库商品规格【%s】不存在", productSpecificationCode);
    Validate.notNull(ps.getProduct(), "出库商品规格【%s】,上层商品不存在", productSpecificationCode);
    Set<ProductUnitSpecificationAndPrice> productUnitAndPrices = ps.getProduct().getProductUnitSpecificationAndPrices();
    Validate.isTrue(CollectionUtils.isNotEmpty(productUnitAndPrices), "出库商品规格【%s】,上层商品未指定单位", productSpecificationCode);
    Boolean exist = false;
    for (ProductUnitSpecificationAndPrice unitAndPrice : productUnitAndPrices) {
      if (unitCode.equals(unitAndPrice.getProductUnit().getUnitCode()) && unitAndPrice.getProductSpecification().getProductSpecificationCode().equals(productSpecificationCode)) {
        exist = true;
        BigDecimal conversionRatio = unitAndPrice.getConversionRatio();
        Validate.notNull(conversionRatio, "商品规格【%s】单位未指定转换率", productSpecificationCode);
        quantity = quantity.multiply(conversionRatio).setScale(4 , RoundingMode.HALF_UP);
      }
    }
    Validate.isTrue(exist, "入参单位，在商品规格【%s】的上层商品中不存在", productSpecificationCode);
    String quantityStr = quantity.toString();
    Validate.isTrue(quantityStr.endsWith(".0000"), "出库数量转基本单位数量时有小数！");
    return quantity;
  }

  /**
   * 查询当前车辆的有效车载库存
   * @param vehicleCode 车辆编码
   */
  @Override
  public List<VehicleProductStock> findEffectiveVehicleSaleStockByVehicleCode(String vehicleCode) {
    if(StringUtils.isAnyBlank(vehicleCode,TenantUtils.getTenantCode())){
      return Lists.newArrayList();
    }
    return vehicleProductStockRepository.findEffectiveVehicleSaleStockByVehicleCodeAndTenantCode(vehicleCode,TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public void transforVehicleProductType(String vehicleCode, String productSpecificationCode, BigDecimal quantity,Integer oldType, Integer newType) {
    Validate.notBlank(vehicleCode,"车销商品类型转换时,车辆编号");
    Validate.notBlank(productSpecificationCode,"车销商品类型转换时,商品规格编号不能为空");
    Validate.notNull(oldType,"车销商品类型转换时,商品转换前的类型不能为空");
    Validate.notNull(newType,"车销商品类型转换时,商品转换后的类型不能为空");
    Validate.isTrue(quantity != null && quantity.compareTo(BigDecimal.ZERO) > 0,"转换数量必须大于0");
    Validate.isTrue(!oldType.equals(newType),"商品转换类型不能相同");
    String redisLockKey = String.format(LOCK_VEHICLE_STOCK_CODE_PREFIX, TenantUtils.getTenantCode(), vehicleCode);
    boolean isLocked = redisMutexService.tryLock(redisLockKey, TimeUnit.SECONDS, 5);
    Validate.isTrue(isLocked, "服务繁忙，请稍后再试！");
    try {
      VehicleProductStock oldTypeVehicleProductStock = vehicleProductStockRepository
              .findByTenantCodeAndVehicleCodeAndProductSpecificationCode(TenantUtils.getTenantCode(), vehicleCode, productSpecificationCode,oldType);
      Vehicle vehicle = vehicleService.findByVehicleCode(vehicleCode);
      Validate.notNull(vehicle, "%s车辆编码查询不到车辆", vehicleCode);
      //转换商品数量必须小于该类型商品的车辆库存的可用库存
      Validate.notNull(oldTypeVehicleProductStock, "规格编号【%s】在车辆【%s】无库存", productSpecificationCode, vehicle.getCarNumber());
      //查询需要转换类型的商品的 可用库存数量
      BigDecimal usableInventory = oldTypeVehicleProductStock.getUsableInventory();
      Validate.isTrue(usableInventory.compareTo(quantity)>=0,"商品 "+oldTypeVehicleProductStock.getProductName()
              +" 规格 "+productSpecificationCode+" 车辆可用库存不足,可用库存数量为"+usableInventory);
      //扣除转换类型前商品的库存
      oldTypeVehicleProductStock.setUsableInventory(oldTypeVehicleProductStock.getUsableInventory().subtract(quantity));
      oldTypeVehicleProductStock.setInventory(oldTypeVehicleProductStock.getInventory().subtract(quantity));
      //查询是否存在转换后类型的商品
      VehicleProductStock newTypeVehicleProductStock = vehicleProductStockRepository
              .findByTenantCodeAndVehicleCodeAndProductSpecificationCode(TenantUtils.getTenantCode(), vehicleCode, productSpecificationCode,newType);
      //如果不存在 初始化该商品
      if (null == newTypeVehicleProductStock){
        newTypeVehicleProductStock = nebulaToolkitService.copyObjectByWhiteList(oldTypeVehicleProductStock, VehicleProductStock.class, HashSet.class, ArrayList.class);
        newTypeVehicleProductStock.setId(null);
        newTypeVehicleProductStock.setVehicle(vehicle);
        newTypeVehicleProductStock.setVehicleProductType(newType);
        newTypeVehicleProductStock.setUsableInventory(BigDecimal.ZERO);
        newTypeVehicleProductStock.setInventory(BigDecimal.ZERO);
        newTypeVehicleProductStock.setPreemptInventory(BigDecimal.ZERO);
      }
      //为转换类型后商品赋值库存数据
      newTypeVehicleProductStock.setUsableInventory(newTypeVehicleProductStock.getUsableInventory().add(quantity));
      newTypeVehicleProductStock.setInventory(newTypeVehicleProductStock.getInventory().add(quantity));
      vehicleProductStockRepository.save(oldTypeVehicleProductStock);
      vehicleProductStockRepository.save(newTypeVehicleProductStock);
      //TODO 生成转换记录

    }finally {
      if (redisMutexService.islock(redisLockKey)){
        redisMutexService.unlock(redisLockKey);
      }
    }
  }
}
