package com.bizunited.empower.business.vehicle.service.internal;

import com.bizunited.empower.business.product.optimize.service.ProductFlatService;
import com.bizunited.empower.business.product.optimize.vo.ProductFlatVo;
import com.bizunited.empower.business.product.optimize.vo.ProductSpecificationFlatVo;
import com.bizunited.empower.business.product.optimize.vo.ProductUnitAndPriceFlatVo;
import com.bizunited.empower.business.vehicle.dto.VehicleProductStockDto;
import com.bizunited.empower.business.vehicle.entity.Vehicle;
import com.bizunited.empower.business.vehicle.entity.VehicleProductStock;
import com.bizunited.empower.business.vehicle.enums.VehicleProductTypeEnum;
import com.bizunited.empower.business.vehicle.repository.VehicleProductStockRepository;
import com.bizunited.empower.business.vehicle.repository.internal.VehicleProductStockRepositoryCustom;
import com.bizunited.empower.business.vehicle.service.VehicleProductStockService;
import com.bizunited.empower.business.vehicle.service.VehicleProductStockVoService;
import com.bizunited.empower.business.vehicle.service.VehicleService;
import com.bizunited.empower.business.vehicle.vo.VehicleProductStockVo;
import com.bizunited.empower.business.vehicle.vo.VehicleStockQuantityVo;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.bizunited.platform.common.constant.Constants.DEFAULT_PAGEABLE;

/**
 * @Author: wjx
 * @Date: 2021/4/1 14:44
 */
@Service("VehicleProductStockVoServiceImpl")
public class VehicleProductStockVoServiceImpl implements VehicleProductStockVoService {

  @Autowired
  private VehicleService vehicleService;
  @Autowired
  @Qualifier("_VehicleProductStockRepositoryImpl")
  private VehicleProductStockRepositoryCustom vehicleProductStockRepositoryCustom;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private ProductFlatService productFlatService;
  @Autowired
  private VehicleProductStockRepository vehicleProductStockRepository;
  @Autowired
  private VehicleProductStockService vehicleProductStockService;

  @Override
  public VehicleStockQuantityVo findStockByVehicleCode(String vehicleCode) {
    if (StringUtils.isBlank(vehicleCode)){
      return null;
    }
    VehicleStockQuantityVo vehicleStockQuantityVo = new VehicleStockQuantityVo();
    vehicleStockQuantityVo.setVehicleCode(vehicleCode);
    Vehicle vehicle = vehicleService.findByVehicleCode(vehicleCode);
    Set<VehicleProductStock> vehicleProductStocks = vehicle.getVehicleProductStocks();
    if (CollectionUtils.isEmpty(vehicleProductStocks)){
      return vehicleStockQuantityVo;
    }
    //根据商品规格去重 获得商品数量(筛选商品数量大于0的)
    Long productQuantity = vehicleProductStocks.stream()
            .filter(product -> product.getInventory().compareTo(BigDecimal.ZERO)==1)
            .map(VehicleProductStock::getProductSpecificationCode).distinct().count();
    vehicleStockQuantityVo.setProductQuantity(productQuantity);
    //统计所有商品库存数量
    BigDecimal stockProductTotal = vehicleProductStocks.stream().map(product -> product.getInventory())
            .reduce(BigDecimal.ZERO, BigDecimal::add);
    vehicleStockQuantityVo.setStockProductTotal(stockProductTotal);
    return vehicleStockQuantityVo;
  }

  @Override
  public Page<VehicleProductStockVo> findByConditions(Pageable pageable, VehicleProductStockDto dto) {
    if (null == dto || StringUtils.isEmpty(dto.getVehicleCode())){
      return new PageImpl<>(Lists.newArrayList(), pageable, 0);
    }
    Vehicle vehicle = vehicleService.findByVehicleCode(dto.getVehicleCode());
    Validate.notNull(vehicle,"查询车辆库存,未获取到车辆信息");
    String tenantCode = TenantUtils.getTenantCode();
    dto.setTenantCode(tenantCode);
    pageable = ObjectUtils.defaultIfNull(pageable, DEFAULT_PAGEABLE);
    Page<VehicleProductStock> byConditions = vehicleProductStockRepositoryCustom.findByConditions(pageable, dto,vehicle);
    Page<VehicleProductStockVo> vehicleReturnVoPage;
    // 分页信息需要自行转换
    List<VehicleProductStock> content = byConditions.getContent();
    if (!CollectionUtils.isEmpty(content)) {
      Collection<VehicleProductStockVo> vehicleProductStockVos = this.nebulaToolkitService
              .copyCollectionByWhiteList(content, VehicleProductStock.class, VehicleProductStockVo.class, LinkedHashSet.class, ArrayList.class);
      buildProductDetailVo(content,vehicleProductStockVos);
      vehicleReturnVoPage = new PageImpl<>(new ArrayList<>(vehicleProductStockVos), pageable, byConditions.getTotalElements());
    } else {
      vehicleReturnVoPage = new PageImpl<>(Lists.newArrayList(), pageable, 0);
    }
    return vehicleReturnVoPage;
  }

  @Override
  public Page<VehicleProductStockVo> findProductByConditions(Pageable pageable, VehicleProductStockDto dto) {
    if (null == dto || StringUtils.isEmpty(dto.getVehicleCode())){
      return new PageImpl<>(Lists.newArrayList(), pageable, 0);
    }
    Page<VehicleProductStockVo> vehicleReturnVoPage;
    Vehicle vehicle = vehicleService.findByVehicleCode(dto.getVehicleCode());
    Validate.notNull(vehicle,"查询车辆库存,未获取到车辆信息");
    String tenantCode = TenantUtils.getTenantCode();
    dto.setTenantCode(tenantCode);
    pageable = ObjectUtils.defaultIfNull(pageable, DEFAULT_PAGEABLE);
    //分页查询根据商品规格编号去重
    Page<String> byConditions = vehicleProductStockRepositoryCustom.findProductSpecificationCodeByConditions(pageable, dto,vehicle);
    //查询车辆中全部商品库存数据
    List<VehicleProductStock> allVehicleProductStock = vehicleProductStockRepository.findByVehicleIdAndTenantCode(vehicle.getId(), TenantUtils.getTenantCode());
    // 分页信息需要自行转换
    List<String> content = byConditions.getContent();
    if (!CollectionUtils.isEmpty(content)) {
      //最终返回的数据
      List<VehicleProductStockVo> vehicleProductStockVos = new ArrayList<>();
      //商品信息赋值vo
      Collection<VehicleProductStockVo> vehicleProductStockVoCollection = this.nebulaToolkitService
              .copyCollectionByWhiteList(allVehicleProductStock, VehicleProductStock.class, VehicleProductStockVo.class, LinkedHashSet.class, ArrayList.class);
      buildProductDetailVo(allVehicleProductStock,vehicleProductStockVoCollection);
      for (String productSpecificationCode : content) {
        //车仓中可能会有多个商品属于不同的类型 需要合并到同一个商品 展示不同类型的库存数量
        //全部商品根据商品规格编号分组
        Map<String, List<VehicleProductStockVo>> productMap = vehicleProductStockVoCollection.stream()
                .collect(Collectors.groupingBy(VehicleProductStockVo::getProductSpecificationCode));
        if (productMap.containsKey(productSpecificationCode)){
          List<VehicleProductStockVo> productStockVoList = productMap.get(productSpecificationCode);
          //取第一个商品数据
          VehicleProductStockVo vehicleProductStockVo = this.nebulaToolkitService
                  .copyObjectByWhiteList(productStockVoList.get(0), VehicleProductStockVo.class, LinkedHashSet.class, ArrayList.class);
          //该商品同时存在三种类型的库存
          vehicleProductStockVo.setVehicleProductType(null);
          vehicleProductStockVo.setUsableInventory(BigDecimal.ZERO);
          vehicleProductStockVo.setInventory(BigDecimal.ZERO);
          vehicleProductStockVo.setSalesUsableInventory(BigDecimal.ZERO);
          vehicleProductStockVo.setDistributionUsableInventory(BigDecimal.ZERO);
          vehicleProductStockVo.setNotSaleUsableInventory(BigDecimal.ZERO);
          for (VehicleProductStockVo productStockVo : productStockVoList) {
            vehicleProductStockVo.setUsableInventory(vehicleProductStockVo.getUsableInventory().add(productStockVo.getUsableInventory()));
            vehicleProductStockVo.setInventory(vehicleProductStockVo.getInventory().add(productStockVo.getInventory()));
            if (VehicleProductTypeEnum.VEHICLE_SALES.getType().equals(productStockVo.getVehicleProductType())){
              //如果是车销商品 赋值车销商品库存
              vehicleProductStockVo.setSalesUsableInventory(vehicleProductStockVo.getSalesUsableInventory().add(productStockVo.getUsableInventory()));
            } else if (VehicleProductTypeEnum.DISTRIBUTION.getType().equals(productStockVo.getVehicleProductType())){
              //如果是配送商品 赋值配送商品库存
              vehicleProductStockVo.setDistributionUsableInventory(vehicleProductStockVo.getDistributionUsableInventory().add(productStockVo.getUsableInventory()));
            } else if (VehicleProductTypeEnum.NOT_SALE.getType().equals(productStockVo.getVehicleProductType())){
              //如果是不可售商品 赋值不可售商品库存
              vehicleProductStockVo.setNotSaleUsableInventory(vehicleProductStockVo.getNotSaleUsableInventory().add(productStockVo.getUsableInventory()));
            }
          }
          vehicleProductStockVos.add(vehicleProductStockVo);
        }
      }
      vehicleReturnVoPage = new PageImpl<>(vehicleProductStockVos, pageable, byConditions.getTotalElements());
    } else {
      vehicleReturnVoPage = new PageImpl<>(Lists.newArrayList(), pageable, 0);
    }
    return vehicleReturnVoPage;
  }

  private void buildProductDetailVo(List<VehicleProductStock> content, Collection<VehicleProductStockVo> vehicleProductStockVos){
    List<String> productCodes = content.stream().map(VehicleProductStock::getProductCode).distinct().collect(Collectors.toList());
    //查询商品的详情
    List<ProductFlatVo> productSet = productFlatService.findByProductCodeList(productCodes,TenantUtils.getTenantCode());
    Map<String, ProductFlatVo> map = productSet.stream().collect(Collectors.toMap(ProductFlatVo::getProductCode, Function.identity()));
    for (VehicleProductStockVo vehicleProductStockVo : vehicleProductStockVos) {
      if (map.containsKey(vehicleProductStockVo.getProductCode())){
        ProductFlatVo product = map.get(vehicleProductStockVo.getProductCode());

        Map<String, ProductUnitAndPriceFlatVo> unitAndPriceMap = product.getProductUnitAndPriceMap();
        //单位及参考采购价
        ProductUnitAndPriceFlatVo flatDto = unitAndPriceMap.get(vehicleProductStockVo.getUnitCode());
        if (flatDto != null) {
          vehicleProductStockVo.setPurchasePrice(flatDto.getReferencePurchasePrice());
          vehicleProductStockVo.setSubtotalAmount(vehicleProductStockVo.getPurchasePrice().multiply(vehicleProductStockVo.getInventory()).setScale(4, BigDecimal.ROUND_HALF_UP));
          break;
        }
        //规格及主图
        Map<String, ProductSpecificationFlatVo> specificationMap = product.getProductSpecificationMap();
        ProductSpecificationFlatVo specificationFlatDto = specificationMap.get(vehicleProductStockVo.getProductSpecificationCode());
        if (specificationFlatDto != null) {
          vehicleProductStockVo.setRelativePath(specificationFlatDto.getMainImagePath());
          vehicleProductStockVo.setFileName(specificationFlatDto.getMainImageName());
          break;
        }
      }
    }
  }
}
