package com.bizunited.empower.business.vehicle.service.internal;

import com.bizunited.empower.business.common.util.SecurityUtils;
import com.bizunited.empower.business.vehicle.entity.Vehicle;
import com.bizunited.empower.business.vehicle.enums.VehicleStatusEnum;
import com.bizunited.empower.business.vehicle.enums.VehicleTransportStatusEnum;
import com.bizunited.empower.business.vehicle.repository.VehicleProductStockRepository;
import com.bizunited.empower.business.vehicle.repository.VehicleRepository;
import com.bizunited.empower.business.vehicle.repository.internal.VehicleProductStockRepositoryCustom;
import com.bizunited.empower.business.vehicle.service.VehicleService;
import com.bizunited.empower.business.vehicle.service.notify.VehicleListener;
import com.bizunited.empower.business.vehicle.vo.VehicleProductStockVo;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static com.bizunited.empower.business.vehicle.constant.VehicleConstants.VEHICLE_CODE_PREFIX;
import static com.bizunited.empower.business.vehicle.constant.VehicleRedisKey.VEHICLE_CODE_AUTO_INC_KEY;
import static com.bizunited.platform.common.constant.Constants.DEFAULT_PAGEABLE;

/**
 * Vehicle业务模型的服务层接口实现
 * @author saturn
 */
@Service("VehicleServiceImpl")
public class VehicleServiceImpl implements VehicleService { 
  @Autowired
  private VehicleRepository vehicleRepository;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private VehicleProductStockRepository vehicleProductStockRepository;
  @Autowired
  @Qualifier("_VehicleProductStockRepositoryImpl")
  private VehicleProductStockRepositoryCustom vehicleProductStockRepositoryCustom;
  @Autowired(required = false)
  private List<VehicleListener> vehicleListenerList;
  
  @Transactional
  @Override
  public Vehicle create(Vehicle vehicle) { 
    Vehicle current = this.createForm(vehicle);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public Vehicle createForm(Vehicle vehicle) { 
   /* 
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */
    Date now = new Date();
    vehicle.setCreateAccount(SecurityUtils.getUserAccount());
    vehicle.setCreateTime(now);
    vehicle.setModifyAccount(SecurityUtils.getUserAccount());
    vehicle.setModifyTime(now);
    vehicle.setTenantCode(TenantUtils.getTenantCode());
    vehicle.setVehicleCode(this.generateCode(TenantUtils.getTenantCode()));
    //车辆名称采用车牌号
    vehicle.setVehicleName(vehicle.getCarNumber());
    //默认启用
    vehicle.setVehicleStatus(VehicleStatusEnum.ENABLE.getType());
    //默认空置
    vehicle.setVehicleTransportStatus(VehicleTransportStatusEnum.VACANT.getType());
    this.createValidation(vehicle);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.vehicleRepository.save(vehicle);
    // 返回最终处理的结果，里面带有详细的关联信息
    return vehicle;
  }
  /**
   * 在创建一个新的Vehicle模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(Vehicle vehicle) { 
    Validate.notNull(vehicle , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(vehicle.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    vehicle.setId(null);
    Validate.notBlank(vehicle.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(vehicle.getVehicleCode(), "添加信息时，车辆编码不能为空！");
    Validate.notBlank(vehicle.getVehicleName(), "添加信息时，车辆名称不能为空！");
    Validate.notBlank(vehicle.getCarNumber(), "添加信息时，车牌号不能为空！");
    Validate.notNull(vehicle.getVehicleStatus(), "添加信息时，车辆状态不能为空！");
    Validate.notNull(vehicle.getVehicleType(), "添加信息时，车辆类型不能为空！");
    Validate.notBlank(vehicle.getPowerType(), "添加信息时，动力类型不能为空！");
    Validate.notNull(vehicle.getColdChain(), "添加信息时，是否冷链不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(vehicle.getExtend1() == null || vehicle.getExtend1().length() < 255 , "扩展字段1,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getExtend2() == null || vehicle.getExtend2().length() < 255 , "扩展字段2,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getExtend3() == null || vehicle.getExtend3().length() < 255 , "扩展字段3,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getExtend4() == null || vehicle.getExtend4().length() < 255 , "扩展字段4,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getExtend5() == null || vehicle.getExtend5().length() < 255 , "扩展字段5,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getExtend6() == null || vehicle.getExtend6().length() < 255 , "扩展字段6,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getExtend7() == null || vehicle.getExtend7().length() < 255 , "扩展字段7,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getTenantCode() == null || vehicle.getTenantCode().length() < 255 , "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getVehicleCode() == null || vehicle.getVehicleCode().length() < 64 , "车辆编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicle.getVehicleName() == null || vehicle.getVehicleName().length() < 255 , "车辆名称,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getCarNumber() == null || vehicle.getCarNumber().length() < 9 , "车牌号,在进行添加时填入值超过了限定长度(8)，请检查!");
    Validate.isTrue(vehicle.getPowerType() == null || vehicle.getPowerType().length() < 64 , "动力类型,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicle.getAxesQuantity() == null || vehicle.getAxesQuantity().length() < 64 , "轴数,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicle.getEmissionStandard() == null || vehicle.getEmissionStandard().length() < 64 , "排放标准,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicle.getLicenseColor() == null || vehicle.getLicenseColor().length() < 64 , "车牌颜色,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicle.getPurpose() == null || vehicle.getPurpose().length() < 64 , "货车用途,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicle.getRemark() == null || vehicle.getRemark().length() < 255 , "备注,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getWeight()== null || (vehicle.getWeight() <= 100 && vehicle.getWeight()>0), "总重量范围0-100吨 请检查!");
    Validate.isTrue(vehicle.getApprovedWeight() == null || (vehicle.getApprovedWeight() <= 100 && vehicle.getApprovedWeight()>0), "核定重量范围0-100吨 请检查!");
    Validate.isTrue(vehicle.getLength() == null || (vehicle.getLength() <= 25 && vehicle.getLength()>0), "车长最长为25米 请检查!");
    Validate.isTrue(vehicle.getWide() == null || (vehicle.getWide() <= 5 && vehicle.getWide()>0), "车宽最宽为5米 请检查!");
    Validate.isTrue(vehicle.getHigh() == null || (vehicle.getHigh() <= 10 && vehicle.getHigh()>0), "车高最高为10米 请检查!");
    Vehicle byVehicleCode = this.findByVehicleCode(vehicle.getVehicleCode());
    Validate.isTrue(byVehicleCode == null, "车辆编号已存在,请检查");
    Vehicle byCarNumber = this.findByCarNumber(vehicle.getCarNumber());
    Validate.isTrue(byCarNumber == null, "车牌号已存在,请检查");
  }
  @Transactional
  @Override
  public Vehicle update(Vehicle vehicle) { 
    Vehicle current = this.updateForm(vehicle);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public Vehicle updateForm(Vehicle vehicle) { 
    /* 
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     * 
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    
    this.updateValidation(vehicle);
    // ===================基本信息
    String currentId = vehicle.getId();
    Optional<Vehicle> op_currentVehicle = this.vehicleRepository.findById(currentId);
    Vehicle currentVehicle = op_currentVehicle.orElse(null);
    currentVehicle = Validate.notNull(currentVehicle ,"未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentVehicle.setModifyAccount(SecurityUtils.getUserAccount());
    currentVehicle.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentVehicle.setExtend1(vehicle.getExtend1());
    currentVehicle.setExtend2(vehicle.getExtend2());
    currentVehicle.setExtend3(vehicle.getExtend3());
    currentVehicle.setExtend4(vehicle.getExtend4());
    currentVehicle.setExtend5(vehicle.getExtend5());
    currentVehicle.setExtend6(vehicle.getExtend6());
    currentVehicle.setExtend7(vehicle.getExtend7());
    currentVehicle.setExtend8(vehicle.getExtend8());
    currentVehicle.setExtend9(vehicle.getExtend9());
    currentVehicle.setExtend10(vehicle.getExtend10());
    currentVehicle.setExtend11(vehicle.getExtend11());
    currentVehicle.setTenantCode(vehicle.getTenantCode());
    currentVehicle.setVehicleCode(vehicle.getVehicleCode());
    currentVehicle.setVehicleName(vehicle.getVehicleName());
    currentVehicle.setCarNumber(vehicle.getCarNumber());
    currentVehicle.setVehicleStatus(vehicle.getVehicleStatus());
    currentVehicle.setVehicleType(vehicle.getVehicleType());
    currentVehicle.setPowerType(vehicle.getPowerType());
    currentVehicle.setColdChain(vehicle.getColdChain());
    currentVehicle.setWeight(vehicle.getWeight());
    currentVehicle.setApprovedWeight(vehicle.getApprovedWeight());
    currentVehicle.setLength(vehicle.getLength());
    currentVehicle.setWide(vehicle.getWide());
    currentVehicle.setHigh(vehicle.getHigh());
    currentVehicle.setAxesQuantity(vehicle.getAxesQuantity());
    currentVehicle.setEmissionStandard(vehicle.getEmissionStandard());
    currentVehicle.setLicenseColor(vehicle.getLicenseColor());
    currentVehicle.setPurpose(vehicle.getPurpose());
    currentVehicle.setRemark(vehicle.getRemark());
    
    this.vehicleRepository.saveAndFlush(currentVehicle);
    return currentVehicle;
  }
  /**
   * 在更新一个已有的Vehicle模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(Vehicle vehicle) { 
    Validate.isTrue(!StringUtils.isBlank(vehicle.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(vehicle.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(vehicle.getVehicleCode(), "修改信息时，车辆编码不能为空！");
    Validate.notBlank(vehicle.getVehicleName(), "修改信息时，车辆名称不能为空！");
    Validate.notBlank(vehicle.getCarNumber(), "修改信息时，车牌号不能为空！");
    Validate.notNull(vehicle.getVehicleStatus(), "修改信息时，车辆状态不能为空！");
    Validate.notNull(vehicle.getVehicleType(), "修改信息时，车辆类型不能为空！");
    Validate.notBlank(vehicle.getPowerType(), "修改信息时，动力类型不能为空！");
    Validate.notNull(vehicle.getColdChain(), "修改信息时，是否冷链不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(vehicle.getExtend1() == null || vehicle.getExtend1().length() < 255 , "扩展字段1,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getExtend2() == null || vehicle.getExtend2().length() < 255 , "扩展字段2,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getExtend3() == null || vehicle.getExtend3().length() < 255 , "扩展字段3,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getExtend4() == null || vehicle.getExtend4().length() < 255 , "扩展字段4,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getExtend5() == null || vehicle.getExtend5().length() < 255 , "扩展字段5,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getExtend6() == null || vehicle.getExtend6().length() < 255 , "扩展字段6,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getExtend7() == null || vehicle.getExtend7().length() < 255 , "扩展字段7,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getTenantCode() == null || vehicle.getTenantCode().length() < 255 , "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getVehicleCode() == null || vehicle.getVehicleCode().length() < 64 , "车辆编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicle.getVehicleName() == null || vehicle.getVehicleName().length() < 255 , "车辆名称,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicle.getCarNumber() == null || vehicle.getCarNumber().length() < 64 , "车牌号,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicle.getPowerType() == null || vehicle.getPowerType().length() < 64 , "动力类型,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicle.getAxesQuantity() == null || vehicle.getAxesQuantity().length() < 64 , "轴数,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicle.getEmissionStandard() == null || vehicle.getEmissionStandard().length() < 64 , "排放标准,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicle.getLicenseColor() == null || vehicle.getLicenseColor().length() < 64 , "车牌颜色,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicle.getPurpose() == null || vehicle.getPurpose().length() < 64 , "货车用途,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicle.getRemark() == null || vehicle.getRemark().length() < 255 , "备注,在进行修改时填入值超过了限定长度(255)，请检查!");
  }


  /**
   * 生成车辆编码
   */
  private String generateCode(String tenantCode) {
    String redisKey = String.format(VEHICLE_CODE_AUTO_INC_KEY, tenantCode);
    String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
    return StringUtils.join(VEHICLE_CODE_PREFIX, index);
  }

  @Override
  public Vehicle findDetailsById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.vehicleRepository.findDetailsById(id);
  }
  @Override
  public Vehicle findById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    Optional<Vehicle> op = vehicleRepository.findById(id);
    return op.orElse(null); 
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    Vehicle current = this.findById(id);
    if(current != null) { 
      this.vehicleRepository.delete(current);
    }
  }

  @Override
  public Vehicle findByVehicleCode(String vehicleCode) {
    if(StringUtils.isBlank(vehicleCode) || StringUtils.isBlank(TenantUtils.getTenantCode())) {
      return null;
    }
    return vehicleRepository.findByVehicleCodeAndTenantCode(vehicleCode,TenantUtils.getTenantCode());
  }

  /**
   * 根据车牌号查询
   * @param carNumber
   * @return
   */
  private Vehicle findByCarNumber(String carNumber) {
    if(StringUtils.isBlank(carNumber)) {
      return null;
    }
    return vehicleRepository.findByCarNumberAndTenantCode(carNumber,TenantUtils.getTenantCode());
  }

  /**
   * 根据编号更改车辆状态
   * @param vehicleCode
   * @param vehicleStatus
   */
  @Override
  @Transactional
  public void updateVehicleStatus(String vehicleCode, Integer vehicleStatus) {
    Validate.notBlank(vehicleCode , "根据编号更改车辆状态，必须给定编号信息!!");
    Validate.notNull(vehicleStatus , "根据编号更改车辆状态，必须给定状态信息!!");
    //启用或禁用车辆，车辆使用状态不能为占用
    Vehicle vehicle = vehicleRepository.findByVehicleCodeAndTenantCode(vehicleCode, TenantUtils.getTenantCode());
    Validate.notNull(vehicleStatus , "根据编号更改车辆状态，未查询到车辆信息!");
    Validate.isTrue(!VehicleTransportStatusEnum.OCCUPY.getType().equals(vehicle.getVehicleTransportStatus()),
            "禁用车辆时，车辆状态不能为占用");
    //通知出车任务 和路线 查询是否有关联该车辆
    if(!CollectionUtils.isEmpty(vehicleListenerList)){
      for(VehicleListener listener : vehicleListenerList){
        listener.onDisable(vehicle);
      }
    }
    vehicleRepository.updateVehicleStatusByVehicleCodeAndTenantCode(vehicleCode,TenantUtils.getTenantCode(),vehicleStatus);
  }

  /**
   * 查询租户下全部已启用车辆
   * @return
   */
  @Override
  public List<Vehicle> findAll() {
    return vehicleRepository.findAllByTenantCodeAndStatus(TenantUtils.getTenantCode(),VehicleStatusEnum.ENABLE.getType());
  }

  /**
   * 分页查询
   * @param pageable
   * @param conditions
   * @return
   */
  @Override
  public Page<Vehicle> findByConditions(Pageable pageable, Map<String, Object> conditions) {
    String tenantCode = TenantUtils.getTenantCode();
    conditions.put("tenantCode", tenantCode);
    pageable = ObjectUtils.defaultIfNull(pageable, DEFAULT_PAGEABLE);
    return vehicleRepository.findByConditions(pageable, conditions);
  }

  /**
   * 根据编号更改车辆使用状态
   * @param vehicleCode
   * @param vehicleTransportStatus
   */
  @Override
  public void updateVehicleTransportStatus(String vehicleCode, Integer vehicleTransportStatus) {
    Validate.notBlank(vehicleCode , "根据编号更改车辆使用状态，必须给定编号信息!!");
    Validate.notNull(vehicleTransportStatus , "根据编号更改车辆使用状态，必须给定状态信息!!");
    Vehicle vehicle = vehicleRepository.findByVehicleCodeAndTenantCode(vehicleCode, TenantUtils.getTenantCode());
    Validate.notNull(vehicle , "根据编号更改车辆使用状态，未查询到车辆!");
    //车辆状态非空置
    Validate.isTrue(VehicleTransportStatusEnum.VACANT.getType().equals(vehicle.getVehicleTransportStatus()),
            "不可禁用正在执行出车任务或已关联出车计划的的车辆!");
    vehicleRepository.updateVehicleTransportStatus(vehicleCode,TenantUtils.getTenantCode(),vehicleTransportStatus);
  }

  /**
   * 收车退货对车辆的操作
   * @param vehicleCode
   */
  @Override
  public void returnVehicleByVehicleCode(String vehicleCode) {
    Validate.notBlank(vehicleCode , "收车退货更改车辆时，必须给定编号信息!!");
    Vehicle vehicle = vehicleRepository.findByVehicleCodeAndTenantCode(vehicleCode, TenantUtils.getTenantCode());
    Validate.notNull(vehicle , "收车退货更改车辆时，未查询到车辆!!");
    vehicleRepository.updateVehicleTransportStatus(vehicleCode,TenantUtils.getTenantCode(),VehicleTransportStatusEnum.VACANT.getType());
    vehicleProductStockRepository.deleteByVehicleIdAndTenantCode(vehicle.getId(),TenantUtils.getTenantCode());
  }

  @Override
  public List<Vehicle> findByUserAccount(String userAccount) {
    if (StringUtils.isEmpty(userAccount)){
      return null;
    }
    return vehicleRepository.findByUserAccount(userAccount,TenantUtils.getTenantCode());
  }

  @Override
  public List<VehicleProductStockVo> findProductStockStatistics(String vehicleId, String tenantCode) {
    if (StringUtils.isEmpty(vehicleId)||StringUtils.isEmpty(tenantCode)){
      return Lists.newArrayList();
    }
    return vehicleProductStockRepositoryCustom.findProductStockStatistics(vehicleId,tenantCode);
  }

  @Override
  public List<Vehicle> findByVehicleCodes(List<String> vehicleCodeList) {
    if (CollectionUtils.isEmpty(vehicleCodeList)){
      return Lists.newArrayList();
    }
    return vehicleRepository.findByVehicleCodesAndTenantCode(vehicleCodeList,TenantUtils.getTenantCode());
  }

  @Override
  public void deleteVehicle(String vehicleCode) {
    Validate.notBlank(vehicleCode , "删除车辆时，必须给定编号信息!!");
    Vehicle vehicle = vehicleRepository.findByVehicleCodeAndTenantCode(vehicleCode, TenantUtils.getTenantCode());
    Validate.notNull(vehicle , "删除车辆时，未查询到车辆!!");
    //车辆状态非空置
    Validate.isTrue(!VehicleTransportStatusEnum.OCCUPY.getType().equals(vehicle.getVehicleTransportStatus()),
            "不可删除正在执行出车任务或已关联出车计划的的车辆!");
    vehicle.setDelete(true);
    vehicleRepository.save(vehicle);
    //通知出车任务 和路线 查询是否有关联该车辆
    if(!CollectionUtils.isEmpty(vehicleListenerList)){
      for(VehicleListener listener : vehicleListenerList){
        listener.onDelete(vehicle);
      }
    }
  }

  @Override
  public Vehicle findByVehicleCodeAndTenantCode(String vehicleCode, String tenantCode) {
    if (StringUtils.isBlank(vehicleCode)||StringUtils.isBlank(tenantCode)){
      return null;
    }
    return vehicleRepository.findByVehicleCodeAndTenantCode(vehicleCode,tenantCode);
  }
} 
