package com.bizunited.empower.business.vehicle.service.internal;

import com.bizunited.empower.business.common.util.SecurityUtils;
import com.bizunited.empower.business.vehicle.entity.VehicleUnload;
import com.bizunited.empower.business.vehicle.entity.VehicleUnloadProduct;
import com.bizunited.empower.business.vehicle.enums.VehicleProductTypeEnum;
import com.bizunited.empower.business.vehicle.enums.VehicleUnloadStatusEnum;
import com.bizunited.empower.business.vehicle.repository.VehicleUnloadRepository;
import com.bizunited.empower.business.vehicle.service.VehicleProductStockService;
import com.bizunited.empower.business.vehicle.service.VehicleUnloadProductService;
import com.bizunited.empower.business.vehicle.service.VehicleUnloadService;
import com.bizunited.empower.business.vehicle.service.notify.VehicleUnloadListener;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import static com.bizunited.empower.business.vehicle.constant.VehicleConstants.VEHICLE_UNLOAD_CODE_PREFIX;
import static com.bizunited.empower.business.vehicle.constant.VehicleRedisKey.VEHICLE_UNLOAD_CODE_AUTO_INC_KEY;

/**
 * VehicleUnload业务模型的服务层接口实现
 * @author saturn
 */
@Service("VehicleUnloadServiceImpl")
public class VehicleUnloadServiceImpl implements VehicleUnloadService {
  @Autowired
  private VehicleUnloadRepository vehicleUnloadRepository;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private VehicleProductStockService vehicleProductStockService;
  @Autowired
  private VehicleUnloadProductService vehicleUnloadProdctService;
  @Autowired(required = false)
  private List<VehicleUnloadListener> vehicleUnloadListenerList;

  @Transactional
  @Override
  public VehicleUnload create(VehicleUnload vehicleUnload) {
    VehicleUnload current = this.createForm(vehicleUnload);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public VehicleUnload createForm(VehicleUnload vehicleUnload) {
   /* 
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */
    Date now = new Date();
    vehicleUnload.setCreateAccount(SecurityUtils.getUserAccount());
    vehicleUnload.setCreateTime(now);
    vehicleUnload.setModifyAccount(SecurityUtils.getUserAccount());
    vehicleUnload.setModifyTime(now);
    vehicleUnload.setTenantCode(TenantUtils.getTenantCode());
    vehicleUnload.setVehicleUnloadCode(this.generateCode(TenantUtils.getTenantCode()));
    if (VehicleUnloadStatusEnum.WAIT_DELIVERY.getType().equals(vehicleUnload.getVehicleUnloadStatus())){
      //生成卸货单占用库存
      for (VehicleUnloadProduct product : vehicleUnload.getProducts()) {
        //根据商品占用库存
        vehicleProductStockService.preemption(product.getProductSpecificationCode(),vehicleUnload.getVehicleCode(),
                product.getQuantity(),product.getUnitCode(),vehicleUnload.getVehicleProductType());
      }
    }
    //收车退货直接生成已完成的出货单
    this.createValidation(vehicleUnload);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.vehicleUnloadRepository.save(vehicleUnload);
    
    // 返回最终处理的结果，里面带有详细的关联信息
    return vehicleUnload;
  }
  /**
   * 在创建一个新的VehicleUnload模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(VehicleUnload vehicleUnload) {
    Validate.notNull(vehicleUnload, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(vehicleUnload.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    vehicleUnload.setId(null);
    Validate.notBlank(vehicleUnload.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(vehicleUnload.getVehicleUnloadCode(), "添加信息时，卸货单编号不能为空！");
    Validate.notNull(vehicleUnload.getVehicleUnloadType(), "添加信息时，出货类型 1运输途中出货 2收车入库出货不能为空！");
    Validate.notNull(vehicleUnload.getVehicleProductType(), "添加信息时，车载商品类型不能为空！");
    Validate.notNull(vehicleUnload.getVehicleUnloadStatus(), "添加信息时，出货(出库)状态 1待确认 2已取消 3已确认 4已作废不能为空！");
    Validate.notBlank(vehicleUnload.getVehicleTaskCode(), "添加信息时，出车任务编号不能为空！");
    Validate.notBlank(vehicleUnload.getVehicleCode(), "添加信息时，车辆编码不能为空！");
    Validate.notBlank(vehicleUnload.getVehicleName(), "添加信息时，车辆名称不能为空！");
    Validate.notBlank(vehicleUnload.getSaleManAccount(), "添加信息时，业务员编号(账号)不能为空！");
    Validate.notBlank(vehicleUnload.getSaleManName(), "添加信息时，业务员名称不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(vehicleUnload.getExtend1() == null || vehicleUnload.getExtend1().length() < 255 , "扩展字段1,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getExtend2() == null || vehicleUnload.getExtend2().length() < 255 , "扩展字段2,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getExtend3() == null || vehicleUnload.getExtend3().length() < 255 , "扩展字段3,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getExtend4() == null || vehicleUnload.getExtend4().length() < 255 , "扩展字段4,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getExtend5() == null || vehicleUnload.getExtend5().length() < 255 , "扩展字段5,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getExtend6() == null || vehicleUnload.getExtend6().length() < 255 , "扩展字段6,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getExtend7() == null || vehicleUnload.getExtend7().length() < 255 , "扩展字段7,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getTenantCode() == null || vehicleUnload.getTenantCode().length() < 255 , "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getVehicleUnloadCode() == null || vehicleUnload.getVehicleUnloadCode().length() < 64 , "卸货单编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleUnload.getVehicleTaskCode() == null || vehicleUnload.getVehicleTaskCode().length() < 64 , "出车任务编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleUnload.getVehicleCode() == null || vehicleUnload.getVehicleCode().length() < 64 , "车辆编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleUnload.getVehicleName() == null || vehicleUnload.getVehicleName().length() < 64 , "车辆名称,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleUnload.getSaleManAccount() == null || vehicleUnload.getSaleManAccount().length() < 255 , "业务员编号(账号),在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getSaleManName() == null || vehicleUnload.getSaleManName().length() < 255 , "业务员名称,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getCustomerCode() == null || vehicleUnload.getCustomerCode().length() < 64 , "客户编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleUnload.getCustomerName() == null || vehicleUnload.getCustomerName().length() < 64 , "客户名称,在进行添加时填入值超过了限定长度(64)，请检查!");
    VehicleUnload byVehicleUnloadCode = this.findByVehicleUnloadCode(vehicleUnload.getVehicleUnloadCode());
    Validate.isTrue(byVehicleUnloadCode == null, "卸货单编号已存在,请检查");
  }
  @Transactional
  @Override
  public VehicleUnload update(VehicleUnload vehicleUnload) {
    VehicleUnload current = this.updateForm(vehicleUnload);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public VehicleUnload updateForm(VehicleUnload vehicleUnload) {
    /* 
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     * 
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    
    this.updateValidation(vehicleUnload);
    // ===================基本信息
    String currentId = vehicleUnload.getId();
    Optional<VehicleUnload> op_currentVehicleUnload = this.vehicleUnloadRepository.findById(currentId);
    VehicleUnload currentVehicleUnload = op_currentVehicleUnload.orElse(null);
    currentVehicleUnload = Validate.notNull(currentVehicleUnload,"未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentVehicleUnload.setModifyAccount(SecurityUtils.getUserAccount());
    currentVehicleUnload.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentVehicleUnload.setExtend1(vehicleUnload.getExtend1());
    currentVehicleUnload.setExtend2(vehicleUnload.getExtend2());
    currentVehicleUnload.setExtend3(vehicleUnload.getExtend3());
    currentVehicleUnload.setExtend4(vehicleUnload.getExtend4());
    currentVehicleUnload.setExtend5(vehicleUnload.getExtend5());
    currentVehicleUnload.setExtend6(vehicleUnload.getExtend6());
    currentVehicleUnload.setExtend7(vehicleUnload.getExtend7());
    currentVehicleUnload.setExtend8(vehicleUnload.getExtend8());
    currentVehicleUnload.setExtend9(vehicleUnload.getExtend9());
    currentVehicleUnload.setExtend10(vehicleUnload.getExtend10());
    currentVehicleUnload.setExtend11(vehicleUnload.getExtend11());
    currentVehicleUnload.setTenantCode(vehicleUnload.getTenantCode());
    currentVehicleUnload.setVehicleUnloadCode(vehicleUnload.getVehicleUnloadCode());
    currentVehicleUnload.setVehicleUnloadType(vehicleUnload.getVehicleUnloadType());
    currentVehicleUnload.setVehicleProductType(vehicleUnload.getVehicleProductType());
    currentVehicleUnload.setVehicleUnloadStatus(vehicleUnload.getVehicleUnloadStatus());
    currentVehicleUnload.setVehicleTaskCode(vehicleUnload.getVehicleTaskCode());
    currentVehicleUnload.setVehicleCode(vehicleUnload.getVehicleCode());
    currentVehicleUnload.setVehicleName(vehicleUnload.getVehicleName());
    currentVehicleUnload.setSaleManAccount(vehicleUnload.getSaleManAccount());
    currentVehicleUnload.setSaleManName(vehicleUnload.getSaleManName());
    currentVehicleUnload.setCustomerCode(vehicleUnload.getCustomerCode());
    currentVehicleUnload.setCustomerName(vehicleUnload.getCustomerName());
    
    this.vehicleUnloadRepository.saveAndFlush(currentVehicleUnload);
    return currentVehicleUnload;
  }
  /**
   * 在更新一个已有的VehicleUnload模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(VehicleUnload vehicleUnload) {
    Validate.isTrue(!StringUtils.isBlank(vehicleUnload.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(vehicleUnload.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(vehicleUnload.getVehicleUnloadCode(), "修改信息时，卸货单编号不能为空！");
    Validate.notNull(vehicleUnload.getVehicleUnloadType(), "修改信息时，出货类型 1运输途中出货 2收车入库出货不能为空！");
    Validate.notNull(vehicleUnload.getVehicleProductType(), "修改信息时，车载商品类型不能为空！");
    Validate.notNull(vehicleUnload.getVehicleUnloadStatus(), "修改信息时，出货(出库)状态 1待确认 2已取消 3已确认 4已作废不能为空！");
    Validate.notBlank(vehicleUnload.getVehicleTaskCode(), "修改信息时，出车任务编号不能为空！");
    Validate.notBlank(vehicleUnload.getVehicleCode(), "修改信息时，车辆编码不能为空！");
    Validate.notBlank(vehicleUnload.getVehicleName(), "修改信息时，车辆名称不能为空！");
    Validate.notBlank(vehicleUnload.getSaleManAccount(), "修改信息时，业务员编号(账号)不能为空！");
    Validate.notBlank(vehicleUnload.getSaleManName(), "修改信息时，业务员名称不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(vehicleUnload.getExtend1() == null || vehicleUnload.getExtend1().length() < 255 , "扩展字段1,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getExtend2() == null || vehicleUnload.getExtend2().length() < 255 , "扩展字段2,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getExtend3() == null || vehicleUnload.getExtend3().length() < 255 , "扩展字段3,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getExtend4() == null || vehicleUnload.getExtend4().length() < 255 , "扩展字段4,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getExtend5() == null || vehicleUnload.getExtend5().length() < 255 , "扩展字段5,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getExtend6() == null || vehicleUnload.getExtend6().length() < 255 , "扩展字段6,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getExtend7() == null || vehicleUnload.getExtend7().length() < 255 , "扩展字段7,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getTenantCode() == null || vehicleUnload.getTenantCode().length() < 255 , "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getVehicleUnloadCode() == null || vehicleUnload.getVehicleUnloadCode().length() < 64 , "卸货单编号,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleUnload.getVehicleTaskCode() == null || vehicleUnload.getVehicleTaskCode().length() < 64 , "出车任务编号,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleUnload.getVehicleCode() == null || vehicleUnload.getVehicleCode().length() < 64 , "车辆编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleUnload.getVehicleName() == null || vehicleUnload.getVehicleName().length() < 64 , "车辆名称,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleUnload.getSaleManAccount() == null || vehicleUnload.getSaleManAccount().length() < 255 , "业务员编号(账号),在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getSaleManName() == null || vehicleUnload.getSaleManName().length() < 255 , "业务员名称,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUnload.getCustomerCode() == null || vehicleUnload.getCustomerCode().length() < 64 , "客户编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleUnload.getCustomerName() == null || vehicleUnload.getCustomerName().length() < 64 , "客户名称,在进行修改时填入值超过了限定长度(64)，请检查!");
  }

  /**
   * 生成装货单编码
   */
  private String generateCode(String tenantCode) {
    String redisKey = String.format(VEHICLE_UNLOAD_CODE_AUTO_INC_KEY, tenantCode);
    String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
    SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
    return StringUtils.join(VEHICLE_UNLOAD_CODE_PREFIX,format.format(new Date()), index);
  }

  @Override
  public VehicleUnload findByVehicleUnloadCode(String vehicleUnloadCode) {
    if(StringUtils.isBlank(vehicleUnloadCode) || StringUtils.isBlank(TenantUtils.getTenantCode())) {
      return null;
    }
    return vehicleUnloadRepository.findByVehicleUnloadCodeAndTenantCode(vehicleUnloadCode,TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public VehicleUnload deliveryVehicleUnload(String vehicleUnloadCode) {
    Validate.notBlank(vehicleUnloadCode, "根据卸货单编码完成交货，卸货单编码不能为空！");
    VehicleUnload vehicleUnload = vehicleUnloadRepository.findByVehicleUnloadCodeAndTenantCode(vehicleUnloadCode, TenantUtils.getTenantCode());
    Validate.isTrue(VehicleUnloadStatusEnum.WAIT_DELIVERY.getType().equals(vehicleUnload.getVehicleUnloadStatus()),
            "根据卸货单编码完成交货,卸货单状态必须为待交货状态");
    vehicleUnload.setVehicleUnloadStatus(VehicleUnloadStatusEnum.DELIVERY.getType());
    vehicleUnload.setVehicleUnloadTime(new Date());
    vehicleUnloadRepository.saveAndFlush(vehicleUnload);
    //完成库存扣减
    this.preemptionClose(vehicleUnloadCode);
    //如果该卸货单是含有关联订单编码的卸货单，并且卸货商品为车销商品 那么需要通知订单配送状态流转
    if(!CollectionUtils.isEmpty(vehicleUnloadListenerList)
            && StringUtils.isNotBlank(vehicleUnload.getRelevanceCode()) && VehicleProductTypeEnum.VEHICLE_SALES.getType().equals(vehicleUnload.getVehicleProductType())){
      for(VehicleUnloadListener listener : vehicleUnloadListenerList){
        listener.onCompleted(vehicleUnload);
      }
    }
    return vehicleUnload;
  }

  @Override
  @Transactional
  public void cancelVehicleUnload(String vehicleUnloadCode) {
    Validate.notBlank(vehicleUnloadCode, "根据卸货单编码取消时，卸货单编码不能为空！");
    VehicleUnload vehicleUnload = vehicleUnloadRepository.findByVehicleUnloadCodeAndTenantCode(vehicleUnloadCode, TenantUtils.getTenantCode());
    Validate.isTrue(VehicleUnloadStatusEnum.WAIT_DELIVERY.getType().equals(vehicleUnload.getVehicleUnloadStatus()),
            "根据卸货单编码取消时,卸货单状态必须为待交货状态");
    vehicleUnload.setVehicleUnloadStatus(VehicleUnloadStatusEnum.CANCEL.getType());
    vehicleUnloadRepository.saveAndFlush(vehicleUnload);
    //取消库存预占
    this.cancelPreemption(vehicleUnloadCode);
  }

  @Override
  @Transactional
  public void cancelByRelevanceCode(String relevanceCode) {
    Validate.notBlank(relevanceCode, "通过关联订单编码取消卸货单时，关联的订单编码不能为空！");
    String tenantCode = TenantUtils.getTenantCode();
    VehicleUnload vehicleUnload = vehicleUnloadRepository.findByRelevanceCodeAndVehicleProductTypeAndTenantCode(relevanceCode, VehicleProductTypeEnum.VEHICLE_SALES.getType(),tenantCode);
    Validate.notNull(vehicleUnload, "通过关联订单编码取消卸货单时，查询到的卸货单为空！");
    Validate.isTrue(VehicleUnloadStatusEnum.WAIT_DELIVERY.getType().equals(vehicleUnload.getVehicleUnloadStatus()),
            "取消卸货单时,卸货单状态必须为待交货状态");
    vehicleUnload.setVehicleUnloadStatus(VehicleUnloadStatusEnum.CANCEL.getType());
    vehicleUnloadRepository.saveAndFlush(vehicleUnload);
    //取消库存预占
    this.cancelPreemption(vehicleUnload.getVehicleUnloadCode());
  }

  @Override
  @Transactional
  public VehicleUnload saveVehicleUnload(VehicleUnload vehicleUnload) {
    vehicleUnload = create(vehicleUnload);
    //保存卸货商品
    vehicleUnloadProdctService.batchSave(vehicleUnload.getProducts());
    return vehicleUnload;
  }

  @Override
  public VehicleUnload findByRelevanceCode(String relevanceCode) {
    String tenantCode = TenantUtils.getTenantCode();
    if(StringUtils.isAnyBlank(relevanceCode,tenantCode)){
      return null;
    }
    return vehicleUnloadRepository.findByRelevanceCodeAndVehicleProductTypeAndTenantCode(relevanceCode, VehicleProductTypeEnum.VEHICLE_SALES.getType(),tenantCode);
  }

  @Override
  public List<VehicleUnload> findByVehicleTaskCodeAndUnloadStatus(String vehicleTaskCode, VehicleUnloadStatusEnum vehicleUnloadStatusEnum) {
    if (StringUtils.isEmpty(vehicleTaskCode)||null==vehicleUnloadStatusEnum){
      return Lists.newArrayList();
    }
    return vehicleUnloadRepository.findByVehicleTaskCodeAndUnloadStatusAndTenantCode(
            vehicleTaskCode,vehicleUnloadStatusEnum.getType(),TenantUtils.getTenantCode());
  }

  @Override
  public List<VehicleUnload> findByVehicleTaskCodeAndProductTypeAndUnloadStatus(String vehicleTaskCode, VehicleProductTypeEnum vehicleProductType, VehicleUnloadStatusEnum vehicleUnloadStatusEnum) {
    if (StringUtils.isEmpty(vehicleTaskCode)||null==vehicleProductType||null==vehicleUnloadStatusEnum){
      return Lists.newArrayList();
    }
    return vehicleUnloadRepository.findByVehicleTaskCodeAndProductTypeAndUnloadStatusAndTenantCode(
            vehicleTaskCode,vehicleProductType.getType(),vehicleUnloadStatusEnum.getType(),TenantUtils.getTenantCode());
  }

  @Override
  public List<VehicleUnload> findByVehicleTaskCodeAndProductType(String vehicleTaskCode, VehicleProductTypeEnum vehicleProductType) {
    if (StringUtils.isEmpty(vehicleTaskCode)||null==vehicleProductType){
      return Lists.newArrayList();
    }
    return vehicleUnloadRepository.findByVehicleTaskCodeAndProductTypeAndTenantCode(vehicleTaskCode,vehicleProductType.getType(),TenantUtils.getTenantCode());
  }

  @Override
  public List<VehicleUnload> findByVehicleTaskCode(String vehicleTaskCode) {
    if (StringUtils.isEmpty(vehicleTaskCode)){
      return null;
    }
    return vehicleUnloadRepository.findByVehicleTaskCodeAndTenantCode(vehicleTaskCode,TenantUtils.getTenantCode());
  }

  /**
   * 根据任务编号和车辆编号，查询卸货记录
   * 注：只查询运输途中出货，且非取消状态的出库信息
   * 包含车销出库和配送出库，不包含收车出库
   * @param vehicleTaskCode 出车任务编码
   * @param vehicleCode 车辆编码
   */
  @Override
  public List<VehicleUnload> findByVehicleTaskCodeAndVehicleCode(String vehicleTaskCode, String vehicleCode) {
    if(StringUtils.isAnyBlank(vehicleTaskCode,vehicleCode,TenantUtils.getTenantCode())){
      return Lists.newArrayList();
    }
    return vehicleUnloadRepository.findByVehicleTaskCodeAndVehicleCodeAndTenantCode(vehicleTaskCode,vehicleCode,TenantUtils.getTenantCode());
  }

  @Override
  public List<VehicleUnload> findByVehicleTaskCodeAndUnloadType(String vehicleTaskCode, Integer VehicleUnloadType) {
    if (StringUtils.isEmpty(vehicleTaskCode)||null==VehicleUnloadType){
      return Lists.newArrayList();
    }
    return vehicleUnloadRepository.findByVehicleTaskCodeAndUnloadTypeAndTenantCode(
            vehicleTaskCode,VehicleUnloadType,TenantUtils.getTenantCode());
  }

  @Override
  public VehicleUnload findDetailsById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.vehicleUnloadRepository.findDetailsById(id);
  }
  @Override
  public VehicleUnload findById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    Optional<VehicleUnload> op = vehicleUnloadRepository.findById(id);
    return op.orElse(null); 
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    VehicleUnload current = this.findById(id);
    if(current != null) { 
      this.vehicleUnloadRepository.delete(current);
    }
  }

  @Override
  @Transactional
  public VehicleUnload preemption(String vehicleUnloadCode) {
    Validate.notBlank(vehicleUnloadCode , "卸货单编码为空");
    VehicleUnload vehicleUnload = vehicleUnloadRepository.findByVehicleUnloadCodeAndTenantCode(vehicleUnloadCode, TenantUtils.getTenantCode());
    Validate.notNull(vehicleUnload , "未查询到卸货单信息");
    Set<VehicleUnloadProduct> products = vehicleUnload.getProducts();
    Validate.notEmpty(products, "查询到卸货单中的商品为空");
    for (VehicleUnloadProduct product : products) {
      //根据商品占用库存
      vehicleProductStockService.preemption(product.getProductSpecificationCode(),vehicleUnload.getVehicleCode(),
              product.getQuantity(),product.getUnitCode(),vehicleUnload.getVehicleProductType());
    }
    return vehicleUnload;
  }

  @Override
  @Transactional
  public VehicleUnload cancelPreemption(String vehicleUnloadCode) {
    Validate.notBlank(vehicleUnloadCode , "卸货单编码为空");
    VehicleUnload vehicleUnload = vehicleUnloadRepository.findByVehicleUnloadCodeAndTenantCode(vehicleUnloadCode, TenantUtils.getTenantCode());
    Validate.notNull(vehicleUnload , "未查询到卸货单信息");
    Set<VehicleUnloadProduct> products = vehicleUnload.getProducts();
    Validate.notEmpty(products, "查询到卸货单中的商品为空");
    for (VehicleUnloadProduct product : products) {
      //根据商品取消车辆库存预占
      vehicleProductStockService.cancelPreemption(product.getProductSpecificationCode(),vehicleUnload.getVehicleCode(),
              product.getQuantity(),product.getUnitCode(),vehicleUnload.getVehicleProductType());
    }
    return vehicleUnload;
  }

  @Override
  @Transactional
  public VehicleUnload preemptionClose(String vehicleUnloadCode) {
    Validate.notBlank(vehicleUnloadCode , "卸货单编码为空");
    VehicleUnload vehicleUnload = vehicleUnloadRepository.findByVehicleUnloadCodeAndTenantCode(vehicleUnloadCode, TenantUtils.getTenantCode());
    Validate.notNull(vehicleUnload , "未查询到卸货单信息");
    Set<VehicleUnloadProduct> products = vehicleUnload.getProducts();
    Validate.notEmpty(products, "查询到卸货单中的商品为空");
    for (VehicleUnloadProduct product : products) {
      //根据商品预占的数据清空，库存消减
      vehicleProductStockService.preemptionClose(product.getProductSpecificationCode(),vehicleUnload.getVehicleCode(),
              product.getQuantity(),product.getUnitCode(),vehicleUnload.getVehicleProductType());
      //更改出车任务商品数量
    }
    return vehicleUnload;
  }
} 
