package com.bizunited.empower.business.vehicle.service.internal;
import com.bizunited.empower.business.vehicle.entity.VehicleUserMapping;
import com.bizunited.empower.business.vehicle.repository.VehicleUserMappingRepository;
import com.bizunited.empower.business.vehicle.service.VehicleUserMappingService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.List;
import java.util.Optional;

/**
 * VehicleUserMapping业务模型的服务层接口实现
 * @author saturn
 */
@Service("VehicleUserMappingServiceImpl")
public class VehicleUserMappingServiceImpl implements VehicleUserMappingService { 
  @Autowired
  private VehicleUserMappingRepository vehicleUserMappingRepository;
  @Transactional
  @Override
  public VehicleUserMapping create(VehicleUserMapping vehicleUserMapping) { 
    VehicleUserMapping current = this.createForm(vehicleUserMapping);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public VehicleUserMapping createForm(VehicleUserMapping vehicleUserMapping) { 
   /* 
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */
    vehicleUserMapping.setTenantCode(TenantUtils.getTenantCode());
    this.createValidation(vehicleUserMapping);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.vehicleUserMappingRepository.save(vehicleUserMapping);
    // 返回最终处理的结果，里面带有详细的关联信息
    return vehicleUserMapping;
  }
  /**
   * 在创建一个新的VehicleUserMapping模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(VehicleUserMapping vehicleUserMapping) { 
    Validate.notNull(vehicleUserMapping , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(vehicleUserMapping.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    vehicleUserMapping.setId(null);
    Validate.notBlank(vehicleUserMapping.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(vehicleUserMapping.getVehicleCode(), "添加信息时，车辆编码不能为空！");
    Validate.notBlank(vehicleUserMapping.getUserAccount(), "添加信息时，业务员账号不能为空！");
    Validate.notBlank(vehicleUserMapping.getUserName(), "添加信息时，业务员名称不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(vehicleUserMapping.getTenantCode() == null || vehicleUserMapping.getTenantCode().length() < 255 , "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUserMapping.getVehicleCode() == null || vehicleUserMapping.getVehicleCode().length() < 64 , "车辆编码,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleUserMapping.getUserAccount() == null || vehicleUserMapping.getUserAccount().length() < 128 , "业务员账号,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(vehicleUserMapping.getUserName() == null || vehicleUserMapping.getUserName().length() < 128 , "业务员名称,在进行添加时填入值超过了限定长度(128)，请检查!");
  }
  @Transactional
  @Override
  public VehicleUserMapping update(VehicleUserMapping vehicleUserMapping) { 
    VehicleUserMapping current = this.updateForm(vehicleUserMapping);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  } 
  @Transactional
  @Override
  public VehicleUserMapping updateForm(VehicleUserMapping vehicleUserMapping) { 
    /* 
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     * 
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    
    this.updateValidation(vehicleUserMapping);
    // ===================基本信息
    String currentId = vehicleUserMapping.getId();
    Optional<VehicleUserMapping> op_currentVehicleUserMapping = this.vehicleUserMappingRepository.findById(currentId);
    VehicleUserMapping currentVehicleUserMapping = op_currentVehicleUserMapping.orElse(null);
    currentVehicleUserMapping = Validate.notNull(currentVehicleUserMapping ,"未发现指定的原始模型对象信");
    // 开始重新赋值——一般属性
    currentVehicleUserMapping.setTenantCode(vehicleUserMapping.getTenantCode());
    currentVehicleUserMapping.setVehicleCode(vehicleUserMapping.getVehicleCode());
    currentVehicleUserMapping.setUserAccount(vehicleUserMapping.getUserAccount());
    currentVehicleUserMapping.setUserName(vehicleUserMapping.getUserName());
    
    this.vehicleUserMappingRepository.saveAndFlush(currentVehicleUserMapping);
    return currentVehicleUserMapping;
  }
  /**
   * 在更新一个已有的VehicleUserMapping模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(VehicleUserMapping vehicleUserMapping) { 
    Validate.isTrue(!StringUtils.isBlank(vehicleUserMapping.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(vehicleUserMapping.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(vehicleUserMapping.getVehicleCode(), "修改信息时，车辆编码不能为空！");
    Validate.notBlank(vehicleUserMapping.getUserAccount(), "修改信息时，业务员账号不能为空！");
    Validate.notBlank(vehicleUserMapping.getUserName(), "修改信息时，业务员名称不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(vehicleUserMapping.getTenantCode() == null || vehicleUserMapping.getTenantCode().length() < 255 , "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(vehicleUserMapping.getVehicleCode() == null || vehicleUserMapping.getVehicleCode().length() < 64 , "车辆编码,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(vehicleUserMapping.getUserAccount() == null || vehicleUserMapping.getUserAccount().length() < 128 , "业务员账号,在进行修改时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(vehicleUserMapping.getUserName() == null || vehicleUserMapping.getUserName().length() < 128 , "业务员名称,在进行修改时填入值超过了限定长度(128)，请检查!");
  } 
  @Override
  public VehicleUserMapping findDetailsById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.vehicleUserMappingRepository.findDetailsById(id);
  }
  @Override
  public VehicleUserMapping findById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    Optional<VehicleUserMapping> op = vehicleUserMappingRepository.findById(id);
    return op.orElse(null); 
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    VehicleUserMapping current = this.findById(id);
    if(current != null) { 
      this.vehicleUserMappingRepository.delete(current);
    }
  }

  @Override
  @Transactional
  public void deleteByVehicleCode(String vehicleCode) {
    Validate.notBlank(vehicleCode , "进行删除关联时，必须给定车辆编号信息!!");
    vehicleUserMappingRepository.deleteByVehicleCodeAndTenantCode(vehicleCode, TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public void batchSave(List<VehicleUserMapping> vehicleUserMappings) {
    Validate.notEmpty(vehicleUserMappings,"进行保存车辆与业务员关联信息时，集合不能为空!!");
    for (VehicleUserMapping vehicleUserMapping : vehicleUserMappings) {
      create(vehicleUserMapping);
    }
  }

  @Override
  public List<VehicleUserMapping> findByVehicleCode(String vehicleCode) {
    if (StringUtils.isEmpty(vehicleCode)){
      return Lists.newArrayList();
    }
    return vehicleUserMappingRepository.findByVehicleCode(vehicleCode,TenantUtils.getTenantCode());
  }
} 
