package com.bizunited.platform.dictionary.service.local.controller;

import com.alibaba.fastjson.JSON;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseCode;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.dictionary.common.service.dictcategory.DictCategoryService;
import com.bizunited.platform.dictionary.common.vo.DictCategoryVo;
import com.bizunited.platform.dictionary.common.vo.DictItemVo;
import com.bizunited.platform.dictionary.common.vo.DictVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * DictCategoryController
 *
 * @description:
 * @author: yanwe
 * @date: 12/Jun/2019 09:56
 */
@RestController
@RequestMapping("/v1/nebula/dictCategories")
public class DictCategoryController extends BaseController {

  private static final Logger LOGGER = LoggerFactory.getLogger(DictCategoryController.class);

  @Autowired
  private DictCategoryService dictCategoryService;

  @ApiOperation(value = "创建字典分类")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel create(
      @ApiParam(name = "字典分类信息") @RequestBody DictCategoryVo dictCategoryVo,
      @RequestParam(name = "parentId", required=false) @ApiParam(name = "parentId", value = "父级字典ID", required=false)
          String parentId) {
    LOGGER.debug("创建字典分类：{}", JSON.toJSONString(dictCategoryVo));
    try {
      DictCategoryVo entity = dictCategoryService.create(dictCategoryVo,parentId);
      return this.buildHttpResultW(entity, "");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "修改字典分类")
  @RequestMapping(value = "", method = RequestMethod.PATCH)
  public ResponseModel update(
      @ApiParam(name = "字典分类信息") @RequestBody DictCategoryVo dictCategoryVo) {
    LOGGER.debug("修改字典分类：{}", JSON.toJSONString(dictCategoryVo));
    try {
      DictCategoryVo entity = dictCategoryService.update(dictCategoryVo);
      return this.buildHttpResultW(entity, "");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "设置字典分类层级")
  @RequestMapping(value = "/bind", method = RequestMethod.POST)
  public ResponseModel bind(
      @RequestParam(name = "childId") @ApiParam(name = "childId", value = "子级字典ID") String childId,
      @RequestParam(name = "parentId") @ApiParam(name = "parentId", value = "父级字典ID")
          String parentId) {
    try {
      dictCategoryService.bind(childId, parentId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "根据数据字典业务编号code进行查询，不包括任何关联信息")
  @RequestMapping(value = "/findByCode", method = RequestMethod.GET)
  public ResponseModel findByCode(@RequestParam("code") @ApiParam(name = "code" , value = "业务编号code在全系统唯一") String code) {
    try {
      DictCategoryVo current = dictCategoryService.findByCode(code);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "根据数据字典业务编号code进行查询，包括字典及字典详情关联信息")
  @RequestMapping(value = "/findCascadedDetailByCode", method = RequestMethod.GET)
  public ResponseModel findCascadedDetailByCode(@RequestParam @ApiParam(name = "code" , value = "业务编号code在全系统唯一") String code) {
    try {
      DictCategoryVo current = dictCategoryService.findCascadedDetailByCode(code);
      return this.buildHttpResultW(current, "dicts", "dicts.dictItems");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "查询所有字典分类")
  @RequestMapping(value = "/findAll", method = RequestMethod.GET)
  public ResponseModel findAll() {
    try {
      Set<DictCategoryVo> entities = dictCategoryService.findAll();
      // 组装过滤返回树级结构
      List<DictCategoryVo> entitiesList = new ArrayList<>();
      entitiesList.addAll(entities);
      this.filterDictCate(entitiesList);
      ResponseModel responseModel = new ResponseModel();
      responseModel.setTimestamp(System.currentTimeMillis());
      responseModel.setData(entitiesList);
      responseModel.setSuccess(true);
      responseModel.setResponseCode(ResponseCode.E0);
      return responseModel;
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 自组装返还的多层级字典分类结构,过滤掉entity中关联的非子级数据
   * @param entities
   */
  private void filterDictCate(List<DictCategoryVo> entities) {
    if (CollectionUtils.isEmpty(entities)) {
      return;
    }
    for (DictCategoryVo entity : entities) {
      entity.setParentCategory(null);
      Set<DictVo> dicts = entity.getDicts();
      if (!CollectionUtils.isEmpty(dicts)) {
        for (DictVo dict : dicts) {
          dict.setCategory(null);
          if(!CollectionUtils.isEmpty(dict.getDictItems())){
            for(DictItemVo item : dict.getDictItems()){
              item.setDictEntity(null);
            }
          }
        }
      }
      Set<DictCategoryVo> child = entity.getChildCategory();
      if(child == null){
        child = new HashSet<>();
      }
      List<DictCategoryVo> childList = new ArrayList<>();
      childList.addAll(child);
      if (!CollectionUtils.isEmpty(child)) {
        this.filterDictCate(childList);
      }
    }
  }

  @ApiOperation(value = "根据id查询基本信息，没有任何关联信息")
  @GetMapping("/findById")
  public ResponseModel findById(@RequestParam("id") @ApiParam(name = "id" , value = "字典分组id") String id) {
    try {
      DictCategoryVo current = dictCategoryService.findById(id);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "根据id查询字典分类的基本信息和关联信息（但是不包括字典分类的子级集合关联）")
  @GetMapping("/findDetailsById")
  public ResponseModel findDetailsById(@RequestParam("id") @ApiParam(name = "id" , value = "字典分组id") String id) {
    try {
      DictCategoryVo current = dictCategoryService.findDetailsById(id);
      return this.buildHttpResultW(current, "parentCategory");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据分组id删除分组（不能删除有字典的分组）
   * @param id
   * @return
   */
  @ApiOperation(value = "根据分组id删除分组（不能删除有字典的分组）")
  @DeleteMapping("deleteById")
  public ResponseModel deleteById(@RequestParam("id") @ApiParam(name = "id" , value = "字典分组id") String id) {
    try {
      this.dictCategoryService.deleteById(id);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }
}
