package com.bizunited.platform.dictionary.service.local.service.init;

import com.alibaba.fastjson.JSONArray;
import com.bizunited.platform.core.service.init.InitProcessEnvironmentService;
import com.bizunited.platform.dictionary.common.service.dict.DictService;
import com.bizunited.platform.dictionary.common.service.dictItem.DictItemService;
import com.bizunited.platform.dictionary.common.service.dictcategory.DictCategoryService;
import com.bizunited.platform.dictionary.common.vo.DictCategoryVo;
import com.bizunited.platform.dictionary.common.vo.DictItemVo;
import com.bizunited.platform.dictionary.common.vo.DictVo;
import com.bizunited.platform.dictionary.service.local.repository.DictItemRepository;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Set;

/**
 * 系统初始化时增加组织机构枚举类
 * @Author: TanBoQiuYun
 * @Date: 2019/12/2 14:14
 */
@Component("DictionaryInitProcess")
public class DictionaryInitProcess implements InitProcessEnvironmentService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DictionaryInitProcess.class);

  private static final String INIT_FILE_PATH = ResourceLoader.CLASSPATH_URL_PREFIX.concat("initjsons/dicts.json");
  @Autowired
  private DictCategoryService dictCategoryService;
  @Autowired
  private DictService dictService;
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private DictItemService dictItemService;
  @Autowired
  private DictItemRepository dictItemRepository;

  @Override
  public int sort() {
    // 初始化动作靠前
    return 5;
  }

  /**
   * 只有返回true才进行初始化动作
   *
   * @return
   */
  @Override
  public boolean doInitForAppCode(String appCode) {
    return true;
  }

  @Override
  public boolean stopOnException() {
    // 一旦出现异常，则终止进程启动过程
    return true;
  }

  @Override
  public void initForAppCode(String appCode) {
    Resource resource = applicationContext.getResource(INIT_FILE_PATH);
    if(!resource.exists()) {
      LOGGER.warn("init：正在进行数据字典模块的初始化，但未发现对应的初始化文件信息：{}" , INIT_FILE_PATH);
      return;
    }
    try (InputStream inputStream = resource.getInputStream()) {
      StringBuilder sb = new StringBuilder();
      byte[] buffer = new byte[512];
      int line;
      while ((line = inputStream.read(buffer, 0, buffer.length)) != -1) {
        sb.append(new String(buffer, 0, line, StandardCharsets.UTF_8));
      }
      String json = sb.toString();
      if (StringUtils.isBlank(json)) {
        LOGGER.warn("初始化数据字典json为空！！");
        return;
      }
      List<DictCategoryVo> dictCategoryVos = JSONArray.parseArray(json, DictCategoryVo.class);
      if(CollectionUtils.isEmpty(dictCategoryVos)) {
        return;
      }
      dictCategoryVos.forEach(this::initDictCategory);
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
    }
  }

  /**
   * 初始化字典分类表
   * @return
   * @param dictCategoryVo
   */
  private void initDictCategory(DictCategoryVo dictCategoryVo) {
    if(dictCategoryVo == null) {
      return;
    }
    // 初始化数据字典分组，根据分类编码查询是否存在，不存在就初始化
    DictCategoryVo categoryVo = this.dictCategoryService.findByCode(dictCategoryVo.getCateCode());
    if (categoryVo == null) {
      dictCategoryVo.setCreateAccount("admin");
      dictCategoryVo.setModifyAccount("admin");
      categoryVo = this.dictCategoryService.create(dictCategoryVo, null);
    }
    this.initDict(dictCategoryVo.getDicts(), categoryVo.getId());
  }

  /**
   * 初始化数据字典
   * @param dicts
   * @param categoryId
   */
  private void initDict(Set<DictVo> dicts, String categoryId) {
    if(CollectionUtils.isEmpty(dicts)) {
      return;
    }
    dicts.forEach(dictVo -> {
      DictVo oldDict = this.dictService.findByDictCode(dictVo.getDictCode());
      if(oldDict == null) {
        this.dictService.create(dictVo, categoryId);
      }else{
        dictVo.setId(oldDict.getId());
        initDictItem(dictVo.getDictItems(), dictVo);
      }
    });
  }

  /**
   * 初始化数据字典明细
   * @param dictItems
   * @param dictVo
   */
  private void initDictItem(Set<DictItemVo> dictItems, DictVo dictVo) {
    if(CollectionUtils.isEmpty(dictItems)) {
      return;
    }
    dictItems.forEach(item -> {
      long dictKeyCount = this.dictItemRepository.countByDictKeyAndDictId(item.getDictKey(), dictVo.getId());
      long dictValueCount = this.dictItemRepository.countByDictValueAndDictId(item.getDictValue(), dictVo.getId());
      if(dictKeyCount == 0L && dictValueCount == 0L) {
        this.dictItemService.create(dictVo.getDictCode(), item);
      }
    });
  }
}
