package com.bizunited.platform.dictionary.service.local.service.internal;

import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.dictionary.common.service.dict.DictService;
import com.bizunited.platform.dictionary.common.service.dictItem.DictItemService;
import com.bizunited.platform.dictionary.common.vo.DictItemVo;
import com.bizunited.platform.dictionary.common.vo.DictVo;
import com.bizunited.platform.dictionary.service.local.entity.DictEntity;
import com.bizunited.platform.dictionary.service.local.entity.DictItemEntity;
import com.bizunited.platform.dictionary.service.local.repository.DictItemRepository;
import com.bizunited.platform.dictionary.service.local.repository.DictRepository;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

@Service("DictItemServiceImpl")
public class DictItemServiceImpl implements DictItemService {

  private static final String ERROR_MESS = "字典明细ID不能为空！";
  @Autowired 
  private DictRepository dictRepository;
  @Autowired 
  private DictItemRepository dictItemRepository;
  @Autowired
  private DictService dictService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public List<DictItemVo> findItemsByCode(String dictCode) {
    DictEntity dictEntity = dictRepository.findByDictCode(dictCode);
    if (dictEntity == null) {
      return Lists.newArrayList();
    }
    List<DictItemEntity> dictItems = dictItemRepository.findByDictEntityOrderByDictSort(dictEntity);
    Collection<DictItemVo> dictItemVos = this.nebulaToolkitService.copyCollectionByWhiteList(dictItems, DictItemEntity.class, DictItemVo.class, LinkedHashSet.class, ArrayList.class);
    return Lists.newArrayList(dictItemVos);
  }

  /**
   * 明细的更新目前只涉及新增和更新，删除暂不开发
   */
  @Override
  @Transactional
  public Set<DictItemVo> save(String dictCode, Set<DictItemVo> items) {
    if(CollectionUtils.isEmpty(items)){
      return Sets.newHashSet();
    }
    Validate.notBlank(dictCode,"字典编码不能为空");
    DictVo dictVo = dictService.findDetailsByDictCode(dictCode);
    Validate.notNull(dictVo, "未找到该编码匹配字典！");
    DictEntity dbDictEntity = this.nebulaToolkitService.copyObjectByWhiteList(dictVo, DictEntity.class, HashSet.class, ArrayList.class);
    //校验分析（区分新增和更新，且验重）
    Collection<DictItemEntity> data = nebulaToolkitService.copyCollectionByWhiteList(items,DictItemVo.class,DictItemEntity.class,LinkedHashSet.class,ArrayList.class);
    if(!CollectionUtils.isEmpty(data)){
      dictItemRepository.deleteDictItemEntityByDictEntity(dbDictEntity);
      dictItemRepository.flush();
      data.stream().forEach(item->item.setDictEntity(dbDictEntity));
      dictItemRepository.saveAll(data);
    }
    Collection<DictItemVo> dictItemVos = this.nebulaToolkitService.copyCollectionByWhiteList(data, DictItemEntity.class, DictItemVo.class, LinkedHashSet.class, ArrayList.class);
    return Sets.newLinkedHashSet(dictItemVos);
  }

  @Override
  @Transactional
  public DictItemVo create(String dictCode, DictItemVo dictItem) {
    //参数校验
    this.createValidation(dictCode, dictItem);

    DictVo dict = dictService.findByDictCode(dictCode);
    dict.setItemNum(dict.getItemNum() + 1);
    DictEntity dictEntity = this.nebulaToolkitService.copyObjectByWhiteList(dict, DictEntity.class, HashSet.class, ArrayList.class, "category");
    dictRepository.save(dictEntity);

    DictItemEntity dictItemEntity = this.nebulaToolkitService.copyObjectByWhiteList(dictItem, DictItemEntity.class, HashSet.class, ArrayList.class);
    dictItemEntity.setCreateTime(new Date());
    dictItemEntity.setDictSort(dictItem.getDictSort() == null ? 1 : dictItem.getDictSort());
    dictItemEntity.setDictEntity(dictEntity);
    dictItemEntity.setDictItemStatus(dictItem.getDictItemStatus() == null ? Boolean.TRUE : dictItem.getDictItemStatus());
    DictItemEntity entity = dictItemRepository.save(dictItemEntity);
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, DictItemVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public DictItemVo findDetailsById(String id) {
    Validate.notNull(id, "查询字典值明细参数为空，请检查!");
    DictItemEntity entity = dictItemRepository.findDetailById(id);
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, DictItemVo.class, HashSet.class, ArrayList.class,  "dict");
  }

  /**
   * 根据字典dictCode和字典值的key查询一个数据字典值详细
   *
   * @param dictCode
   * @param key
   * @return
   */
  @Override
  public DictItemVo findByKeyAndDictCode(String dictCode, String key) {
    if (StringUtils.isBlank(dictCode) || StringUtils.isBlank(key)){
      return null;
    }
    DictItemEntity dictItem = this.dictItemRepository.findByKeyAndDictCode(dictCode, key);
    if (dictItem == null){
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(dictItem, DictItemVo.class, HashSet.class, ArrayList.class,  "dictEntity");
  }

  @Override
  @Transactional
  public void deleteByIds(Set<String> ids) {
    if(CollectionUtils.isEmpty(ids)){
      return;
    }
    List<DictItemEntity> existItems = dictItemRepository.findAllById(ids);
    if(!CollectionUtils.isEmpty(existItems)){
      dictItemRepository.deleteInBatch(existItems);
    }
  }

  /**
   * 新增-参数校验
   * @param dictCode
   * @param dictItem
   */
  private void createValidation(String dictCode, DictItemVo dictItem) {
    Validate.notBlank(dictCode, "字典编码不能为空，请检查");
    DictVo dict = dictService.findByDictCode(dictCode);
    Validate.notNull(dict, "未找到该编码匹配字典！");

    Validate.notNull(dictItem, "保存对象不存在!");
    Validate.isTrue(dictItem.getId() == null, "保存数据ID必须为空!");
    Validate.notBlank(dictItem.getDictKey(), "字典名称不能为空，请检查!");
    Validate.notBlank(dictItem.getDictValue(), "字典值不能为空，请检查!");

    long countByDictKey = dictItemRepository.countByDictKeyAndDictId(dictItem.getDictKey(), dict.getId());
    Validate.isTrue(countByDictKey == 0L, "字典名称重复，请检查!");
    long countByDictValue = dictItemRepository.countByDictValueAndDictId(dictItem.getDictValue(), dict.getId());
    Validate.isTrue(countByDictValue == 0L, "字典值重复，请检查!");
  }

  @Override
  @Transactional
  public DictItemVo enable(String dictItemId) {
    Validate.notBlank(dictItemId, ERROR_MESS);
    DictItemEntity dictItemEntity = dictItemRepository.findDetailById(dictItemId);
    Validate.notNull(dictItemEntity, ERROR_MESS);
    dictItemEntity.setDictItemStatus(true);
    DictItemEntity dictItem = dictItemRepository.save(dictItemEntity);
    return this.nebulaToolkitService.copyObjectByWhiteList(dictItem, DictItemVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public DictItemVo disable(String dictItemId) {
    Validate.notBlank(dictItemId, ERROR_MESS);
    DictItemEntity dictItemEntity = dictItemRepository.findDetailById(dictItemId);
    Validate.notNull(dictItemEntity, ERROR_MESS);
    dictItemEntity.setDictItemStatus(false);
    DictItemEntity dictItem = dictItemRepository.save(dictItemEntity);
    return this.nebulaToolkitService.copyObjectByWhiteList(dictItem, DictItemVo.class, HashSet.class, ArrayList.class);
  }
}
