package com.bizunited.platform.rbac2.server.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.rbac2.server.starter.entity.ButtonEntity;
import com.bizunited.platform.rbac2.server.starter.service.ButtonService;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Set;

/**
 * 按钮的管理类，负责提供相关接口
 * @author: zengxingwang yinwenjie
 * @version 2.0
 * @date: 2020/1/13 14:37
 */
@RestController
@RequestMapping(value="/v1/rbac/buttons")
public class ButtonController extends BaseController {

  /** 日志 */
  private static final Logger LOGGER = LoggerFactory.getLogger(ButtonController.class);

  @Autowired
  private ButtonService buttonService;

  /**
   * 创建按钮
   * @param button
   * @return
   */
  @PostMapping("")
  @ApiOperation(value = "创建按钮", notes = "创建按钮，必须要传递菜单对象，并且菜单对象ID不能为空")
  public ResponseModel create(@ApiParam("按钮实体信息") @RequestBody ButtonEntity button){
    try {
      ButtonEntity result = this.buttonService.create(button);
      return buildHttpResultW(result);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 批量创建按钮
   * @param buttons
   * @return
   */
  @PostMapping("/batchCreate")
  @ApiOperation(value = "批量创建按钮", notes = "创建按钮，必须要传递菜单对象，并且菜单对象ID不能为空")
  public ResponseModel batchCreate(@ApiParam("按钮实体信息") @RequestBody Set<ButtonEntity> buttons){
    try {
      this.buttonService.batchCreate(buttons);
      return buildHttpResult();
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 修改按钮
   * @param button
   * @return
   */
  @PatchMapping
  @ApiOperation(value = "修改按钮", notes = "修改按钮")
  public ResponseModel update(@ApiParam("按钮实体信息") @RequestBody ButtonEntity button){
    try {
      ButtonEntity result = this.buttonService.update(button);
      return buildHttpResultW(result);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }


  /**
   * 根据id删除按钮(批量)
   * @param ids
   * @return
   */
  @DeleteMapping("batchDelete")
  @ApiOperation(value = "根据id批量删除按钮")
  public ResponseModel batchDelete(@RequestParam(name = "ids") @ApiParam(name = "ids", value = "指定的按钮id信息(可一个是一个或者多个)") String[] ids){
    try {
      this.buttonService.batchDelete(ids);
      return buildHttpResult();
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据id删除按钮
   * @param id
   * @return 
   */
  @DeleteMapping("/{id}")
  @ApiOperation(value = "根据id删除按钮", notes = "根据id删除按钮")
  public ResponseModel deleteById(@ApiParam(name = "id", value = "按钮ID", required = true) @PathVariable("id") String id){
    try {
      this.buttonService.deleteById(id);
      return buildHttpResult();
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }
  
  /**
   * 建立指定角色和指定按钮的绑定关系
   * @param roleId
   * @param buttonIds
   * @return 
   */
  @ApiOperation(value = "建立指定角色和指定按钮的绑定关系")
  @PatchMapping(value = "/bindRole")
  public ResponseModel bindRole(@RequestParam(name = "roleId") @ApiParam(name = "roleId", value = "指定的角色编号信息") String roleId,
                                            @RequestParam(name = "buttonIds") @ApiParam(name = "buttonIds", value = "指定的按钮编号信息(可一个是一个或者多个)") String[] buttonIds) {
    try {
      this.buttonService.bindRole(roleId, buttonIds);
      return this.buildHttpResult();
    } catch(Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 重新建立指定角色和指定按钮的绑定关系
   * @param roleId
   * @param buttonIds
   * @return
   */
  @ApiOperation(value = "重新建立指定角色和指定按钮的绑定关系")
  @PatchMapping(value = "/rebindRole")
  public ResponseModel rebindRole(@RequestParam(name = "roleId") @ApiParam(name = "roleId", value = "指定的角色编号信息") String roleId,
                                            @RequestParam(name = "buttonIds") @ApiParam(name = "buttonIds", value = "指定的按钮编号信息(可一个是一个或者多个)") String[] buttonIds) {
    try {
      this.buttonService.rebindRole(roleId, buttonIds);
      return this.buildHttpResult();
    } catch(Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 取消指定角色和指定按钮的绑定关系
   * @param roleId
   * @param buttonIds
   * @return
   */
  @ApiOperation(value = "取消指定角色和指定按钮的绑定关系")
  @PatchMapping(value = "/unbindRole")
  public ResponseModel unbindRole(@RequestParam(name = "roleId") @ApiParam(name = "roleId", value = "指定的角色编号信息") String roleId,
                                              @RequestParam(name = "buttonIds") @ApiParam(name = "buttonIds", value = "指定的按钮编号信息(可一个是一个或者多个)") String[] buttonIds) {
    try {
      this.buttonService.unbindRole(roleId, buttonIds);
      return this.buildHttpResult();
    } catch(Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}
