package com.bizunited.platform.rbac2.server.starter.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.util.tenant.TenantUtils;

import com.bizunited.platform.rbac2.server.starter.dto.RoleConditionDto;
import com.bizunited.platform.rbac2.server.starter.entity.RoleEntity;
import com.bizunited.platform.rbac2.server.starter.service.RoleService;
import com.google.common.collect.Lists;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * 这里主要是放置和角色写操作相关的HTTP接口，以及直接从数据库持久层（性能不高）读取信息的接口
 * @author yinwenjie
 */
@RestController
@RequestMapping("/v1/rbac/roles")
public class RoleController extends BaseController {

  @Autowired
  private RoleService roleService;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(RoleController.class);

  /**
   * 添加一个角色信息
   * @param role
   * @return 创建后的角色基本信息将会被返回
   */
  @ApiOperation(value = "添加一个角色信息")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel createRole(@RequestBody @ApiParam(value = "role", name = "新建的角色对象") RoleEntity role) {
    try {
      if (role == null) {
        throw new IllegalArgumentException("role info not be null!");
      }
      RoleEntity currentRole = roleService.create(role);
      return this.buildHttpResultW(currentRole);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 修改一个指定的角色信息，注意配置在roles.deleteDeny属性的信息不能进行修改操作<br>
   * 且指定的一个角色只能修改角色的comment信息
   *
   * @param role 指定的修改信息
   */
  @ApiOperation(value = "修改一个指定的角色信息，注意配置在roles.sDeny属性的信息不能进行修改操作。且指定的一个角色只能修改角色的comment信息")
  @RequestMapping(value = "", method = RequestMethod.PATCH)
  public ResponseModel updateRole(@RequestBody @ApiParam(value = "role", name = "修改的角色对象") RoleEntity role) {
    try {
      RoleEntity currentRole = this.roleService.update(role);
      return this.buildHttpResultW(currentRole);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 批量重新启用某一个指定的角色信息
   *
   * @param roleIds
   * @return
   */
  @ApiOperation(value = "重新启用某一个指定的角色信息")
  @RequestMapping(value = "/enable", method = RequestMethod.POST)
  public ResponseModel enableRole(@RequestBody String[] roleIds) {
    try {
      this.roleService.enable(roleIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 批量禁用某一个指定的角色信息（相当于删除）<br>
   * 只是系统中不能真正的删除某一个角色，只能是将这个角色作废掉或者恢复正常状态
   *
   * @param roleIds
   * @return
   */
  @ApiOperation(value = "禁用某一个指定的角色信息（相当于删除）<br>" + "只是系统中不能真正的删除某一个角色，只能是将这个角色作废掉或者恢复正常状态")
  @RequestMapping(value = "/disable", method = RequestMethod.POST)
  public ResponseModel disableRole(@RequestBody String[] roleIds) {
    try {
      this.roleService.disable(roleIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据角色编码删除（只能删除无绑定关系的角色）
   * @param roleIds
   * @return
   */
  @ApiOperation(value = "根据角色技术编码删除（只能删除无绑定关系的角色）")
  @PostMapping(value = "/deleteByIds")
  public ResponseModel deleteByIds(@ApiParam(value = "角色编码数组") @RequestParam(name = "roleIds") String[] roleIds) {
    try {
      this.roleService.deleteByIds(roleIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 根据角色名称，或则角色编码，或则角色编码查询角色信息 （角色名称支持模糊查询）
   *
   * @param roleName
   * @return
   */
  @ApiOperation(value = "根据角色名称，或则角色编码，或则角色编码查询角色信息 （角色名称支持模糊查询）")
  @GetMapping(value = "/findByConditions")
  public ResponseModel findByConditions(@ApiParam(value = "分页参数") @PageableDefault(size = 50) Pageable pageable , RoleConditionDto conditions) {
    try {
      Page<RoleEntity> roles = this.roleService.findByConditions(conditions, pageable);
      return this.buildHttpResultW(roles, "parent");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 根据id查询详情信息
   * @param id
   * @return
   */
  @GetMapping("/findDetailsByIds")
  @ApiOperation("根据id查询详情信息,多个信息")
  public ResponseModel findDetailsByIds(@RequestParam(name = "ids") @ApiParam("主键ID，可以传入多个") String[] ids) {
    try {
      if(ids == null) {
        return this.buildHttpResult();
      }
      Set<RoleEntity> roles = roleService.findDetailsByIds(Lists.newArrayList(ids));
      return this.buildHttpResultW(roles, "parent");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 根据角色编码查询(支持多个)
   * @return
   */
  @ApiOperation(value = "根据角色编码查询(支持多个)")
  @GetMapping(value = "/findByCodes")
  public ResponseModel findByCode(@ApiParam(value = "角色编码数组") @RequestParam(name = "roleCodes") String[] roleCodes) {
    try {
      if(roleCodes == null) {
        return this.buildHttpResult();
      }
      String tenantCode = TenantUtils.getTenantCode();
      Set<RoleEntity> roles = this.roleService.findDetailsByCodes(tenantCode, Lists.newArrayList(roleCodes));
      return this.buildHttpResultW(roles);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}
