package com.bizunited.platform.rbac2.server.starter.service.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.platform.rbac2.sdk.service.ButtonVoService;
import com.bizunited.platform.rbac2.sdk.service.RoleVoCacheService;
import com.bizunited.platform.rbac2.sdk.vo.ButtonVo;
import com.bizunited.platform.rbac2.sdk.vo.RoleVo;
import com.bizunited.platform.rbac2.server.starter.entity.ButtonEntity;
import com.bizunited.platform.rbac2.server.starter.entity.CompetenceEntity;
import com.bizunited.platform.rbac2.server.starter.entity.RoleEntity;
import com.bizunited.platform.rbac2.server.starter.repository.ButtonRepository;
import com.google.common.collect.Sets;

/**
 * 为了加快查询速度，所有的按钮信息都存在缓存
 * @author yinwenjie
 */
public class ButtonVoServiceImpl implements ButtonVoService {
  @Autowired
  private RoleVoCacheService roleVoCacheService;
  @Autowired
  private ButtonRepository buttonRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public ButtonVo findById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    ButtonEntity button = this.buttonRepository.findDateilsById(id);
    if(button == null) {
      return null;
    }
    return this.copyButton(button);
  }

  @Override
  public ButtonVo findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    ButtonEntity button = this.buttonRepository.findByCode(code);
    if(button == null) {
      return null;
    }
    return this.copyButton(button);
  }

  @Override
  public Set<ButtonVo> findByAcconut(String account) { 
    if(StringUtils.isBlank(account)) {
      return null;
    }
    String tenantCode = TenantUtils.getTenantCode();
    
    /*
     * 查询过程为：
     * 1、首先查询指定账号已经具备的角色（从缓存中）
     * 2、再根据得到的角色返回按钮信息
     * */
    Set<RoleVo> roles = this.roleVoCacheService.findByTenantCodeAndUserAccount(tenantCode, account);
    if(CollectionUtils.isEmpty(roles)) {
      return null;
    }
    List<String> roleCodes = roles.stream().map(RoleVo::getRoleCode).collect(Collectors.toList());
    Set<ButtonEntity> buttons = this.buttonRepository.findByRoleCodesAndTenantCode(roleCodes, tenantCode);
    if(CollectionUtils.isEmpty(buttons)) {
      return null;
    }
    return this.copyButtons(buttons);
  }
  
  @Override
  public Set<ButtonVo> findByTopCompetenceId(String topCompetenceId) {
    if(StringUtils.isEmpty(topCompetenceId)) {
      return null;
    }
    Set<ButtonEntity> buttons = this.buttonRepository.findByTopCompetenceId(topCompetenceId);
    if(CollectionUtils.isEmpty(buttons)) {
      return null;
    }
    return this.copyButtons(buttons);
  }
  
  @Override
  public Set<ButtonVo> findByRoleCodesAndTopCompetenceId(List<String> roleCodes, String topCompetenceId) {
    if(CollectionUtils.isEmpty(roleCodes)) {
      return null;
    }
    if(StringUtils.isBlank(topCompetenceId)) {
      return null;
    }
    String tenantCode = TenantUtils.getTenantCode();
    Set<ButtonEntity> buttons = this.buttonRepository.findByRoleCodesAndTenantCode(roleCodes, tenantCode);
    if(CollectionUtils.isEmpty(buttons)) {
      return null;
    }
    return this.copyButtons(buttons);
  }
  
  /**
   * 该私有方法用于完成单个按钮信息（和关联信息）的中度拷贝
   * @param button
   * @return
   */
  private ButtonVo copyButton(ButtonEntity button) {
    ButtonVo buttonVo = this.nebulaToolkitService.copyObjectByWhiteList(button, ButtonVo.class, LinkedHashSet.class, ArrayList.class);
    // 关联的角色信息
    Set<RoleEntity> roles = button.getRoles();
    if(!CollectionUtils.isEmpty(roles)) {
      Set<String> roleCodes = roles.stream().map(item -> StringUtils.join(item.getTenantCode() , "|" , item.getRoleCode())).distinct().collect(Collectors.toSet());
      buttonVo.setRoles(roleCodes);
    }
    // 按钮关联的主要功能菜单
    CompetenceEntity topCompetence = button.getTopCompetence();
    if(topCompetence != null) {
      buttonVo.setTopCompetenceCode(topCompetence.getCode());
    }
    // 关联的功能信息
    Set<CompetenceEntity> competences = button.getCompetences();
    if(!CollectionUtils.isEmpty(competences)) {
      Set<String> competenceCodes = competences.stream().map(CompetenceEntity::getCode).distinct().collect(Collectors.toSet());
      buttonVo.setCompetenceCodes(competenceCodes);
    }
    return buttonVo;
  }
  
  /**
   * 该私有方法用于完成多个按钮信息（和关联信息）的中度拷贝
   * @param button
   * @return
   */
  private Set<ButtonVo> copyButtons(Collection<ButtonEntity> buttons) {
    if(CollectionUtils.isEmpty(buttons)) {
      return null;
    }
    
    Set<ButtonVo> buttonVos = Sets.newLinkedHashSet();
    for (ButtonEntity button : buttons) {
      buttonVos.add(this.copyButton(button));
    }
    return buttonVos;
  }
}