package com.bizunited.platform.rbac2.server.starter.service.internal;

import com.bizunited.platform.common.entity.UuidEntity;
import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.platform.rbac2.sdk.constant.Constants;
import com.bizunited.platform.rbac2.sdk.event.CompetenceEventListener;
import com.bizunited.platform.rbac2.sdk.service.CompetenceVoService;
import com.bizunited.platform.rbac2.sdk.vo.CompetenceVo;
import com.bizunited.platform.rbac2.server.starter.dto.CompetenceConditionDto;
import com.bizunited.platform.rbac2.server.starter.entity.ButtonEntity;
import com.bizunited.platform.rbac2.server.starter.entity.CompetenceEntity;
import com.bizunited.platform.rbac2.server.starter.entity.RoleEntity;
import com.bizunited.platform.rbac2.server.starter.repository.ButtonRepository;
import com.bizunited.platform.rbac2.server.starter.repository.CompetenceRepository;
import com.bizunited.platform.rbac2.server.starter.repository.RoleRepository;
import com.bizunited.platform.rbac2.server.starter.service.CompetenceService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * 该服务实现有几个基于数据持久层的非缓存查询方法，直接从数据库查询数据。</br>
 * 特别注意，由于角色缓存中并没有缓存任何和功能相关的信息，所以功能/按钮信息发生变化后，无需清理角色相关的缓存信息
 * @author yinwenjie
 */
@Service
public class CompetenceServiceImpl implements CompetenceService {

  @Autowired
  private CompetenceRepository competenceRepository;
  @Autowired
  private ButtonRepository buttonRepository;
  @Autowired
  private RoleRepository roleRepository;
  @Autowired
  private CompetenceVoService competenceVoService;
  @Autowired(required = false)
  private List<CompetenceEventListener> competenceListeners;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  
  @Transactional
  @Override
  public CompetenceEntity create(CompetenceEntity comp) {
    this.currentCreate(comp);
    String appCode = TenantUtils.getAppCode();
    this.competenceVoService.notifyCacheRefresh(appCode);
    return comp;
  }
  
  /**
   * 真正进行菜单/功能创建动作的操作
   * @param comp
   * @return
   */
  private void currentCreate(CompetenceEntity comp) {
    // 验证
    this.validCompetence(comp);
    if(comp.getViewItem().equals(NormalStatusEnum.ENABLE.getStatus())) {
      String code = comp.getCode();
      Validate.notNull(code, "菜单编码不能为空");
      long count = competenceRepository.countByCode(comp.getCode());
      Validate.isTrue(count == 0, "菜单编码不能重复");
    }
    comp.setId(null);
    // 验证是否已经存在
    String account = this.findOpUser();
    comp.setMethods(comp.getMethods().toUpperCase());
    comp.setTstatus(NormalStatusEnum.ENABLE.getStatus());
    comp.setCreateTime(new Date());
    comp.setModifyTime(new Date());
    comp.setCreateAccount(account);
    comp.setModifyAccount(account);
    //添加默认排序索引
    if(comp.getSortIndex() == null){
      comp.setSortIndex(Constants.DEFAULT_COMPETENCE_SORT_INDEX);
    }
    // 确认新的菜单/功能的父级设定
    CompetenceEntity parent = comp.getParent();
    if(parent != null) {
      String parentId = parent.getId();
      Validate.notBlank(parentId , "创建菜单时，可以指定上级菜单，但指定的上级菜单需要指定编号信息!!");
      CompetenceEntity cParent = competenceRepository.findById(parentId).orElse(null);
      Validate.notNull(cParent, "未找到上级菜单");
      Validate.isTrue(!parentId.equals(comp.getId()), "不能将自身设置为上级菜单");
      comp.setParent(parent);
    } else {
      comp.setParent(null);
    }
    
    competenceRepository.saveAndFlush(comp);
    // 监听创建事件
    if(!CollectionUtils.isEmpty(this.competenceListeners)) {
      CompetenceVo competenceVo = this.nebulaToolkitService.copyObjectByWhiteList(comp, CompetenceVo.class, LinkedHashSet.class, ArrayList.class);
      this.competenceListeners.forEach(item -> item.onCreated(competenceVo));
    }
  }
  
  /**
   * 验证权限对象和字段.
   * @param comp 权限对象
   */
  private void validCompetence(CompetenceEntity comp) {
    // 验证对象是否存在
    Validate.notNull(comp, "功能对象不能为空！");
    // 权限串
    String resource = comp.getResource();
    Validate.notNull(resource, "功能URL串不能为空! 如果没有功能的url，请传递空字符串!");
    // 涉及的方法描述（POST/GET……）
    String method = comp.getMethods();
    Validate.notBlank(method, "方法描述不能为空！");
    // 备注
    String comment = comp.getComment();
    Validate.notBlank(comment, "功能备注不能为空！");
    //功能名
    Validate.notNull(comp.getViewItem(), "功能项是否显示在菜单树中必传");
    if(comp.getParent() != null) {
      Validate.notBlank(comp.getParent().getId(), "设置上级菜单必须传入上级菜单id");
    }
  }
  
  @Transactional
  @Override
  public List<CompetenceEntity> create(List<CompetenceEntity> comps) {
    Validate.isTrue(comps != null && !comps.isEmpty() , "批量添加功能时，必须进行批量传参");
    for (CompetenceEntity competence : comps) {
      this.currentCreate(competence);
    }
    
    String appCode = TenantUtils.getAppCode();
    this.competenceVoService.notifyCacheRefresh(appCode);
    return comps;
  }
  
  @Transactional
  @Override
  public CompetenceEntity update(CompetenceEntity comp) {
    Validate.notNull(comp , "必须传入需要修改的功能信息!!");
    String compId = comp.getId();
    Validate.notBlank(compId , "错误的功能信息编号，请检查!!");
    this.validCompetence(comp);
    Optional<CompetenceEntity> op = this.competenceRepository.findById(compId);
    // 验证对象
    Validate.isTrue(op.isPresent(), "id为%s的功能对象在数据库中不存在！" , compId);
    CompetenceEntity currentComp = op.get();
    //更新验证功能备注是否重复
    if (comp.getViewItem().equals(NormalStatusEnum.ENABLE.getStatus())){
      long count = competenceRepository.countByCommentAndViewItemAndExceptId(currentComp.getId(), comp.getComment(), comp.getViewItem());
      Validate.isTrue(count == 0, "菜单备注名重复，请重新输入菜单名!!");
    }
    String account = this.findOpUser();
    currentComp.setResource(comp.getResource());
    currentComp.setMethods(comp.getMethods());
    currentComp.setComment(comp.getComment());
    currentComp.setDescription(comp.getDescription());
    currentComp.setType(comp.getType());
    currentComp.setModifyAccount(account);
    currentComp.setModifyTime(new Date());
    //排序索引
    if(comp.getSortIndex() == null) {
      currentComp.setSortIndex(Constants.DEFAULT_COMPETENCE_SORT_INDEX);
    } else {
      currentComp.setSortIndex(comp.getSortIndex());
    }
    currentComp.setIcon(comp.getIcon());
    currentComp.setModifyTime(new Date());
    // 验证通过后methods统一大写存储
    currentComp.setMethods(currentComp.getMethods().toUpperCase());
    
    // 确认菜单/功能的父级信息
    CompetenceEntity parent = comp.getParent();
    CompetenceEntity oldParent = currentComp.getParent();
    // 当传入的父级ID不为null，并且旧父级为null或者新父级不等于旧父级时，则更新父级
    if(parent != null && (oldParent == null || !oldParent.getId().equals(parent.getId()))){
      CompetenceEntity entity = competenceRepository.findById(parent.getId()).orElse(null);
      Validate.notNull(entity, "未找到上级菜单");
      Validate.isTrue(!parent.getId().equals(currentComp.getId()), "不能将自身设置为上级菜单");
      Set<String> compStack = new HashSet<>();
      compStack.add(currentComp.getId());
      this.validCompetenceCircular(entity, compStack);
      currentComp.setParent(entity);
    } else if(parent == null) {
      currentComp.setParent(null);
    }
    this.competenceRepository.saveAndFlush(currentComp);
    String appCode = TenantUtils.getAppCode();
    this.competenceVoService.notifyCacheRefresh(appCode);
    return comp;
  }
  
  /**
   * 判断是否形成循环依赖
   * @param competence
   * @param compStack
   */
  private void validCompetenceCircular(CompetenceEntity competence, Set<String> compStack) {
    Validate.notNull(competence, "必须传入菜单信息");
    CompetenceEntity parent = competence.getParent();
    if(parent == null) {
      return;
    }
    Validate.isTrue(!compStack.contains(parent.getId()), "形成循环依赖，更新失败，请检查！");
    compStack.add(parent.getId());
    this.validCompetenceCircular(parent, compStack);
  }
  
  @Transactional
  @Override
  public void updateStatus(String id, Boolean flag) {
    if (StringUtils.isEmpty(id) || flag == null) {
      throw new IllegalArgumentException("updateStatus操作时参数错误！");
    }
    this.recursiveUpdateStatus(id, flag);
    String appCode = TenantUtils.getAppCode();
    this.competenceVoService.notifyCacheRefresh(appCode);
  }
  
  /**
   * 递归更新子级功能信息的状态
   * @param parentCompetenceId
   * @param flag
   */
  private void recursiveUpdateStatus(String parentCompetenceId, Boolean flag) {
    Optional<CompetenceEntity> op = this.competenceRepository.findById(parentCompetenceId);
    Validate.isTrue(op.isPresent(), "指定的功能信息不存在！");
    CompetenceEntity competenceEntity = op.get();
    if (Boolean.TRUE.equals(flag)) {
      competenceEntity.setTstatus(1);
    } else {
      competenceEntity.setTstatus(0);
    }
    competenceEntity.setModifyTime(new Date());
    competenceRepository.save(competenceEntity);
    // 子级功能信息
    List<CompetenceEntity> childList = this.competenceRepository.findByParentId(parentCompetenceId);
    for(int index = 0 ; childList != null && index < childList.size() ; index++) {
      this.recursiveUpdateStatus(childList.get(index).getId(), flag);
    }
  }
  
  /**
   * 该方法帮助寻找正确的操作者
   * @return
   */
  private String findOpUser() {
    // 可能的操作者信息(如果没有就写入admin)
    Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
    String account = "admin";
    if(authentication != null ) {
      account = authentication.getName();
    }
    return account;
  }
  
  @Transactional
  @Override
  public void bindRole(String roleId, String[] competenceIds) {
    Validate.notBlank(roleId , "进行功能和角色绑定时，必须指定当前角色技术编号!!");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent() , "没有发现指定的角色信息");
    Validate.isTrue(competenceIds != null && competenceIds.length > 0 , "进行功能和角色绑定时，必须至少指定一个功能编号!!");
    for (String competenceId : competenceIds) {
      Optional<CompetenceEntity> opCompetence = this.competenceRepository.findById(competenceId);
      Validate.isTrue(opCompetence.isPresent() , "没有发现指定的功能信息[%s]，请检查!!" , competenceId);
      
      // 该判断逻辑保证不会进行重复绑定
      long count = this.competenceRepository.countByRoleIdAndCompetenceId(roleId, competenceId);
      if(count == 0) {
        this.competenceRepository.bindRole(roleId, competenceId);
      }
    }
    
    // 进行通知
    String appCode = TenantUtils.getAppCode();
    this.competenceVoService.notifyCacheRefresh(appCode);
  }
  
  @Override
  @Transactional
  public void reBindRole(String roleId, String[] competenceIds) {
    // 首先删除绑定，然后再重新进行绑定
    Validate.notBlank(roleId , "进行功能和角色绑定时，必须指定当前角色技术编号!!");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent() , "没有发现指定的角色信息");
    //解绑角色与菜单之间的关联;
    List<CompetenceEntity> competences = this.competenceRepository.findByRoleId(roleId);
    if(!CollectionUtils.isEmpty(competences)){
      String[] exsitCompetenceIds = competences.stream().map(UuidEntity::getId).toArray(String[]::new);
      for (String competenceId : exsitCompetenceIds) {
        Optional<CompetenceEntity> optional = this.competenceRepository.findById(competenceId);
        Validate.isTrue(optional.isPresent() , "没有发现指定的功能信息[%s]，请检查!!" , competenceId);
        this.competenceRepository.unbindRole(roleId, competenceId);
      }
    }
    // 然后重新进行绑定
    if(competenceIds == null || competenceIds.length == 0) {
      return;
    }
    for (String competenceId : competenceIds) {
      Optional<CompetenceEntity> opCompetence = this.competenceRepository.findById(competenceId);
      Validate.isTrue(opCompetence.isPresent() , "没有发现指定的功能信息[%s]，请检查!!" , competenceId);
      // 该判断逻辑保证不会进行重复绑定
      long count = this.competenceRepository.countByRoleIdAndCompetenceId(roleId, competenceId);
      if(count == 0) {
        this.competenceRepository.bindRole(roleId, competenceId);
      }
    }
    
    // 进行通知
    String appCode = TenantUtils.getAppCode();
    this.competenceVoService.notifyCacheRefresh(appCode);
  }

  @Transactional
  @Override
  public void unbindRole(String roleId, String[] competenceIds) {
    Validate.notBlank(roleId , "进行功能和角色解绑时，必须指定当前角色编号!!");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent() , "没有发现指定的角色信息");
    Validate.isTrue(competenceIds != null && competenceIds.length > 0 , "进行功能和角色解绑时，必须至少指定一个功能编号!!");

    for (String competenceId : competenceIds) {
      Optional<CompetenceEntity> optional = this.competenceRepository.findById(competenceId);
      Validate.isTrue(optional.isPresent() , "没有发现指定的功能信息[%s]，请检查!!" , competenceId);
      this.competenceRepository.unbindRole(roleId, competenceId);
    }
    
    // 进行通知
    String appCode = TenantUtils.getAppCode();
    this.competenceVoService.notifyCacheRefresh(appCode);
  }
  
  @Transactional
  @Override
  public void unbindAllRoles(String roleId) {
    Validate.notBlank(roleId,"角色主键不能为空");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent() , "没有发现指定的角色信息");
    //解绑角色与菜单之间的关联;
    List<CompetenceEntity> competences = this.competenceRepository.findByRoleId(roleId);
    if(CollectionUtils.isEmpty(competences)){
      return;
    }
    String[] competenceIds = competences.stream().map(UuidEntity::getId).toArray(String[]::new);
    for (String competenceId : competenceIds) {
      Optional<CompetenceEntity> optional = this.competenceRepository.findById(competenceId);
      Validate.isTrue(optional.isPresent() , "没有发现指定的功能信息[%s]，请检查!!" , competenceId);
      this.competenceRepository.unbindRole(roleId, competenceId);
    }
    
    // 进行通知
    String appCode = TenantUtils.getAppCode();
    this.competenceVoService.notifyCacheRefresh(appCode);
  }
  
  @Override
  @Transactional
  public void bindButtons(String buttonId, String[] competenceIds) {
    Validate.notBlank(buttonId, "进行按钮和接口绑定时，必须指定当前按钮编号!!");
    Optional<ButtonEntity> op = this.buttonRepository.findById(buttonId);
    Validate.isTrue(op.isPresent(), "没有发现指定的按钮信息");
    Validate.notEmpty(competenceIds, "进行按钮和接口绑定时，必须至少指定一个按钮编号!!");
    for (String competenceId : competenceIds) {
      CompetenceEntity opCompetence = this.competenceRepository.findById(competenceId).orElse(null);
      Validate.notNull(opCompetence, "没有发现指定的接口信息[%s]，请检查!!", competenceId);
      long count = this.buttonRepository.countByCompetenceIdAndButtonId(buttonId, competenceId);
      if(count == 0L) {
        this.buttonRepository.bindCompetence(buttonId, competenceId);
      }
    }

    // 进行通知
    String appCode = TenantUtils.getAppCode();
    this.competenceVoService.notifyCacheRefresh(appCode);
  }

  @Override
  @Transactional
  public void reBindButtons(String buttonId, String[] competenceIds) {
    Validate.notBlank(buttonId, "按钮ID不能为空");
    ButtonEntity button = buttonRepository.findById(buttonId).orElse(null);
    Validate.notNull(button, "按钮不存在，请检查！！");
    // 首先删除之前的绑定，然后再重新进行绑定
    this.buttonRepository.unbindAllByButtonId(buttonId);
    // 然后重新绑定
    if(competenceIds != null && competenceIds.length > 0) {
      for (String competenceId : competenceIds) {
        CompetenceEntity opCompetence = this.competenceRepository.findById(competenceId).orElse(null);
        Validate.notNull(opCompetence, "没有发现指定的功能信息[%s]，请检查!!", competenceId);
        this.buttonRepository.bindCompetence(buttonId, competenceId);
      }
    }
    
    // 进行通知
    String appCode = TenantUtils.getAppCode();
    this.competenceVoService.notifyCacheRefresh(appCode);
  }

  @Override
  @Transactional
  public void unbindButtons(String buttonId, String[] competenceIds) {
    Validate.notBlank(buttonId, "进行按钮和接口解绑时，必须指定当前按钮编号!!");
    Optional<ButtonEntity> op = this.buttonRepository.findById(buttonId);
    Validate.isTrue(op.isPresent(), "没有发现指定的按钮信息");
    Validate.notEmpty(competenceIds, "进行按钮和接口解绑时，必须至少指定一个按钮编号!!");
    for (String competenceId : competenceIds) {
      CompetenceEntity opCompetence = this.competenceRepository.findById(competenceId).orElse(null);
      Validate.notNull(opCompetence, "没有发现指定的接口信息[%s]，请检查!!", competenceId);
      this.buttonRepository.unbindCompetence(buttonId, competenceId);
    }
    // 进行通知
    String appCode = TenantUtils.getAppCode();
    this.competenceVoService.notifyCacheRefresh(appCode);
  }
  
  @Transactional
  @Override
  public void deleteById(String competenceId) {
    Validate.notBlank(competenceId , "删除功能时，指定的功能编号必须传递!!");
    Optional<CompetenceEntity> op = this.competenceRepository.findById(competenceId);
    Validate.isTrue(op.isPresent() , "未发现指定功能编号的功能，请检查参数!!");
    CompetenceEntity currentCompetence = op.get();
    Set<RoleEntity> roles = currentCompetence.getRoles();
    Validate.isTrue(CollectionUtils.isEmpty(roles) , "指定的功能已经绑定角色，不能进行删除!!");
    List<CompetenceEntity> childs = this.competenceRepository.findByParentId(competenceId);
    Validate.isTrue(CollectionUtils.isEmpty(childs) , "指定的功能存在子级功能，不能进行删除!!");
    // 有按钮绑定，也不能进行删除
    Set<ButtonEntity> buttons = currentCompetence.getButtons();
    Validate.isTrue(!CollectionUtils.isEmpty(buttons) , "指定的功能已经绑定按钮，不能进行删除!!");
    // 正式进行删除
    this.competenceRepository.delete(currentCompetence);
    // 监听删除事件
    if(!CollectionUtils.isEmpty(this.competenceListeners)) {
      CompetenceVo competenceVo = this.nebulaToolkitService.copyObjectByWhiteList(currentCompetence, CompetenceVo.class, LinkedHashSet.class, ArrayList.class);
      this.competenceListeners.forEach(item -> item.onCreated(competenceVo));
    }
    String appCode = TenantUtils.getAppCode();
    this.competenceVoService.notifyCacheRefresh(appCode);
  }
  
  // ========= 以下直接从数据库进行查询操作

  @Override
  public Page<CompetenceEntity> findByConditions(CompetenceConditionDto competence, Pageable pageable) {
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    if(pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
    // Page中的内容必须自己转换
    return this.competenceRepository.findByConditions(pageable, competence);
  }
  
  @Override
  public CompetenceEntity findByCommentAndViewItemAndParent(String comment, int viewItem, String parentId) {
    if(StringUtils.isAnyBlank(comment, parentId)) {
      return null;
    }
    CompetenceEntity competence = competenceRepository.findByCommentAndViewItemAndParent(comment, viewItem, parentId);
    return competence == null ? null : nebulaToolkitService.copyObjectByWhiteList(competence, CompetenceEntity.class, HashSet.class, ArrayList.class);
  }
  
  @Override
  public CompetenceEntity findByCommentAndViewItemAndParentNull(String comment, int viewItem) {
    if(StringUtils.isBlank(comment)) {
      return null;
    }
    CompetenceEntity competence = competenceRepository.findByCommentAndViewItemAndNullParent(comment, viewItem);
    return competence == null ? null : nebulaToolkitService.copyObjectByWhiteList(competence, CompetenceEntity.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<CompetenceEntity> findAll() {
    return competenceRepository.findAll();
  }


  @Override
  public long countByViewItem(int viewItem) {
    return competenceRepository.countByViewItem(viewItem);
  }
  
  @Override
  public long count() {
    return competenceRepository.count();
  }
}
