package com.bizunited.platform.rbac2.server.starter.service.internal;

import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.platform.rbac2.sdk.config.RbacCustomProperties;
import com.bizunited.platform.rbac2.sdk.service.ButtonVoService;
import com.bizunited.platform.rbac2.sdk.service.CompetenceVoService;
import com.bizunited.platform.rbac2.sdk.vo.ButtonVo;
import com.bizunited.platform.rbac2.sdk.vo.CompetenceVo;
import com.bizunited.platform.rbac2.server.starter.entity.CompetenceEntity;
import com.bizunited.platform.rbac2.server.starter.entity.RoleEntity;
import com.bizunited.platform.rbac2.server.starter.repository.CompetenceRepository;
import com.bizunited.platform.rbac2.server.starter.repository.RoleRepository;
import com.bizunited.platform.rbac2.server.starter.service.strategy.CompetenceQueryStrategy;
import com.bizunited.platform.rbac2.server.starter.service.strategy.QueryByButtonCodeAndRoleCodes;
import com.bizunited.platform.rbac2.server.starter.service.strategy.QueryByParentNull;
import com.bizunited.platform.rbac2.server.starter.service.strategy.QueryByResourcesAndMethodAndRoleCodes;
import com.bizunited.platform.rbac2.server.starter.service.strategy.QueryByViewItemAndRoleCodes;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.redisson.Redisson;
import org.redisson.api.RTopic;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock.ReadLock;
import java.util.concurrent.locks.ReentrantReadWriteLock.WriteLock;
import java.util.stream.Collectors;

/**
 * 整个全局菜单的管理遵守以下原则：</p>
 * 1、系统将所有菜单、功能、按钮构建成完整的树结构（无论这些信息本身是否可用，是否存在角色绑定），并缓存在内存中</p>
 * 2、一旦在诸如菜单变化、角色绑定、按钮关联等事件发生时，树结构发生了变化，缓存信息都将会被清空，并重新加载（TODO 此步骤以后可以换成局部操作）</p>
 * 3、所有租户基于这样的完整功能树，在查询需要的情况下，自行遍历自己需要的结构，并形成一个子树副本进行返回</p>
 * @author yinwenjie
 */
public class CompetenceVoServiceImpl implements CompetenceVoService {
  @Autowired
  private CompetenceRepository competenceRepository;
  @Autowired
  private RoleRepository roleRepository;
  @Autowired
  private ButtonVoService buttonVoService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private RbacCustomProperties rbacCustomProperties;
  @Autowired
  private Redisson redisson;
  /**
   * 易变的属性标记当前全菜单树缓存是否正在进行刷新，以及是那个线程在进行刷新
   */
  private volatile Thread flashingThread = null;
  private AtomicReference<Thread> atomicReference = new AtomicReference<Thread>(null);
  /**
   * 功能信息的缓存按照功能唯一业务编号进行存储（注意，都是从顶层菜单开始，且没有进行最终排序）
   * 第一级Key：appCode，因为一个应用系统可能为多个顶级租户服务
   * 第二级Key：顶级菜单的的code信息
   * V：菜单信息以及其下所有关联信息
   */
  private static Map<String , Map<String , CompetenceVo>> competencesCacheMapping = Maps.newConcurrentMap();
  /**
   * 保证在缓存刷新过程中，需要读取缓存的线程进行阻塞(一个锁)</br>
   * TODO 后续再优化成每一个appCode一个锁
   * @see #RebatePolicyVoServiceImpl
   */
  private static ReentrantReadWriteLock competenceloadLock = new ReentrantReadWriteLock();

  @Override
  public void clearCache(String appCode) {
    Validate.notBlank(appCode , "要求清理功能树缓存时，必须传入appCode");
    WriteLock writeLock = competenceloadLock.writeLock();
    try {
      writeLock.lock();
      Map<String, CompetenceVo> competencesCache = competencesCacheMapping.get(appCode);
      if(competencesCache != null) {
        competencesCache.clear();
      }
    } finally {
      writeLock.unlock();
    }
  }
  
  @Override
  public void notifyCacheRefresh(String appCode) {
    Validate.notBlank(appCode , "要求清理功能树缓存时，必须传入appCode");
    RTopic topic = redisson.getTopic(CompetenceVoService.ALL_COMPETENCE_NOTIFY);
    topic.publish(appCode);
  }
  
  /**
   * 该方法从数据层，从最顶层功能菜单开始，提取当前租户（厂商）下有效的所有菜单、子级功能、按钮信息。并放入缓存
   * @return
   */
  private void findDetailsFromRepository() {
    /*
     * 以下代码段落用于保证全量功能数据加载成树结构的过程中，工作安全性，具体过程如下：
     * 1、首先释放当前线程占用的读锁，修改，读锁一旦释放，后续同时具有读锁的线程也会进入该方法（这是因为读锁是AQS共享模式）
     * 2、判定当前试图进行全量数据刷新的所有线程，是否是flashingThread线程，如果不是，则自旋
     * 直到flashingThread为null后退出该方法(原子操作)
     * 3、只有flashingThread线程，可以获得写锁，并进行后续的是指更新动作
     * 至于加写锁的原因，是让后续未获得读锁的线程，进入阻塞状态
     * */
    String appCode = TenantUtils.getAppCode();
    
    // 1、======
    ReadLock readLock = competenceloadLock.readLock();
    readLock.unlock();
    
    // 2、=======
    Thread currentThread = Thread.currentThread();
    boolean isFlashingThread = false;
    if(isFlashingThread = this.atomicReference.compareAndSet(null, currentThread) ) {
      this.flashingThread = currentThread;
    } else {
      Thread.yield();
    }
    // 如果没有抢占到加载任务，则一直自旋，直到刷新过程结束
    if(!isFlashingThread) {
      while(flashingThread != null) {
        Thread.yield();
      }
      readLock.lock();
      return;
    }
    
    // 3、======
    WriteLock writeLock = competenceloadLock.writeLock();
    try {
      writeLock.lock();
      /*
       * 请注意，从以下过程开始，该私有方法并不负责对可重入锁的管理，而是专注于处理从数据库查询数据。 
       * 该方法将采用深度遍历的方式，对整个功能和菜单树进行构建，处理过程如下：
       * 一、首先对顶级菜单进行查询，并以每一个顶级菜单A为开始，进行结构遍历
       * ========= 以下都是递归处理recursiveLoadCompetence
       * 1、求得A对应的角色信息（当然是缓存全部的角色）
       * 2、求得A对应的按钮信息（当然是缓存全部的按钮），包括按钮对应的角色信息，
       * 3、递归求得A的子级功能信息（当然是可用的菜单/功能）:
       *   包括每一子级节点对应的按钮、下一子级和角色信息
       * ========= 以上都是递归处理
       * 
       * 二、然后对顶层功能进行查询，并以每一个顶级功能开始，进行结构遍历
       * ========= 接着，同样调用recursiveLoadCompetence方法
       * 
       * 最后，将完成遍历的所有菜单和功能，放入competenceCacheMapping
       * */
      
      // 一、=======
      Map<String , CompetenceVo> competenceCacheMapping = Maps.newConcurrentMap();
      List<CompetenceEntity> allCompetences = this.competenceRepository.findAll();
      Validate.isTrue(!CollectionUtils.isEmpty(allCompetences) , "菜单信息为空，请联系管理员!!");
      //ViewItem = 1 标识需要显示在菜单树上的信息
      List<CompetenceEntity> currentTopCompetences = allCompetences.stream().filter(e -> e.getViewItem() == 1).collect(Collectors.toList());
      Validate.isTrue(!CollectionUtils.isEmpty(currentTopCompetences) , "发现错误的顶层菜单信息，请联系管理员!!");
      List<CompetenceVo> topCompetences = Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(currentTopCompetences, CompetenceEntity.class, CompetenceVo.class, LinkedHashSet.class, ArrayList.class,"parent"));
      Validate.isTrue(topCompetences.stream().noneMatch(item -> StringUtils.isBlank(item.getCode())), "发现错误的顶层菜单信息(没有唯一业务编号)，请联系管理员!!");
      // TODO 这里的性能还要想办法优化（但实际上还好，理论上只会有一次全量树的扫描和加载）
      // 出于性能的考虑，在遍历过程中，已经完成查询的按钮信息放置在这个MAP中 
      for (CompetenceVo topCompetence : topCompetences) {
        this.recursiveLoadCompetence(topCompetence);
        competenceCacheMapping.put(topCompetence.getCode(), topCompetence);
      }
      
      // 二、======
      //ViewItem = 0 标识不需要显示在菜单树上的信息
      List<CompetenceEntity> currentCompetenceEntitys = allCompetences.stream().filter(e -> e.getViewItem() == 0).collect(Collectors.toList());
      if(!CollectionUtils.isEmpty(currentCompetenceEntitys)) {
        topCompetences = Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(currentCompetenceEntitys, CompetenceEntity.class, CompetenceVo.class, LinkedHashSet.class, ArrayList.class));
        for (CompetenceVo currentCompetence : topCompetences) {
          this.recursiveLoadCompetence(currentCompetence);
          competenceCacheMapping.put(currentCompetence.getCode(), currentCompetence);
        }
      }
      competencesCacheMapping.put(appCode, competenceCacheMapping);
    } finally {
      this.atomicReference.compareAndSet(currentThread , null);
      this.flashingThread = null;
      writeLock.unlock();
      readLock.lock();
    }
  }
  
  /**
   * 工作过程请参见findDetailsFromRepository 
   * @param competence
   * @param tenantCode 
   * @see #findDetailsFromRepository(ReentrantReadWriteLock)
   */
  private void recursiveLoadCompetence(CompetenceVo competence) {
    // 1、======
    String competenceId = competence.getId();
    Set<ButtonVo> buttonVos = this.buttonVoService.findByTopCompetenceId(competenceId);
    if(!CollectionUtils.isEmpty(buttonVos)) {
      competence.setButtons(buttonVos);
    }
    
    // 2、======
    List<RoleEntity> roles = this.roleRepository.findByCompetenceId(competenceId);
    if(!CollectionUtils.isEmpty(roles)) {
      List<String> roleCodes = roles.stream().map(item -> StringUtils.join(item.getTenantCode() , "|" , item.getRoleCode())).distinct().collect(Collectors.toList());
      competence.setRoles(StringUtils.join(roleCodes , ","));
    }
    
    // 3、======
    List<CompetenceEntity> childrens = this.competenceRepository.findByParentId(competenceId);
    if(CollectionUtils.isEmpty(childrens)) {
      return;
    }
    // 对每一个子级菜单/功能进行构造
    Set<CompetenceVo> childrenVos = Sets.newLinkedHashSet(this.nebulaToolkitService.copyCollectionByWhiteList(childrens, CompetenceEntity.class, CompetenceVo.class, LinkedHashSet.class, ArrayList.class,"buttons"));
    for (CompetenceVo childCompetence : childrenVos) {
      this.recursiveLoadCompetence(childCompetence);
      childCompetence.setParent(competence);
    }
    competence.setChildren(childrenVos);
  }
  
  @Override
  public Set<CompetenceVo> findByViewItemAndRoleCodesAndStatus(Boolean viewItem, String tenantCode, String[] roleCodes, Integer status) {
    String appCode = TenantUtils.getAppCode();
    if(roleCodes == null || roleCodes.length == 0 || StringUtils.isAnyBlank(tenantCode , appCode)) {
      return null;
    }
    ReadLock readLock = competenceloadLock.readLock();
    /*
     * 注意，返回的一定是副本信息，特别注意父级Competence对象的构造，处理过程如下
     * 1、判定当前传入的角色是否存在管理员角色，并根据结果确认是否需要进行角色code的查询限制
     * ======== 以下处理过程只能是在获取了读锁的情况下，才能进行操作
     * 2、如果发现competencesCacheMapping没有缓存，则调用缓存加载方法
     * 3、从顶层菜单开始遍历，并按照条件要求进行过滤
     * */
    
    // 1、======
    // 对当前用户的角色名称与yml配置的管理员角色信息取交集，判定是否具有管理员角色
    boolean isAdmin = this.comfirmAdmin(roleCodes);
    CompetenceQueryStrategy competenceQueryStrategy = new QueryByViewItemAndRoleCodes(viewItem, isAdmin, tenantCode, roleCodes, status, this.rbacCustomProperties.getIgnoreMethodCheckRoles());
    Set<CompetenceVo> results = Sets.newLinkedHashSet();
    try {
      readLock.lock();
      // 2、======
      if(CollectionUtils.isEmpty(competencesCacheMapping.get(appCode))) {
        this.findDetailsFromRepository();
      }
      // 3、=======
      Collection<CompetenceVo> topCompetences = competencesCacheMapping.get(appCode).values();
      // 注意：value没有排序，这里要进行排序
      List<CompetenceVo> sortedTopCompetences = topCompetences.stream().filter(e -> e.getParent() == null || StringUtils.isBlank(e.getParent().getId())).sorted(Comparator.comparingInt(CompetenceVo::getSortIndex)).collect(Collectors.toList());
      for (CompetenceVo topCompetence : sortedTopCompetences) {
        CompetenceVo copyTopCompetence = this.recursiveQueryCompetenceForTree(competenceQueryStrategy, topCompetence);
        if(copyTopCompetence != null) {
          results.add(copyTopCompetence);
        }
      }
    } finally {
      readLock.unlock();
    }
    return results;
  }


  @Override
  public Set<CompetenceVo> findByViewItemAndCurrentAccount(Boolean viewItem) {
    String tenantCode = TenantUtils.getTenantCode();
    // 获取当前登录用户的信息（主要是角色code信息）
    SecurityContext securityContext = SecurityContextHolder.getContext();
    if(securityContext == null) {
      return null;
    }
    Authentication authentication = securityContext.getAuthentication();
    if(authentication == null) {
      return null;
    }
    Collection<? extends GrantedAuthority> authorities = authentication.getAuthorities();
    if(CollectionUtils.isEmpty(authorities)) {
      return null;
    }
    Set<String> roleCodes = authorities.stream().map(GrantedAuthority::getAuthority).collect(Collectors.toSet());
    if(CollectionUtils.isEmpty(roleCodes)) {
      return null;
    }
    
    return this.findByViewItemAndRoleCodesAndStatus(viewItem, tenantCode, roleCodes.toArray(new String[] {}), null);
  }
  
  @Override
  public Set<CompetenceVo> findByViewItemAndStatusAndParentNull(Boolean viewItem, String tenantCode, Integer status) {
    // 这里注意，由于是构造的树形结构，所以一定策略是：只要执行过滤的节点存在上级节点，就都进行返回
    String appCode = TenantUtils.getAppCode();
    if(viewItem == null || StringUtils.isAnyBlank(tenantCode , appCode)) {
      return null;
    }
    ReadLock readLock = competenceloadLock.readLock();
    Set<CompetenceVo> results = Sets.newLinkedHashSet();
    try {
      readLock.lock();
      if(CollectionUtils.isEmpty(competencesCacheMapping.get(appCode))) {
        this.findDetailsFromRepository();
      }
      
      Collection<CompetenceVo> topCompetences = competencesCacheMapping.get(appCode).values();
      QueryByParentNull queryByParentNull = new QueryByParentNull(viewItem, tenantCode, status);
      // 注意：value没有排序，这里要进行排序
      List<CompetenceVo> sortedTopCompetences = topCompetences.stream().filter(e -> e.getParent() == null || StringUtils.isBlank(e.getParent().getId())).sorted(Comparator.comparingInt(CompetenceVo::getSortIndex)).collect(Collectors.toList());
      for (CompetenceVo topCompetence : sortedTopCompetences) {
        CompetenceVo copyTopCompetence = this.recursiveQueryCompetenceForTree(queryByParentNull, topCompetence);
        if(copyTopCompetence != null) {
          results.add(copyTopCompetence);
        }
      }
    } finally {
      readLock.unlock();
    }
    return results;
  }
  
  /**
   * 这是递归查询，和递归构建recursiveLoadCompetence方法存在差异——以树结构的方式
   * @param competence
   * @param viewItem 菜单/功能部分的条件过滤
   * @param isAdmin 当前角色是否拥有管理员角色的过滤
   * @param tenantCode 当前租户的过滤
   * @param roleCodes 当前角色业务编号的过滤（注意大小写）
   * @param status 节点状态的过滤
   */
  private CompetenceVo recursiveQueryCompetenceForTree(CompetenceQueryStrategy competenceQueryStrategy , CompetenceVo competence) {
    /*
     * 1、首先进行节点自身可用性、性质的过滤，如果不满足，则不用再向后处理
     * 2、再次进行节点下按钮信息的过滤，包括按钮状态、按钮角色
     * 3、最后，递归进行下级菜单/功能的过滤
     * */
    // 1、=======
    CompetenceVo copyCompetence = competenceQueryStrategy.filterCompetence(competence);
    // 如果条件成立，说明该菜单/功能节点，经过指定查询策略器过滤后，不具备返回要求
    if(copyCompetence == null) {
      return null;
    }
    Validate.isTrue(copyCompetence != competence , "进行功能树查询和构建时，必须使用中拷副本");
    
    // 2、=======
    // 开始进行按钮的筛选查询
    Set<ButtonVo> buttons = competence.getButtons();
    Set<ButtonVo> copyButtons = Sets.newLinkedHashSet();
    if(!CollectionUtils.isEmpty(buttons)) {
      for (ButtonVo buttonVo : buttons) {
        ButtonVo copyButton = competenceQueryStrategy.filterButton(buttonVo);
        if(copyButton != null) {
          copyButtons.add(copyButton);
        }
      }
    }
    copyCompetence.setButtons(copyButtons);
    
    // 3、======
    Set<CompetenceVo> children = competence.getChildren();
    Set<CompetenceVo> copyChildren = Sets.newLinkedHashSet();
    if(!CollectionUtils.isEmpty(children)) {
      for (CompetenceVo child : children) {
        CompetenceVo copyChild = this.recursiveQueryCompetenceForTree(competenceQueryStrategy , child);
        if(copyChild != null) {
          copyChildren.add(copyChild);
        }
      }
      copyCompetence.setChildren(copyChildren);
    }
    return copyCompetence;
  }

  /**
   * 确认角色查询条件是否包括管理员角色
   * @param roleCodes
   * @return
   */
  private boolean comfirmAdmin(String[] roleCodes) {
    boolean isAdmin = false;
    if(roleCodes != null && roleCodes.length > 0) {
      Sets.SetView<String> intersections = Sets.intersection(Sets.newHashSet(this.rbacCustomProperties.getIgnoreMethodCheckRoles()), Sets.newHashSet(roleCodes));
      isAdmin = !CollectionUtils.isEmpty(intersections);
    }
    
    return isAdmin;
  }

  @Override
  public Set<CompetenceVo> findByResource(String resource, Integer tstatus) {
    if(StringUtils.isBlank(resource)) {
      return null;
    }
    
    Set<CompetenceVo> results = Sets.newLinkedHashSet();
    this.findByResources(new String[] {resource} , null ,tstatus, null, null, results);
    return results;
  }

  @Override
  public Set<CompetenceVo> findByResourceLike(String resource, Integer status) {
    if(StringUtils.isBlank(resource)) {
      return null;
    }
    
    Set<CompetenceVo> results = Sets.newLinkedHashSet();
    this.findByResources(new String[] {resource} , null ,status, null, null, results);
    return results;
  }

  @Override
  public Set<CompetenceVo> findByResources(String[] resources , String tenantCode, String[] roleCodes) {
    if(resources == null || resources.length == 0) {
      return null;
    }
    
    Set<CompetenceVo> results = Sets.newLinkedHashSet();
    this.findByResources(resources , null ,null, tenantCode, roleCodes, results);
    return results;
  }
  
  @Override
  public CompetenceVo findByResourceAndMethods(String resource, String methods, Integer status) {
    if(StringUtils.isBlank(resource)) {
      return null;
    }
    
    Set<CompetenceVo> results = Sets.newLinkedHashSet();
    this.findByResources(new String[] {resource} , methods ,status, null, null, results);
    if(!CollectionUtils.isEmpty(results)) {
      return null;
    }
    return results.iterator().next();
  }

  @Override
  public Set<CompetenceVo> findByResourcesAndCurrentAccount(String[] resources) {
    if(resources == null || resources.length == 0) {
      return null;
    }
    String tenantCode = TenantUtils.getTenantCode();
    Set<CompetenceVo> results = Sets.newLinkedHashSet();
    this.findByResources(resources, null, null, tenantCode, null, results);
    if(!CollectionUtils.isEmpty(results)) {
      return null;
    }
    return results;
  }

  /**
   * 这是findByResources真实的工作逻辑 
   * @param resources 可能的一个或者多个菜单/功能路径信息 
   * @param viewItem 可能参考过滤的功能节点状态 
   * @param status 可能参考过滤的状态信息 
   * @param tenantCode 可能参考过滤的租户（经销商）编号 
   * @param roleCodes 可能参考过滤的角色编号 
   */
  private void findByResources(String[] resources , String method , Integer status, String tenantCode , String[] roleCodes , Set<CompetenceVo> results) {
    String appCode = TenantUtils.getAppCode();
    if(resources == null || resources.length == 0) {
      return;
    }
    // 确认是否有管理员角色
    boolean isAdmin = this.comfirmAdmin(roleCodes);
    // 由功能根节点开始进行遍历
    CompetenceQueryStrategy competenceQueryStrategy = new QueryByResourcesAndMethodAndRoleCodes(resources, false, method, status, tenantCode, roleCodes, isAdmin , this.rbacCustomProperties.getIgnoreMethodCheckRoles());
    ReadLock readLock = competenceloadLock.readLock();
    try {
      readLock.lock();
      if(CollectionUtils.isEmpty(competencesCacheMapping.get(appCode))) {
        this.findDetailsFromRepository();
      }
      
      Collection<CompetenceVo> topCompetences = competencesCacheMapping.get(appCode).values();
      // 注意：value没有排序，这里要进行排序
      List<CompetenceVo> sortedTopCompetences = topCompetences.stream().sorted((source , target) -> source.getSortIndex() - target.getSortIndex()).collect(Collectors.toList());
      for (CompetenceVo topCompetence : sortedTopCompetences) {
        this.recursiveQueryCompetenceForList(competenceQueryStrategy, topCompetence , results);
      }
    } finally {
      readLock.unlock();
    }
  }

  @Override
  public Set<CompetenceVo> findByButtonCode(String buttonCode) {
    if(StringUtils.isBlank(buttonCode)) {
      return null;
    }
    Set<CompetenceVo> results = Sets.newLinkedHashSet();
    this.findByButtonCodeAndRoleCodes(buttonCode, null, null, null, results);
    return results;
  }

  @Override
  public Set<CompetenceVo> findByRoleCodes(String tenantCode , String[] roleCodes) {
    if(StringUtils.isBlank(tenantCode) || roleCodes == null || roleCodes.length == 0) {
      return null;
    }
    
    Set<CompetenceVo> results = Sets.newLinkedHashSet();
    this.findByButtonCodeAndRoleCodes(null, tenantCode, roleCodes, null, results);
    return results;
  }

  @Override
  public Set<CompetenceVo> findByRoleCodesAndParentNull(String tenantCode , String[] roleCodes) {
    if(StringUtils.isBlank(tenantCode) || roleCodes == null || roleCodes.length == 0) {
      return null;
    }
    
    Set<CompetenceVo> results = Sets.newLinkedHashSet();
    this.findByButtonCodeAndRoleCodes(null, tenantCode, roleCodes, true, results);
    if(!CollectionUtils.isEmpty(results)) {
      return Sets.newLinkedHashSet(results.stream().filter(item -> item.getParent() == null).collect(Collectors.toList()));
    } 
    return results;
  }
  
  /**
   * 真正按照按钮编号、角色信息进行查询的方法，基于recursiveQueryCompetenceForList
   * @param buttonCode 
   * @param tenantCode
   * @param roleCodes
   * @param viewItem
   * @return
   */
  private void findByButtonCodeAndRoleCodes(String buttonCode , String tenantCode , String[] roleCodes , Boolean viewItem , Set<CompetenceVo> results) {
    String appCode = TenantUtils.getAppCode();
    // 确认是否有管理员角色
    boolean isAdmin = this.comfirmAdmin(roleCodes);
    CompetenceQueryStrategy competenceQueryStrategy = new QueryByButtonCodeAndRoleCodes(tenantCode, roleCodes, viewItem, buttonCode, isAdmin, roleCodes);
    ReadLock readLock = competenceloadLock.readLock();
    try {
      readLock.lock();
      if(CollectionUtils.isEmpty(competencesCacheMapping.get(appCode))) {
        this.findDetailsFromRepository();
      }
      
      Collection<CompetenceVo> topCompetences = competencesCacheMapping.get(appCode).values();
      // 注意：value没有排序，这里要进行排序
      List<CompetenceVo> sortedTopCompetences = topCompetences.stream().sorted((source , target) -> source.getSortIndex() - target.getSortIndex()).collect(Collectors.toList());
      for (CompetenceVo topCompetence : sortedTopCompetences) {
        this.recursiveQueryCompetenceForList(competenceQueryStrategy, topCompetence , results);
      }
    } finally {
      readLock.unlock();
    }
  }

  /**
   * 这是递归查询，和递归构建recursiveLoadCompetenceForTree方法存在的差异是——该方法以线性表的方式进行遍历
   * @param competence
   * @param viewItem 菜单/功能部分的条件过滤
   * @param isAdmin 当前角色是否拥有管理员角色的过滤
   * @param tenantCode 当前租户的过滤
   * @param roleCodes 当前角色业务编号的过滤（注意大小写）
   * @param status 节点状态的过滤
   */
  private void recursiveQueryCompetenceForList(CompetenceQueryStrategy competenceQueryStrategy , CompetenceVo competence , Set<CompetenceVo> queryResult) {
    /*
     * 由于进行的是所有符合查询要求的菜单/功能集合，所以这就代表了即使当前节点不满足要求，但只要还有子级节点就要向下遍历
     * 从外部传入的记录查询结果的queryResult，建议使用LinkedHashSet
     * 
     * 1、首先进行节点自身可用性、性质的过滤，如果不满足，则继续向后遍历
     * 后续如果返回的集合不为null，则要增加到当前的queryResult集合中（按顺序增加）
     * 2、再次进行节点下按钮信息的过滤，包括按钮状态、按钮角色
     * 注意，这里只增加满足查询要求的按钮
     * 3、最后，递归进行下级菜单/功能的过滤 , 另外，由于是线性排列，所以得到的CompetenceVo不需要记录下级功能的关联
     * */
    
    // 1、=======
    CompetenceVo copyCompetence = competenceQueryStrategy.filterCompetence(competence);
    // 如果条件成立，说明该菜单/功能节点，经过指定查询策略器过滤后，不具备返回要求
    if(copyCompetence != null) {
      Validate.isTrue(copyCompetence != competence , "进行功能树查询和构建时，必须使用中拷副本");
      queryResult.add(copyCompetence);
    }
    
    // 2、=======
    // 如果当前菜单/功能节点满足要求，才开始进行按钮的筛选查询
    if(copyCompetence != null) {
      Set<ButtonVo> buttons = competence.getButtons();
      Set<ButtonVo> copyButtons = Sets.newLinkedHashSet();
      if(!CollectionUtils.isEmpty(buttons)) {
        for (ButtonVo buttonVo : buttons) {
          ButtonVo copyButton = competenceQueryStrategy.filterButton(buttonVo);
          if(copyButton != null) {
            copyButtons.add(copyButton);
          }
        }
      }
      copyCompetence.setButtons(copyButtons);
    }
    
    // 3、======
    Set<CompetenceVo> children = competence.getChildren();
    if(!CollectionUtils.isEmpty(children)) {
      for (CompetenceVo child : children) {
        this.recursiveQueryCompetenceForList(competenceQueryStrategy , child , queryResult);
      }
    }
  }
}
